/*
 * Copyright (C) 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.view.translation;

import android.annotation.NonNull;
import android.os.Parcelable;

import com.android.internal.util.DataClass;

import java.util.concurrent.Executor;
import java.util.function.Consumer;

/**
 * Info class holding information for {@link Translator}s and used by
 * {@link TranslationManager#createOnDeviceTranslator(TranslationContext, Executor, Consumer)}.
 */
@DataClass(genHiddenConstDefs = true, genToString = true, genBuilder = true)
public final class TranslationContext implements Parcelable {

    /**
     * This context will perform translations in low latency mode.
     */
    public static final @TranslationFlag int FLAG_LOW_LATENCY = 0x1;
    /**
     * This context will enable the {@link Translator} to return transliteration results.
     */
    public static final @TranslationFlag int FLAG_TRANSLITERATION = 0x2;
    /**
     * This context will enable the {@link Translator} to return dictionary definitions.
     */
    public static final @TranslationFlag int FLAG_DEFINITIONS = 0x4;

    /**
     * {@link TranslationSpec} describing the source data to be translated.
     */
    @NonNull
    private final TranslationSpec mSourceSpec;

    /**
     * {@link TranslationSpec} describing the target translated data.
     */
    @NonNull
    private final TranslationSpec mTargetSpec;

    /**
     * Translation flags to be used by the {@link Translator}
     */
    private final @TranslationFlag int mTranslationFlags;

    private static int defaultTranslationFlags() {
        return 0;
    }

    @DataClass.Suppress({"setSourceSpec", "setTargetSpec"})
    abstract static class BaseBuilder {

    }



    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/android/view/translation/TranslationContext.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    /** @hide */
    @android.annotation.IntDef(flag = true, prefix = "FLAG_", value = {
        FLAG_LOW_LATENCY,
        FLAG_TRANSLITERATION,
        FLAG_DEFINITIONS
    })
    @java.lang.annotation.Retention(java.lang.annotation.RetentionPolicy.SOURCE)
    @DataClass.Generated.Member
    public @interface TranslationFlag {}

    /** @hide */
    @DataClass.Generated.Member
    public static String translationFlagToString(@TranslationFlag int value) {
        return com.android.internal.util.BitUtils.flagsToString(
                value, TranslationContext::singleTranslationFlagToString);
    }

    @DataClass.Generated.Member
    static String singleTranslationFlagToString(@TranslationFlag int value) {
        switch (value) {
            case FLAG_LOW_LATENCY:
                    return "FLAG_LOW_LATENCY";
            case FLAG_TRANSLITERATION:
                    return "FLAG_TRANSLITERATION";
            case FLAG_DEFINITIONS:
                    return "FLAG_DEFINITIONS";
            default: return Integer.toHexString(value);
        }
    }

    @DataClass.Generated.Member
    /* package-private */ TranslationContext(
            @NonNull TranslationSpec sourceSpec,
            @NonNull TranslationSpec targetSpec,
            @TranslationFlag int translationFlags) {
        this.mSourceSpec = sourceSpec;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mSourceSpec);
        this.mTargetSpec = targetSpec;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mTargetSpec);
        this.mTranslationFlags = translationFlags;

        com.android.internal.util.Preconditions.checkFlagsArgument(
                mTranslationFlags,
                FLAG_LOW_LATENCY
                        | FLAG_TRANSLITERATION
                        | FLAG_DEFINITIONS);

        // onConstructed(); // You can define this method to get a callback
    }

    /**
     * {@link TranslationSpec} describing the source data to be translated.
     */
    @DataClass.Generated.Member
    public @NonNull TranslationSpec getSourceSpec() {
        return mSourceSpec;
    }

    /**
     * {@link TranslationSpec} describing the target translated data.
     */
    @DataClass.Generated.Member
    public @NonNull TranslationSpec getTargetSpec() {
        return mTargetSpec;
    }

    /**
     * Translation flags to be used by the {@link Translator}
     */
    @DataClass.Generated.Member
    public @TranslationFlag int getTranslationFlags() {
        return mTranslationFlags;
    }

    @Override
    @DataClass.Generated.Member
    public String toString() {
        // You can override field toString logic by defining methods like:
        // String fieldNameToString() { ... }

        return "TranslationContext { " +
                "sourceSpec = " + mSourceSpec + ", " +
                "targetSpec = " + mTargetSpec + ", " +
                "translationFlags = " + translationFlagToString(mTranslationFlags) +
        " }";
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull android.os.Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        dest.writeTypedObject(mSourceSpec, flags);
        dest.writeTypedObject(mTargetSpec, flags);
        dest.writeInt(mTranslationFlags);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    /* package-private */ TranslationContext(@NonNull android.os.Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        TranslationSpec sourceSpec = (TranslationSpec) in.readTypedObject(TranslationSpec.CREATOR);
        TranslationSpec targetSpec = (TranslationSpec) in.readTypedObject(TranslationSpec.CREATOR);
        int translationFlags = in.readInt();

        this.mSourceSpec = sourceSpec;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mSourceSpec);
        this.mTargetSpec = targetSpec;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mTargetSpec);
        this.mTranslationFlags = translationFlags;

        com.android.internal.util.Preconditions.checkFlagsArgument(
                mTranslationFlags,
                FLAG_LOW_LATENCY
                        | FLAG_TRANSLITERATION
                        | FLAG_DEFINITIONS);

        // onConstructed(); // You can define this method to get a callback
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<TranslationContext> CREATOR
            = new Parcelable.Creator<TranslationContext>() {
        @Override
        public TranslationContext[] newArray(int size) {
            return new TranslationContext[size];
        }

        @Override
        public TranslationContext createFromParcel(@NonNull android.os.Parcel in) {
            return new TranslationContext(in);
        }
    };

    /**
     * A builder for {@link TranslationContext}
     */
    @SuppressWarnings("WeakerAccess")
    @DataClass.Generated.Member
    public static final class Builder extends BaseBuilder {

        private @NonNull TranslationSpec mSourceSpec;
        private @NonNull TranslationSpec mTargetSpec;
        private @TranslationFlag int mTranslationFlags;

        private long mBuilderFieldsSet = 0L;

        /**
         * Creates a new Builder.
         *
         * @param sourceSpec
         *   {@link TranslationSpec} describing the source data to be translated.
         * @param targetSpec
         *   {@link TranslationSpec} describing the target translated data.
         */
        public Builder(
                @NonNull TranslationSpec sourceSpec,
                @NonNull TranslationSpec targetSpec) {
            mSourceSpec = sourceSpec;
            com.android.internal.util.AnnotationValidations.validate(
                    NonNull.class, null, mSourceSpec);
            mTargetSpec = targetSpec;
            com.android.internal.util.AnnotationValidations.validate(
                    NonNull.class, null, mTargetSpec);
        }

        /**
         * Translation flags to be used by the {@link Translator}
         */
        @DataClass.Generated.Member
        public @NonNull Builder setTranslationFlags(@TranslationFlag int value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x4;
            mTranslationFlags = value;
            return this;
        }

        /** Builds the instance. This builder should not be touched after calling this! */
        public @NonNull TranslationContext build() {
            checkNotUsed();
            mBuilderFieldsSet |= 0x8; // Mark builder used

            if ((mBuilderFieldsSet & 0x4) == 0) {
                mTranslationFlags = defaultTranslationFlags();
            }
            TranslationContext o = new TranslationContext(
                    mSourceSpec,
                    mTargetSpec,
                    mTranslationFlags);
            return o;
        }

        private void checkNotUsed() {
            if ((mBuilderFieldsSet & 0x8) != 0) {
                throw new IllegalStateException(
                        "This Builder should not be reused. Use a new Builder instance instead");
            }
        }
    }

    @DataClass.Generated(
            time = 1621545292157L,
            codegenVersion = "1.0.23",
            sourceFile = "frameworks/base/core/java/android/view/translation/TranslationContext.java",
            inputSignatures = "public static final @android.view.translation.TranslationContext.TranslationFlag int FLAG_LOW_LATENCY\npublic static final @android.view.translation.TranslationContext.TranslationFlag int FLAG_TRANSLITERATION\npublic static final @android.view.translation.TranslationContext.TranslationFlag int FLAG_DEFINITIONS\nprivate final @android.annotation.NonNull android.view.translation.TranslationSpec mSourceSpec\nprivate final @android.annotation.NonNull android.view.translation.TranslationSpec mTargetSpec\nprivate final @android.view.translation.TranslationContext.TranslationFlag int mTranslationFlags\nprivate static  int defaultTranslationFlags()\nclass TranslationContext extends java.lang.Object implements [android.os.Parcelable]\nclass BaseBuilder extends java.lang.Object implements []\n@com.android.internal.util.DataClass(genHiddenConstDefs=true, genToString=true, genBuilder=true)\nclass BaseBuilder extends java.lang.Object implements []")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
