/*
 * Decompiled with CFR 0.152.
 */
package com.android.server.net.watchlist;

import android.content.ContentResolver;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.pm.UserInfo;
import android.os.Bundle;
import android.os.DropBoxManager;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.os.UserHandle;
import android.os.UserManager;
import android.provider.Settings;
import android.text.TextUtils;
import android.util.Slog;
import com.android.internal.annotations.VisibleForTesting;
import com.android.internal.util.ArrayUtils;
import com.android.internal.util.HexDump;
import com.android.server.net.watchlist.DigestUtils;
import com.android.server.net.watchlist.ReportEncoder;
import com.android.server.net.watchlist.WatchlistConfig;
import com.android.server.net.watchlist.WatchlistReportDbHelper;
import com.android.server.net.watchlist.WatchlistSettings;
import java.io.File;
import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

class WatchlistLoggingHandler
extends Handler {
    private static final String TAG = WatchlistLoggingHandler.class.getSimpleName();
    private static final boolean DEBUG = false;
    @VisibleForTesting
    static final int LOG_WATCHLIST_EVENT_MSG = 1;
    @VisibleForTesting
    static final int REPORT_RECORDS_IF_NECESSARY_MSG = 2;
    @VisibleForTesting
    static final int FORCE_REPORT_RECORDS_NOW_FOR_TEST_MSG = 3;
    private static final long ONE_DAY_MS = TimeUnit.DAYS.toMillis(1L);
    private static final String DROPBOX_TAG = "network_watchlist_report";
    private final Context mContext;
    private final DropBoxManager mDropBoxManager;
    private final ContentResolver mResolver;
    private final PackageManager mPm;
    private final WatchlistReportDbHelper mDbHelper;
    private final WatchlistConfig mConfig;
    private final WatchlistSettings mSettings;
    private int mPrimaryUserId = -1;
    private final ConcurrentHashMap<Integer, byte[]> mCachedUidDigestMap = new ConcurrentHashMap();

    WatchlistLoggingHandler(Context context, Looper looper) {
        super(looper);
        this.mContext = context;
        this.mPm = this.mContext.getPackageManager();
        this.mResolver = this.mContext.getContentResolver();
        this.mDbHelper = WatchlistReportDbHelper.getInstance(context);
        this.mConfig = WatchlistConfig.getInstance();
        this.mSettings = WatchlistSettings.getInstance();
        this.mDropBoxManager = this.mContext.getSystemService(DropBoxManager.class);
        this.mPrimaryUserId = this.getPrimaryUserId();
    }

    @Override
    public void handleMessage(Message msg) {
        switch (msg.what) {
            case 1: {
                Bundle data = msg.getData();
                this.handleNetworkEvent(data.getString("host"), data.getStringArray("ipAddresses"), data.getInt("uid"), data.getLong("timestamp"));
                break;
            }
            case 2: {
                this.tryAggregateRecords(WatchlistLoggingHandler.getLastMidnightTime());
                break;
            }
            case 3: {
                if (msg.obj instanceof Long) {
                    long lastRecordTime = (Long)msg.obj;
                    this.tryAggregateRecords(lastRecordTime);
                    break;
                }
                Slog.e(TAG, "Msg.obj needs to be a Long object.");
                break;
            }
            default: {
                Slog.d(TAG, "WatchlistLoggingHandler received an unknown of message.");
            }
        }
    }

    private int getPrimaryUserId() {
        UserInfo primaryUserInfo = ((UserManager)this.mContext.getSystemService("user")).getPrimaryUser();
        if (primaryUserInfo != null) {
            return primaryUserInfo.id;
        }
        return -1;
    }

    private boolean isPackageTestOnly(int uid) {
        ApplicationInfo ai;
        try {
            String[] packageNames = this.mPm.getPackagesForUid(uid);
            if (packageNames == null || packageNames.length == 0) {
                Slog.e(TAG, "Couldn't find package: " + packageNames);
                return false;
            }
            ai = this.mPm.getApplicationInfo(packageNames[0], 0);
        }
        catch (PackageManager.NameNotFoundException e) {
            return false;
        }
        return (ai.flags & 0x100) != 0;
    }

    public void reportWatchlistIfNecessary() {
        Message msg = this.obtainMessage(2);
        this.sendMessage(msg);
    }

    public void forceReportWatchlistForTest(long lastReportTime) {
        Message msg = this.obtainMessage(3);
        msg.obj = lastReportTime;
        this.sendMessage(msg);
    }

    public void asyncNetworkEvent(String host, String[] ipAddresses, int uid) {
        Message msg = this.obtainMessage(1);
        Bundle bundle = new Bundle();
        bundle.putString("host", host);
        bundle.putStringArray("ipAddresses", ipAddresses);
        bundle.putInt("uid", uid);
        bundle.putLong("timestamp", System.currentTimeMillis());
        msg.setData(bundle);
        this.sendMessage(msg);
    }

    private void handleNetworkEvent(String hostname, String[] ipAddresses, int uid, long timestamp) {
        if (this.mPrimaryUserId == -1) {
            this.mPrimaryUserId = this.getPrimaryUserId();
        }
        if (UserHandle.getUserId(uid) != this.mPrimaryUserId) {
            return;
        }
        String cncDomain = this.searchAllSubDomainsInWatchlist(hostname);
        if (cncDomain != null) {
            this.insertRecord(uid, cncDomain, timestamp);
        } else {
            String cncIp = this.searchIpInWatchlist(ipAddresses);
            if (cncIp != null) {
                this.insertRecord(uid, cncIp, timestamp);
            }
        }
    }

    private boolean insertRecord(int uid, String cncHost, long timestamp) {
        if (!this.mConfig.isConfigSecure() && !this.isPackageTestOnly(uid)) {
            return true;
        }
        byte[] digest = this.getDigestFromUid(uid);
        if (digest == null) {
            Slog.e(TAG, "Cannot get digest from uid: " + uid);
            return false;
        }
        boolean result = this.mDbHelper.insertNewRecord(digest, cncHost, timestamp);
        return result;
    }

    private boolean shouldReportNetworkWatchlist(long lastRecordTime) {
        long lastReportTime = Settings.Global.getLong(this.mResolver, "network_watchlist_last_report_time", 0L);
        if (lastRecordTime < lastReportTime) {
            Slog.i(TAG, "Last report time is larger than current time, reset report");
            this.mDbHelper.cleanup(lastReportTime);
            return false;
        }
        return lastRecordTime >= lastReportTime + ONE_DAY_MS;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void tryAggregateRecords(long lastRecordTime) {
        long startTime = System.currentTimeMillis();
        try {
            if (!this.shouldReportNetworkWatchlist(lastRecordTime)) {
                Slog.i(TAG, "No need to aggregate record yet.");
                return;
            }
            Slog.i(TAG, "Start aggregating watchlist records.");
            if (this.mDropBoxManager != null && this.mDropBoxManager.isTagEnabled(DROPBOX_TAG)) {
                Settings.Global.putLong(this.mResolver, "network_watchlist_last_report_time", lastRecordTime);
                WatchlistReportDbHelper.AggregatedResult aggregatedResult = this.mDbHelper.getAggregatedRecords(lastRecordTime);
                if (aggregatedResult == null) {
                    Slog.i(TAG, "Cannot get result from database");
                    return;
                }
                List<String> digestsForReport = this.getAllDigestsForReport(aggregatedResult);
                byte[] secretKey = this.mSettings.getPrivacySecretKey();
                byte[] encodedResult = ReportEncoder.encodeWatchlistReport(this.mConfig, secretKey, digestsForReport, aggregatedResult);
                if (encodedResult != null) {
                    this.addEncodedReportToDropBox(encodedResult);
                }
            } else {
                Slog.w(TAG, "Network Watchlist dropbox tag is not enabled");
            }
            this.mDbHelper.cleanup(lastRecordTime);
        }
        finally {
            long endTime = System.currentTimeMillis();
            Slog.i(TAG, "Milliseconds spent on tryAggregateRecords(): " + (endTime - startTime));
        }
    }

    @VisibleForTesting
    List<String> getAllDigestsForReport(WatchlistReportDbHelper.AggregatedResult record) {
        List<ApplicationInfo> apps = this.mContext.getPackageManager().getInstalledApplications(131072);
        HashSet<String> result = new HashSet<String>(apps.size() + record.appDigestCNCList.size());
        int size = apps.size();
        for (int i = 0; i < size; ++i) {
            byte[] digest = this.getDigestFromUid(apps.get((int)i).uid);
            if (digest != null) {
                result.add(HexDump.toHexString(digest));
                continue;
            }
            Slog.e(TAG, "Cannot get digest from uid: " + apps.get((int)i).uid + ",pkg: " + apps.get((int)i).packageName);
        }
        result.addAll(record.appDigestCNCList.keySet());
        return new ArrayList<String>(result);
    }

    private void addEncodedReportToDropBox(byte[] encodedReport) {
        this.mDropBoxManager.addData(DROPBOX_TAG, encodedReport, 0);
    }

    private byte[] getDigestFromUid(int uid) {
        return this.mCachedUidDigestMap.computeIfAbsent(uid, key -> {
            String[] packageNames = this.mPm.getPackagesForUid((int)key);
            int userId = UserHandle.getUserId(uid);
            if (!ArrayUtils.isEmpty(packageNames)) {
                for (String packageName : packageNames) {
                    try {
                        String apkPath = this.mPm.getPackageInfoAsUser((String)packageName, (int)786432, (int)userId).applicationInfo.publicSourceDir;
                        if (!TextUtils.isEmpty(apkPath)) {
                            return DigestUtils.getSha256Hash(new File(apkPath));
                        }
                        Slog.w(TAG, "Cannot find apkPath for " + packageName);
                    }
                    catch (PackageManager.NameNotFoundException | IOException | NoSuchAlgorithmException e) {
                        Slog.e(TAG, "Should not happen", e);
                        return null;
                    }
                }
            }
            return null;
        });
    }

    private String searchIpInWatchlist(String[] ipAddresses) {
        for (String ipAddress : ipAddresses) {
            if (!this.isIpInWatchlist(ipAddress)) continue;
            return ipAddress;
        }
        return null;
    }

    private boolean isIpInWatchlist(String ipAddr) {
        if (ipAddr == null) {
            return false;
        }
        return this.mConfig.containsIp(ipAddr);
    }

    private boolean isHostInWatchlist(String host) {
        if (host == null) {
            return false;
        }
        return this.mConfig.containsDomain(host);
    }

    private String searchAllSubDomainsInWatchlist(String host) {
        String[] subDomains;
        if (host == null) {
            return null;
        }
        for (String subDomain : subDomains = WatchlistLoggingHandler.getAllSubDomains(host)) {
            if (!this.isHostInWatchlist(subDomain)) continue;
            return subDomain;
        }
        return null;
    }

    @VisibleForTesting
    static String[] getAllSubDomains(String host) {
        if (host == null) {
            return null;
        }
        ArrayList<String> subDomainList = new ArrayList<String>();
        subDomainList.add(host);
        int index = host.indexOf(".");
        while (index != -1) {
            if (!TextUtils.isEmpty(host = host.substring(index + 1))) {
                subDomainList.add(host);
            }
            index = host.indexOf(".");
        }
        return subDomainList.toArray(new String[0]);
    }

    static long getLastMidnightTime() {
        return WatchlistLoggingHandler.getMidnightTimestamp(0);
    }

    static long getMidnightTimestamp(int daysBefore) {
        GregorianCalendar date = new GregorianCalendar();
        date.set(11, 0);
        date.set(12, 0);
        date.set(13, 0);
        date.set(14, 0);
        ((Calendar)date).add(5, -daysBefore);
        return date.getTimeInMillis();
    }

    private static interface WatchlistEventKeys {
        public static final String HOST = "host";
        public static final String IP_ADDRESSES = "ipAddresses";
        public static final String UID = "uid";
        public static final String TIMESTAMP = "timestamp";
    }
}

