/*
 * Decompiled with CFR 0.152.
 */
package com.android.server.display;

import android.content.res.Resources;
import android.content.res.TypedArray;
import android.hardware.display.BrightnessConfiguration;
import android.util.MathUtils;
import android.util.Pair;
import android.util.Slog;
import android.util.Spline;
import com.android.internal.annotations.VisibleForTesting;
import com.android.internal.util.Preconditions;
import com.android.server.display.utils.Plog;
import java.io.PrintWriter;
import java.util.Arrays;

public abstract class BrightnessMappingStrategy {
    private static final String TAG = "BrightnessMappingStrategy";
    private static final boolean DEBUG = false;
    private static final float LUX_GRAD_SMOOTHING = 0.25f;
    private static final float MAX_GRAD = 1.0f;
    private static final Plog PLOG = Plog.createSystemPlog("BrightnessMappingStrategy");

    public static BrightnessMappingStrategy create(Resources resources) {
        int[] backlightRange;
        float[] luxLevels = BrightnessMappingStrategy.getLuxLevels(resources.getIntArray(0x1070011));
        int[] brightnessLevelsBacklight = resources.getIntArray(0x1070010);
        float[] brightnessLevelsNits = BrightnessMappingStrategy.getFloatArray(resources.obtainTypedArray(17235982));
        float autoBrightnessAdjustmentMaxGamma = resources.getFraction(0x1130000, 1, 1);
        float[] nitsRange = BrightnessMappingStrategy.getFloatArray(resources.obtainTypedArray(17236034));
        if (BrightnessMappingStrategy.isValidMapping(nitsRange, backlightRange = resources.getIntArray(17236033)) && BrightnessMappingStrategy.isValidMapping(luxLevels, brightnessLevelsNits)) {
            int minimumBacklight = resources.getInteger(17694862);
            int maximumBacklight = resources.getInteger(17694861);
            if (backlightRange[0] > minimumBacklight || backlightRange[backlightRange.length - 1] < maximumBacklight) {
                Slog.w(TAG, "Screen brightness mapping does not cover whole range of available backlight values, autobrightness functionality may be impaired.");
            }
            BrightnessConfiguration.Builder builder = new BrightnessConfiguration.Builder();
            builder.setCurve(luxLevels, brightnessLevelsNits);
            return new PhysicalMappingStrategy(builder.build(), nitsRange, backlightRange, autoBrightnessAdjustmentMaxGamma);
        }
        if (BrightnessMappingStrategy.isValidMapping(luxLevels, brightnessLevelsBacklight)) {
            return new SimpleMappingStrategy(luxLevels, brightnessLevelsBacklight, autoBrightnessAdjustmentMaxGamma);
        }
        return null;
    }

    private static float[] getLuxLevels(int[] lux) {
        float[] levels = new float[lux.length + 1];
        for (int i = 0; i < lux.length; ++i) {
            levels[i + 1] = lux[i];
        }
        return levels;
    }

    private static float[] getFloatArray(TypedArray array2) {
        int N = array2.length();
        float[] vals = new float[N];
        for (int i = 0; i < N; ++i) {
            vals[i] = array2.getFloat(i, -1.0f);
        }
        array2.recycle();
        return vals;
    }

    private static boolean isValidMapping(float[] x, float[] y) {
        if (x == null || y == null || x.length == 0 || y.length == 0) {
            return false;
        }
        if (x.length != y.length) {
            return false;
        }
        int N = x.length;
        float prevX = x[0];
        float prevY = y[0];
        if (prevX < 0.0f || prevY < 0.0f || Float.isNaN(prevX) || Float.isNaN(prevY)) {
            return false;
        }
        for (int i = 1; i < N; ++i) {
            if (prevX >= x[i] || prevY > y[i]) {
                return false;
            }
            if (Float.isNaN(x[i]) || Float.isNaN(y[i])) {
                return false;
            }
            prevX = x[i];
            prevY = y[i];
        }
        return true;
    }

    private static boolean isValidMapping(float[] x, int[] y) {
        if (x == null || y == null || x.length == 0 || y.length == 0) {
            return false;
        }
        if (x.length != y.length) {
            return false;
        }
        int N = x.length;
        float prevX = x[0];
        int prevY = y[0];
        if (prevX < 0.0f || prevY < 0 || Float.isNaN(prevX)) {
            return false;
        }
        for (int i = 1; i < N; ++i) {
            if (prevX >= x[i] || prevY > y[i]) {
                return false;
            }
            if (Float.isNaN(x[i])) {
                return false;
            }
            prevX = x[i];
            prevY = y[i];
        }
        return true;
    }

    public abstract boolean setBrightnessConfiguration(BrightnessConfiguration var1);

    public abstract float getBrightness(float var1);

    public abstract float getAutoBrightnessAdjustment();

    public abstract boolean setAutoBrightnessAdjustment(float var1);

    public abstract float convertToNits(int var1);

    public abstract void addUserDataPoint(float var1, float var2);

    public abstract void clearUserDataPoints();

    public abstract boolean hasUserDataPoints();

    public abstract boolean isDefaultConfig();

    public abstract BrightnessConfiguration getDefaultConfig();

    public abstract void dump(PrintWriter var1);

    private static float normalizeAbsoluteBrightness(int brightness) {
        brightness = MathUtils.constrain(brightness, 0, 255);
        return (float)brightness / 255.0f;
    }

    private static Pair<float[], float[]> insertControlPoint(float[] luxLevels, float[] brightnessLevels, float lux, float brightness) {
        float[] newBrightnessLevels;
        float[] newLuxLevels;
        int idx = BrightnessMappingStrategy.findInsertionPoint(luxLevels, lux);
        if (idx == luxLevels.length) {
            newLuxLevels = Arrays.copyOf(luxLevels, luxLevels.length + 1);
            newBrightnessLevels = Arrays.copyOf(brightnessLevels, brightnessLevels.length + 1);
            newLuxLevels[idx] = lux;
            newBrightnessLevels[idx] = brightness;
        } else if (luxLevels[idx] == lux) {
            newLuxLevels = Arrays.copyOf(luxLevels, luxLevels.length);
            newBrightnessLevels = Arrays.copyOf(brightnessLevels, brightnessLevels.length);
            newBrightnessLevels[idx] = brightness;
        } else {
            newLuxLevels = Arrays.copyOf(luxLevels, luxLevels.length + 1);
            System.arraycopy(newLuxLevels, idx, newLuxLevels, idx + 1, luxLevels.length - idx);
            newLuxLevels[idx] = lux;
            newBrightnessLevels = Arrays.copyOf(brightnessLevels, brightnessLevels.length + 1);
            System.arraycopy(newBrightnessLevels, idx, newBrightnessLevels, idx + 1, brightnessLevels.length - idx);
            newBrightnessLevels[idx] = brightness;
        }
        BrightnessMappingStrategy.smoothCurve(newLuxLevels, newBrightnessLevels, idx);
        return Pair.create(newLuxLevels, newBrightnessLevels);
    }

    private static int findInsertionPoint(float[] arr, float val) {
        for (int i = 0; i < arr.length; ++i) {
            if (!(val <= arr[i])) continue;
            return i;
        }
        return arr.length;
    }

    private static void smoothCurve(float[] lux, float[] brightness, int idx) {
        float minBrightness;
        float currLux;
        float maxBrightness;
        float currBrightness;
        float newBrightness;
        int i;
        float prevLux = lux[idx];
        float prevBrightness = brightness[idx];
        for (i = idx + 1; i < lux.length && (newBrightness = MathUtils.constrain(currBrightness = brightness[i], prevBrightness, maxBrightness = prevBrightness * BrightnessMappingStrategy.permissibleRatio(currLux = lux[i], prevLux))) != currBrightness; ++i) {
            prevLux = currLux;
            prevBrightness = newBrightness;
            brightness[i] = newBrightness;
        }
        prevLux = lux[idx];
        prevBrightness = brightness[idx];
        for (i = idx - 1; i >= 0 && (newBrightness = MathUtils.constrain(currBrightness = brightness[i], minBrightness = prevBrightness * BrightnessMappingStrategy.permissibleRatio(currLux = lux[i], prevLux), prevBrightness)) != currBrightness; --i) {
            prevLux = currLux;
            prevBrightness = newBrightness;
            brightness[i] = newBrightness;
        }
    }

    private static float permissibleRatio(float currLux, float prevLux) {
        return MathUtils.exp(1.0f * (MathUtils.log(currLux + 0.25f) - MathUtils.log(prevLux + 0.25f)));
    }

    private static float inferAutoBrightnessAdjustment(float maxGamma, float desiredBrightness, float currentBrightness) {
        float adjustment = 0.0f;
        float gamma = Float.NaN;
        if (currentBrightness <= 0.1f || currentBrightness >= 0.9f) {
            adjustment = desiredBrightness - currentBrightness;
        } else if (desiredBrightness == 0.0f) {
            adjustment = -1.0f;
        } else if (desiredBrightness == 1.0f) {
            adjustment = 1.0f;
        } else {
            gamma = MathUtils.log(desiredBrightness) / MathUtils.log(currentBrightness);
            adjustment = -MathUtils.log(gamma) / MathUtils.log(maxGamma);
        }
        adjustment = MathUtils.constrain(adjustment, -1.0f, 1.0f);
        return adjustment;
    }

    private static Pair<float[], float[]> getAdjustedCurve(float[] lux, float[] brightness, float userLux, float userBrightness, float adjustment, float maxGamma) {
        float[] newLux = lux;
        float[] newBrightness = Arrays.copyOf(brightness, brightness.length);
        float gamma = MathUtils.pow(maxGamma, -(adjustment = MathUtils.constrain(adjustment, -1.0f, 1.0f)));
        if (gamma != 1.0f) {
            for (int i = 0; i < newBrightness.length; ++i) {
                newBrightness[i] = MathUtils.pow(newBrightness[i], gamma);
            }
        }
        if (userLux != -1.0f) {
            Pair<float[], float[]> curve = BrightnessMappingStrategy.insertControlPoint(newLux, newBrightness, userLux, userBrightness);
            newLux = (float[])curve.first;
            newBrightness = (float[])curve.second;
        }
        return Pair.create(newLux, newBrightness);
    }

    @VisibleForTesting
    static class PhysicalMappingStrategy
    extends BrightnessMappingStrategy {
        private BrightnessConfiguration mConfig;
        private Spline mBrightnessSpline;
        private final Spline mNitsToBacklightSpline;
        private final BrightnessConfiguration mDefaultConfig;
        private Spline mBacklightToNitsSpline;
        private float mMaxGamma;
        private float mAutoBrightnessAdjustment;
        private float mUserLux;
        private float mUserBrightness;

        public PhysicalMappingStrategy(BrightnessConfiguration config, float[] nits, int[] backlight, float maxGamma) {
            Preconditions.checkArgument(nits.length != 0 && backlight.length != 0, "Nits and backlight arrays must not be empty!");
            Preconditions.checkArgument(nits.length == backlight.length, "Nits and backlight arrays must be the same length!");
            Preconditions.checkNotNull(config);
            Preconditions.checkArrayElementsInRange(nits, 0.0f, Float.MAX_VALUE, "nits");
            Preconditions.checkArrayElementsInRange(backlight, 0, 255, "backlight");
            this.mMaxGamma = maxGamma;
            this.mAutoBrightnessAdjustment = 0.0f;
            this.mUserLux = -1.0f;
            this.mUserBrightness = -1.0f;
            int N = nits.length;
            float[] normalizedBacklight = new float[N];
            for (int i = 0; i < N; ++i) {
                normalizedBacklight[i] = BrightnessMappingStrategy.normalizeAbsoluteBrightness(backlight[i]);
            }
            this.mNitsToBacklightSpline = Spline.createSpline(nits, normalizedBacklight);
            this.mBacklightToNitsSpline = Spline.createSpline(normalizedBacklight, nits);
            this.mDefaultConfig = config;
            this.mConfig = config;
            this.computeSpline();
        }

        @Override
        public boolean setBrightnessConfiguration(BrightnessConfiguration config) {
            if (config == null) {
                config = this.mDefaultConfig;
            }
            if (config.equals(this.mConfig)) {
                return false;
            }
            this.mConfig = config;
            this.computeSpline();
            return true;
        }

        @Override
        public float getBrightness(float lux) {
            float nits = this.mBrightnessSpline.interpolate(lux);
            float backlight = this.mNitsToBacklightSpline.interpolate(nits);
            return backlight;
        }

        @Override
        public float getAutoBrightnessAdjustment() {
            return this.mAutoBrightnessAdjustment;
        }

        @Override
        public boolean setAutoBrightnessAdjustment(float adjustment) {
            if ((adjustment = MathUtils.constrain(adjustment, -1.0f, 1.0f)) == this.mAutoBrightnessAdjustment) {
                return false;
            }
            this.mAutoBrightnessAdjustment = adjustment;
            this.computeSpline();
            return true;
        }

        @Override
        public float convertToNits(int backlight) {
            return this.mBacklightToNitsSpline.interpolate(BrightnessMappingStrategy.normalizeAbsoluteBrightness(backlight));
        }

        @Override
        public void addUserDataPoint(float lux, float brightness) {
            float adjustment;
            float unadjustedBrightness = this.getUnadjustedBrightness(lux);
            this.mAutoBrightnessAdjustment = adjustment = BrightnessMappingStrategy.inferAutoBrightnessAdjustment(this.mMaxGamma, brightness, unadjustedBrightness);
            this.mUserLux = lux;
            this.mUserBrightness = brightness;
            this.computeSpline();
        }

        @Override
        public void clearUserDataPoints() {
            if (this.mUserLux != -1.0f) {
                this.mAutoBrightnessAdjustment = 0.0f;
                this.mUserLux = -1.0f;
                this.mUserBrightness = -1.0f;
                this.computeSpline();
            }
        }

        @Override
        public boolean hasUserDataPoints() {
            return this.mUserLux != -1.0f;
        }

        @Override
        public boolean isDefaultConfig() {
            return this.mDefaultConfig.equals(this.mConfig);
        }

        @Override
        public BrightnessConfiguration getDefaultConfig() {
            return this.mDefaultConfig;
        }

        @Override
        public void dump(PrintWriter pw) {
            pw.println("PhysicalMappingStrategy");
            pw.println("  mConfig=" + this.mConfig);
            pw.println("  mBrightnessSpline=" + this.mBrightnessSpline);
            pw.println("  mNitsToBacklightSpline=" + this.mNitsToBacklightSpline);
            pw.println("  mMaxGamma=" + this.mMaxGamma);
            pw.println("  mAutoBrightnessAdjustment=" + this.mAutoBrightnessAdjustment);
            pw.println("  mUserLux=" + this.mUserLux);
            pw.println("  mUserBrightness=" + this.mUserBrightness);
        }

        private void computeSpline() {
            Pair<float[], float[]> defaultCurve = this.mConfig.getCurve();
            float[] defaultLux = (float[])defaultCurve.first;
            float[] defaultNits = (float[])defaultCurve.second;
            float[] defaultBacklight = new float[defaultNits.length];
            for (int i = 0; i < defaultBacklight.length; ++i) {
                defaultBacklight[i] = this.mNitsToBacklightSpline.interpolate(defaultNits[i]);
            }
            Pair curve = BrightnessMappingStrategy.getAdjustedCurve(defaultLux, defaultBacklight, this.mUserLux, this.mUserBrightness, this.mAutoBrightnessAdjustment, this.mMaxGamma);
            float[] lux = (float[])curve.first;
            float[] backlight = (float[])curve.second;
            float[] nits = new float[backlight.length];
            for (int i = 0; i < nits.length; ++i) {
                nits[i] = this.mBacklightToNitsSpline.interpolate(backlight[i]);
            }
            this.mBrightnessSpline = Spline.createSpline(lux, nits);
        }

        private float getUnadjustedBrightness(float lux) {
            Pair<float[], float[]> curve = this.mConfig.getCurve();
            Spline spline = Spline.createSpline((float[])curve.first, (float[])curve.second);
            return this.mNitsToBacklightSpline.interpolate(spline.interpolate(lux));
        }
    }

    private static class SimpleMappingStrategy
    extends BrightnessMappingStrategy {
        private final float[] mLux;
        private final float[] mBrightness;
        private Spline mSpline;
        private float mMaxGamma;
        private float mAutoBrightnessAdjustment;
        private float mUserLux;
        private float mUserBrightness;

        public SimpleMappingStrategy(float[] lux, int[] brightness, float maxGamma) {
            Preconditions.checkArgument(lux.length != 0 && brightness.length != 0, "Lux and brightness arrays must not be empty!");
            Preconditions.checkArgument(lux.length == brightness.length, "Lux and brightness arrays must be the same length!");
            Preconditions.checkArrayElementsInRange(lux, 0.0f, Float.MAX_VALUE, "lux");
            Preconditions.checkArrayElementsInRange(brightness, 0, Integer.MAX_VALUE, "brightness");
            int N = brightness.length;
            this.mLux = new float[N];
            this.mBrightness = new float[N];
            for (int i = 0; i < N; ++i) {
                this.mLux[i] = lux[i];
                this.mBrightness[i] = BrightnessMappingStrategy.normalizeAbsoluteBrightness(brightness[i]);
            }
            this.mMaxGamma = maxGamma;
            this.mAutoBrightnessAdjustment = 0.0f;
            this.mUserLux = -1.0f;
            this.mUserBrightness = -1.0f;
            this.computeSpline();
        }

        @Override
        public boolean setBrightnessConfiguration(BrightnessConfiguration config) {
            return false;
        }

        @Override
        public float getBrightness(float lux) {
            return this.mSpline.interpolate(lux);
        }

        @Override
        public float getAutoBrightnessAdjustment() {
            return this.mAutoBrightnessAdjustment;
        }

        @Override
        public boolean setAutoBrightnessAdjustment(float adjustment) {
            if ((adjustment = MathUtils.constrain(adjustment, -1.0f, 1.0f)) == this.mAutoBrightnessAdjustment) {
                return false;
            }
            this.mAutoBrightnessAdjustment = adjustment;
            this.computeSpline();
            return true;
        }

        @Override
        public float convertToNits(int backlight) {
            return -1.0f;
        }

        @Override
        public void addUserDataPoint(float lux, float brightness) {
            float adjustment;
            float unadjustedBrightness = this.getUnadjustedBrightness(lux);
            this.mAutoBrightnessAdjustment = adjustment = BrightnessMappingStrategy.inferAutoBrightnessAdjustment(this.mMaxGamma, brightness, unadjustedBrightness);
            this.mUserLux = lux;
            this.mUserBrightness = brightness;
            this.computeSpline();
        }

        @Override
        public void clearUserDataPoints() {
            if (this.mUserLux != -1.0f) {
                this.mAutoBrightnessAdjustment = 0.0f;
                this.mUserLux = -1.0f;
                this.mUserBrightness = -1.0f;
                this.computeSpline();
            }
        }

        @Override
        public boolean hasUserDataPoints() {
            return this.mUserLux != -1.0f;
        }

        @Override
        public boolean isDefaultConfig() {
            return true;
        }

        @Override
        public BrightnessConfiguration getDefaultConfig() {
            return null;
        }

        @Override
        public void dump(PrintWriter pw) {
            pw.println("SimpleMappingStrategy");
            pw.println("  mSpline=" + this.mSpline);
            pw.println("  mMaxGamma=" + this.mMaxGamma);
            pw.println("  mAutoBrightnessAdjustment=" + this.mAutoBrightnessAdjustment);
            pw.println("  mUserLux=" + this.mUserLux);
            pw.println("  mUserBrightness=" + this.mUserBrightness);
        }

        private void computeSpline() {
            Pair curve = BrightnessMappingStrategy.getAdjustedCurve(this.mLux, this.mBrightness, this.mUserLux, this.mUserBrightness, this.mAutoBrightnessAdjustment, this.mMaxGamma);
            this.mSpline = Spline.createSpline((float[])curve.first, (float[])curve.second);
        }

        private float getUnadjustedBrightness(float lux) {
            Spline spline = Spline.createSpline(this.mLux, this.mBrightness);
            return spline.interpolate(lux);
        }
    }
}

