package org.richfaces.renderkit;

import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import javax.el.ValueExpression;
import javax.faces.FactoryFinder;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.render.RenderKit;
import javax.faces.render.RenderKitFactory;
import javax.servlet.ServletRequest;
import javax.servlet.http.HttpSession;

import org.ajax4jsf.Filter;
import org.ajax4jsf.context.AjaxContext;
import org.ajax4jsf.context.AjaxContextImpl;
import org.ajax4jsf.event.AjaxEvent;
import org.ajax4jsf.exception.FileUploadException;
import org.ajax4jsf.javascript.JSFunction;
import org.ajax4jsf.javascript.JSFunctionDefinition;
import org.ajax4jsf.javascript.JSLiteral;
import org.ajax4jsf.javascript.JSReference;
import org.ajax4jsf.javascript.ScriptUtils;
import org.ajax4jsf.renderkit.AjaxRendererUtils;
import org.ajax4jsf.renderkit.RendererUtils;
import org.ajax4jsf.request.MultipartRequest;
import org.ajax4jsf.resource.CountingOutputWriter;
import org.richfaces.component.UIFileUpload;
import org.richfaces.component.UIProgressBar;
import org.richfaces.event.UploadEvent;
import org.richfaces.model.UploadItem;

/**
 * Class provides base renderer for upload file component
 * 
 * @author "Andrey Markavtsov"
 * 
 */
public abstract class FileUploadRendererBase extends
	TemplateEncoderRendererBase {

    /** Attribute name where collection of files uploaded will be stored */
    private static final String _FILES_UPLOADED_ATTRIBUTE_NAME = "uploadData";
    
    /** Multipart request class name */
    private static final String _MULTIPART_REQUEST_CLASS_NAME = MultipartRequest.class.getName();

    /*
     * (non-Javadoc)
     * 
     * @see org.ajax4jsf.renderkit.RendererBase#doDecode(javax.faces.context.FacesContext,
     *      javax.faces.component.UIComponent)
     */
    @Override
    protected void doDecode(FacesContext context, UIComponent component) {
	UIFileUpload fileUpload = (UIFileUpload) component;
	ServletRequest request = (ServletRequest) context.getExternalContext()
		.getRequest();

	String clientId = component.getClientId(context);

	new AjaxEvent(component).queue();
	
	Class requestClazz = request.getClass();
	
	if (_MULTIPART_REQUEST_CLASS_NAME
		.equals(requestClazz.getName())
		|| "org.jboss.seam.web.MultipartRequest".equals(requestClazz.getName())) {
	    
	    HttpSession session = (HttpSession)context.getExternalContext().getSession(false);
	    String uid = request.getParameter(Filter.UPLOAD_FILES_ID);
	    Map<String, MultipartRequest> map = (Map<String, MultipartRequest>)session.getAttribute(Filter.REQUESTS_SESSIONS_BEAN_NAME);
	    
	    MultipartRequest multipartRequest = map.get(uid);
	    
	    clientId = clientId + ":file";
	    String fileName = multipartRequest.getFileName(clientId);
	    Object file = multipartRequest.getFile(clientId);
	    fileUpload.setLocalFileName(fileName);
	    onUploadComplete(context, file, fileUpload);
	}

    }

    /**
     * Method checks if uploaded files count overflowed
     * 
     * @param fileUpload
     * @param map
     * @return
     */
    private boolean checkFileCount(UIFileUpload fileUpload,
	    Collection filesUploaded) {
	Integer max = fileUpload.getMaxFilesQuantity();
	if (filesUploaded != null && filesUploaded.size() >= max) {
	    fileUpload.reset();
	    return false;
	}
	return true;
    }

    /**
     * Put uploaded file into data attribute defined
     * 
     * @param context
     * @param fileUpload
     * @param file
     */
    @SuppressWarnings("unchecked")
    private void storeData(FacesContext context, UIFileUpload fileUpload,
	    Object file) {
	ValueExpression data = fileUpload
		.getValueExpression(_FILES_UPLOADED_ATTRIBUTE_NAME);
	if (data != null) {
	    if (data.getValue(context.getELContext()) instanceof Collection) {
		Collection collection = (Collection) data.getValue(context
			.getELContext());
		if (checkFileCount(fileUpload, collection)) {
		    UploadItem item = new UploadItem(fileUpload
			    .getLocalFileName(), file);
		    new UploadEvent(fileUpload, item).queue();
		    collection.add(item);
		} else {
		    throw new FileUploadException("Files count overflow");
		}
	    }
	}
    }

    /**
     * Call with method after uploading completed
     * 
     * @param context
     * @param file
     * @param fileUpload
     */
    private void onUploadComplete(FacesContext context, Object file,
	    UIFileUpload fileUpload) {
	storeData(context, fileUpload, file);
	try {
	    AjaxContext ajaxContext = AjaxContextImpl
		    .getCurrentInstance(context);
	    ajaxContext.setAjaxRequest(true);
	} catch (Exception e) {
	    e.getMessage();
	}
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ajax4jsf.renderkit.RendererBase#getComponentClass()
     */
    @Override
    protected Class<? extends UIComponent> getComponentClass() {
	return UIFileUpload.class;
    }

    /**
     * Generates common JS script by action value
     * 
     * @param context
     * @param component
     * @param action
     * @param oncomplete
     * @return
     * @throws IOException
     */
    @SuppressWarnings("unchecked")
    private String getActionScript(FacesContext context, UIComponent component,
	    String action, Object oncomplete) throws IOException {
	JSFunction ajaxFunction = AjaxRendererUtils.buildAjaxFunction(
		component, context);
	String clientId = component.getClientId(context);
	Map options = AjaxRendererUtils.buildEventOptions(context, component);
	Map parameters = (Map) options.get("parameters");
	parameters.put("action", action);
	parameters.put(Filter.UPLOAD_FILES_ID, new JSReference("uid"));
	parameters.put(clientId, clientId);
	if (oncomplete != null) {
	    options.put("onbeforedomupdate", oncomplete);
	}
	ajaxFunction.addParameter(options);

	JSFunctionDefinition function = new JSFunctionDefinition("uid");
	function.addParameter("event");
	function.addToBody(ajaxFunction.toScript());

	return function.toScript();
    }

    /**
     * Return accepted types map
     * 
     * @param context
     * @param component
     * @return
     */
    public Object getAcceptedTypes(FacesContext context, UIComponent component) {
	String acceptedTypes = (String) component.getAttributes().get(
		"acceptedTypes");
	if (acceptedTypes != null) {
	    Map<String, Boolean> accepted = new HashMap<String, Boolean>();
	    String[] types = acceptedTypes.split("[,;|]");
	    if (types != null) {
		for (String type : types) {
		    accepted.put(type.toLowerCase(), true);
		}
		return ScriptUtils.toScript(accepted);
	    }
	}
	return JSReference.NULL;
    }

    /**
     * Generates JS script for stopping uploading process
     * 
     * @param context
     * @param component
     * @return
     * @throws IOException
     */
    public String getStopScript(FacesContext context, UIComponent component)
	    throws IOException {
	return getActionScript(context, component, "stop", null);
    }

    /**
     * Generates JS script for getting file size from server
     * 
     * @param context
     * @param component
     * @return
     * @throws IOException
     */
    public String getFileSizeScript(FacesContext context, UIComponent component)
	    throws IOException {
	JSFunctionDefinition oncomplete = new JSFunctionDefinition();
	oncomplete.addParameter("request");
	oncomplete.addParameter("event");
	oncomplete.addParameter("data");
	StringBuffer body = new StringBuffer("$('");
	body.append(component.getClientId(context));
	body.append("').component.getFileSize(data);");
	oncomplete.addToBody(body);
	return getActionScript(context, component, "progress", oncomplete);

    }

    /**
     * Generates progress label markup
     * 
     * @param context
     * @param component
     * @return
     * @throws IOException
     */
    public Object getLabelMarkup(FacesContext context, UIComponent component)
	    throws IOException {
	CountingOutputWriter customWriter = new CountingOutputWriter();
	StringBuffer result = null;
	UIComponent label = component.getFacet("label");
	try {
	    if (label != null) {

		ResponseWriter writer = context.getResponseWriter();

		String defaultRenderKitId = context.getApplication()
			.getDefaultRenderKitId();
		if (null == defaultRenderKitId) {
		    defaultRenderKitId = RenderKitFactory.HTML_BASIC_RENDER_KIT;
		}
		RenderKitFactory renderKitFactory = (RenderKitFactory) FactoryFinder
			.getFactory(FactoryFinder.RENDER_KIT_FACTORY);
		RenderKit renderKit = renderKitFactory.getRenderKit(context,
			defaultRenderKitId);

		ResponseWriter responseWriter = renderKit.createResponseWriter(
			customWriter, null, "UTF-8");
		context.setResponseWriter(responseWriter);
		writeScriptBody(context, label, false);
		if (writer != null) {
		    context.setResponseWriter(writer);
		}
		result = customWriter.getContent();
	    }

	} catch (Exception e) {
	    e.getMessage();
	}
	return (result != null) ? new JSLiteral(result.toString())
		: JSReference.NULL;
    }

    /**
     * Generate component custom events functions
     * 
     * @param context
     * @param component
     * @param attributeName
     * @return
     */
    public String getAsEventHandler(FacesContext context,
	    UIComponent component, String attributeName) {
	Object eventHandler = RendererUtils.getInstance().getAsEventHandler(
		context, component, attributeName, "");
	if (eventHandler != null) {
	    return eventHandler.toString();
	}
	return JSReference.NULL.toScript();
    }

    /**
     * Gets progress bar Id
     * 
     * @param context
     * @param component
     * @return
     * @throws IOException
     */
    public String getProgressBarId(FacesContext context, UIComponent component)
	    throws IOException {
	return getProgressBar(context, component).getClientId(context);
    }

    /**
     * Renders progress bar
     * 
     * @param context
     * @param component
     * @throws IOException
     */
    public void renderProgress(FacesContext context, UIComponent component)
	    throws IOException {
	UIComponent progressBar = getProgressBar(context, component);
	renderChild(context, progressBar);
    }

    /**
     * Creates progress bar component
     * 
     * @param context
     * @param fileUpload
     * @return
     */
    private UIComponent createProgressBar(FacesContext context,
	    UIComponent fileUpload) {
	UIComponent progressBar = fileUpload.getFacet("progress");
	if (null == progressBar) {
	    progressBar = context.getApplication().createComponent(
		    UIProgressBar.COMPONENT_TYPE);
	}
	fileUpload.getFacets().put("progress", progressBar);
	return progressBar;
    }

    /**
     * Returns progress bar
     * 
     * @param context
     * @param component
     * @return
     */
    public UIComponent getProgressBar(FacesContext context,
	    UIComponent component) {
	UIComponent progressBar = component.getFacet("progress");
	if (null == progressBar) {
	    progressBar = createProgressBar(context, component);
	}
	progressBar.getAttributes().put("minValue", -1);
	progressBar.getAttributes().put("enabled", false);
	progressBar.setTransient(false);
	return progressBar;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.richfaces.renderkit.TemplateEncoderRendererBase#encodeChildren(javax.faces.context.FacesContext,
     *      javax.faces.component.UIComponent)
     */
    @Override
    public void encodeChildren(FacesContext context, UIComponent component)
	    throws IOException {
	; // We should not render children
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ajax4jsf.renderkit.RendererBase#doEncodeChildren(javax.faces.context.ResponseWriter,
     *      javax.faces.context.FacesContext, javax.faces.component.UIComponent)
     */
    @Override
    public void doEncodeChildren(ResponseWriter writer, FacesContext context,
	    UIComponent component) throws IOException {
	; // We should not render children
    }

}
