/*
 * Decompiled with CFR 0.152.
 */
package org.rhq.cassandra.schema;

import com.datastax.driver.core.Host;
import com.datastax.driver.core.PreparedStatement;
import com.datastax.driver.core.Query;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.ResultSetFuture;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.google.common.base.Stopwatch;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.RateLimiter;
import java.io.File;
import java.io.IOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.rhq.cassandra.schema.DBConnectionFactory;
import org.rhq.cassandra.schema.MigrationLog;
import org.rhq.cassandra.schema.SchemaUpdateThreadFactory;
import org.rhq.cassandra.schema.Step;

public class MigrateAggregateMetrics
implements Step {
    private static final Log log = LogFactory.getLog(MigrateAggregateMetrics.class);
    private Session session;
    private DBConnectionFactory dbConnectionFactory;
    private PreparedStatement find1HourData;
    private PreparedStatement find6HourData;
    private PreparedStatement find24HourData;
    private RateLimiter writePermits;
    private Semaphore readPermits = new Semaphore(1);
    private AtomicInteger failedMigrations = new AtomicInteger();
    private ListeningExecutorService threadPool = MoreExecutors.listeningDecorator((ExecutorService)Executors.newFixedThreadPool(4, new SchemaUpdateThreadFactory()));
    private String dataDir;

    @Override
    public void setSession(Session session) {
        this.session = session;
    }

    @Override
    public void bind(Properties properties) {
        this.dbConnectionFactory = (DBConnectionFactory)properties.get("relational_db_connection_factory");
        this.dataDir = properties.getProperty("data.dir", System.getProperty("jboss.server.data.dir"));
    }

    public String toString() {
        return this.getClass().getSimpleName();
    }

    @Override
    public void execute() {
        if (this.dbConnectionFactory == null) {
            log.info((Object)"The relational database connection factory is not set. No data migration necessary");
        } else {
            this.writePermits = RateLimiter.create((double)this.calculatePermits(), (long)30L, (TimeUnit)TimeUnit.SECONDS);
            Stopwatch stopwatch = new Stopwatch().start();
            this.initPreparedStatements();
            Set<Integer> scheduleIds = this.loadScheduleIds();
            log.info((Object)("Migrating aggregate metrics for " + scheduleIds.size() + " schedule ids"));
            this.migrate(scheduleIds, this.find1HourData, Bucket.ONE_HOUR);
            this.migrate(scheduleIds, this.find6HourData, Bucket.SIX_HOUR);
            this.migrate(scheduleIds, this.find24HourData, Bucket.TWENTY_FOUR_HOUR);
            stopwatch.stop();
            log.info((Object)("Finished aggregate metrics migration in " + stopwatch.elapsed(TimeUnit.SECONDS) + " seconds"));
            if (this.failedMigrations.get() > 0) {
                throw new RuntimeException("There were " + this.failedMigrations.get() + " failed migrations. The " + "upgrade will have to be run again to complete the migration.");
            }
        }
        this.dropTables();
    }

    private int calculatePermits() {
        int requestLimit = Integer.parseInt(System.getProperty("rhq.storage.request.limit", "20000"));
        return requestLimit * this.getNumberOfUpNodes();
    }

    private int getNumberOfUpNodes() {
        int count = 0;
        for (Host host : this.session.getCluster().getMetadata().getAllHosts()) {
            if (!host.isUp()) continue;
            ++count;
        }
        return count;
    }

    private void migrate(Set<Integer> scheduleIds, PreparedStatement query, Bucket bucket) {
        log.info((Object)("Migrating " + (Object)((Object)bucket) + " data for " + scheduleIds.size() + " schedules"));
        CountDownLatch latch = new CountDownLatch(scheduleIds.size());
        MigrationProgressLogger progressLogger = new MigrationProgressLogger(bucket, latch);
        File logFile = new File(this.dataDir, (Object)((Object)bucket) + "_migration.log");
        MigrationLog migrationLog = null;
        try {
            migrationLog = new MigrationLog(logFile);
            Set<Integer> migratedScheduleIds = migrationLog.read();
            this.threadPool.submit((Runnable)progressLogger);
            for (Integer scheduleId : scheduleIds) {
                if (migratedScheduleIds.contains(scheduleId)) {
                    log.debug((Object)((Object)((Object)bucket) + " data for schedule id " + scheduleId + " has already been migrated. It will " + "be skipped."));
                    latch.countDown();
                    continue;
                }
                this.readPermits.acquire();
                ResultSet resultSet = this.session.execute((Query)query.bind(new Object[]{scheduleId}));
                ListenableFuture migrationFuture = this.threadPool.submit((Callable)new MetricsWriter(scheduleId, bucket, resultSet));
                Futures.addCallback((ListenableFuture)migrationFuture, this.migrationFinished(scheduleId, bucket, latch, migrationLog));
            }
            latch.await();
            log.info((Object)("Finished migrating " + (Object)((Object)bucket) + " data"));
        }
        catch (InterruptedException e) {
            this.threadPool.shutdownNow();
            throw new RuntimeException("Migration of " + (Object)((Object)bucket) + " data did not complete due to an interrupt. The " + "upgrade will have to be run again to finish the migration", e);
        }
        catch (IOException e) {
            throw new RuntimeException("Migration of " + (Object)((Object)bucket) + " data did not complete due to an I/O error. The " + "upgrade will have to be run again to finish the migration", e);
        }
        finally {
            progressLogger.finished();
            try {
                migrationLog.close();
            }
            catch (IOException e) {
                log.warn((Object)("There was an error closing " + logFile.getAbsolutePath()), (Throwable)e);
            }
        }
    }

    private void initPreparedStatements() {
        this.find1HourData = this.session.prepare("SELECT schedule_id, time, type, value, ttl(value), writetime(value) FROM rhq.one_hour_metrics WHERE schedule_id = ?");
        this.find6HourData = this.session.prepare("SELECT schedule_id, time, type, value, ttl(value), writetime(value) FROM rhq.six_hour_metrics WHERE schedule_id = ?");
        this.find24HourData = this.session.prepare("SELECT schedule_id, time, type, value, ttl(value), writetime(value) FROM rhq.twenty_four_hour_metrics WHERE schedule_id = ?");
    }

    private Set<Integer> loadScheduleIds() {
        Connection connection = null;
        Statement statement = null;
        java.sql.ResultSet resultSet = null;
        try {
            connection = this.dbConnectionFactory.newConnection();
            statement = connection.createStatement();
            resultSet = statement.executeQuery("SELECT s.id FROM rhq_measurement_sched s INNER JOIN rhq_measurement_def d on s.definition = d.id WHERE d.data_type = 0");
            HashSet<Integer> scheduleIds = new HashSet<Integer>();
            while (resultSet.next()) {
                scheduleIds.add(resultSet.getInt(1));
            }
            HashSet<Integer> hashSet = scheduleIds;
            return hashSet;
        }
        catch (SQLException e) {
            throw new RuntimeException("Cannot migrate aggregate metrics. There was an error loading schedule ids", e);
        }
        finally {
            if (resultSet != null) {
                try {
                    resultSet.close();
                }
                catch (SQLException e) {
                    log.info((Object)"There was an error closing the SQL result set", (Throwable)e);
                }
            }
            if (statement != null) {
                try {
                    statement.close();
                }
                catch (SQLException e) {
                    log.info((Object)"There was an error closing the SQL statement", (Throwable)e);
                }
            }
            if (connection != null) {
                try {
                    connection.close();
                }
                catch (SQLException e) {
                    log.info((Object)"There was an error closing the SQL connection", (Throwable)e);
                }
            }
        }
    }

    private void dropTables() {
        ResultSet resultSet = this.session.execute("SELECT columnfamily_name FROM system.schema_columnfamilies WHERE keyspace_name = 'rhq'");
        for (Row row : resultSet) {
            String table = row.getString(0);
            if (!table.equals("one_hour_metrics") && !table.equals("six_hour_metrics") && !table.equals("twenty_four_hour_metrics")) continue;
            log.info((Object)("Dropping table " + table));
            this.session.execute("DROP table rhq." + table);
        }
    }

    private FutureCallback<Integer> migrationFinished(final Integer scheduleId, final Bucket bucket, final CountDownLatch latch, final MigrationLog migrationLog) {
        return new FutureCallback<Integer>(){

            public void onSuccess(Integer metricsWritten) {
                latch.countDown();
                MigrateAggregateMetrics.this.readPermits.release();
                try {
                    migrationLog.write(scheduleId);
                }
                catch (IOException e) {
                    log.warn((Object)("Failed to update migration log for bucket " + (Object)((Object)bucket) + " and schedule id " + scheduleId));
                }
            }

            public void onFailure(Throwable t) {
                latch.countDown();
                MigrateAggregateMetrics.this.readPermits.release();
                MigrateAggregateMetrics.this.failedMigrations.incrementAndGet();
            }
        };
    }

    private class MigrationProgressLogger
    implements Runnable {
        private Bucket bucket;
        private CountDownLatch latch;
        private boolean finished;

        public MigrationProgressLogger(Bucket bucket, CountDownLatch latch) {
            this.bucket = bucket;
            this.latch = latch;
        }

        public void finished() {
            this.finished = true;
        }

        @Override
        public void run() {
            try {
                while (!this.finished && this.latch.getCount() > 0L) {
                    log.info((Object)("There are " + this.latch.getCount() + " remaining schedules for the " + (Object)((Object)this.bucket) + " data migration"));
                    Thread.sleep(30000L);
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
    }

    private class MetricsWriter
    implements Callable<Integer>,
    FutureCallback<ResultSet> {
        private Integer scheduleId;
        private Bucket bucket;
        private ResultSet resultSet;
        private boolean writeFailed;
        private AtomicInteger metricsMigrated = new AtomicInteger();

        public MetricsWriter(Integer scheduleId, Bucket bucket, ResultSet resultSet) {
            this.scheduleId = scheduleId;
            this.bucket = bucket;
            this.resultSet = resultSet;
        }

        @Override
        public Integer call() throws Exception {
            List rows = this.resultSet.all();
            if (rows.isEmpty()) {
                log.debug((Object)("No " + (Object)((Object)this.bucket) + " data to migrate for schedule id " + this.scheduleId));
                return 0;
            }
            Date time = ((Row)rows.get(0)).getDate(1);
            Double max = null;
            Double min = null;
            Double avg = null;
            Long writeTime = ((Row)rows.get(0)).getLong(5);
            Integer ttl = ((Row)rows.get(0)).getInt(4);
            for (Row row : rows) {
                if (this.writeFailed) {
                    throw new Exception("Migration of " + (Object)((Object)this.bucket) + " data for schedule id " + this.scheduleId + " failed");
                }
                Date nextTime = row.getDate(1);
                if (nextTime.equals(time)) {
                    int type = row.getInt(2);
                    switch (type) {
                        case 0: {
                            max = row.getDouble(3);
                            break;
                        }
                        case 1: {
                            min = row.getDouble(3);
                            break;
                        }
                        default: {
                            avg = row.getDouble(3);
                            break;
                        }
                    }
                    continue;
                }
                if (this.isDataMissing(avg, max, min)) {
                    log.debug((Object)("We only have a partial " + (Object)((Object)this.bucket) + " metric for {scheduleId: " + this.scheduleId + ", time: " + time.getTime() + "}. It will not be migrated."));
                } else {
                    ResultSetFuture writeFuture = this.writeMetrics(time, avg, max, min, ttl, writeTime);
                    Futures.addCallback((ListenableFuture)writeFuture, (FutureCallback)this);
                }
                time = nextTime;
                max = row.getDouble(3);
                min = null;
                avg = null;
                ttl = row.getInt(4);
                writeTime = row.getLong(5);
            }
            if (this.writeFailed) {
                throw new Exception("Migration of " + (Object)((Object)this.bucket) + " data for schedule id " + this.scheduleId + " failed");
            }
            return this.metricsMigrated.get();
        }

        private boolean isDataMissing(Double avg, Double max, Double min) {
            if (avg == null || Double.isNaN(avg)) {
                return true;
            }
            if (max == null || Double.isNaN(max)) {
                return true;
            }
            return min == null || Double.isNaN(min);
        }

        public void onSuccess(ResultSet resultSet) {
            this.metricsMigrated.incrementAndGet();
        }

        public void onFailure(Throwable t) {
            this.writeFailed = true;
            log.warn((Object)("Migration of " + (Object)((Object)this.bucket) + " data for schedule id " + this.scheduleId + " failed"), t);
        }

        private ResultSetFuture writeMetrics(Date time, Double avg, Double max, Double min, Integer ttl, Long writeTime) {
            MigrateAggregateMetrics.this.writePermits.acquire();
            return MigrateAggregateMetrics.this.session.executeAsync("INSERT INTO rhq.aggregate_metrics(schedule_id, bucket, time, avg, max, min) VALUES (" + this.scheduleId + ", '" + (Object)((Object)this.bucket) + "', " + time.getTime() + ", " + avg + ", " + max + ", " + min + ") USING TTL " + ttl + " AND TIMESTAMP " + writeTime);
        }
    }

    private static enum Bucket {
        ONE_HOUR("one_hour"),
        SIX_HOUR("six_hour"),
        TWENTY_FOUR_HOUR("twenty_four_hour");

        private String tableName;

        private Bucket(String tableName) {
            this.tableName = tableName;
        }

        public String toString() {
            return this.tableName;
        }
    }
}

