/*
 * Decompiled with CFR 0.152.
 */
package org.rhq.server.metrics;

import com.datastax.driver.core.ResultSet;
import com.google.common.base.Function;
import com.google.common.base.Stopwatch;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.joda.time.DateTime;
import org.joda.time.Days;
import org.joda.time.ReadableInstant;
import org.rhq.core.domain.measurement.MeasurementDataNumeric;
import org.rhq.core.domain.measurement.composite.MeasurementDataNumericHighLowComposite;
import org.rhq.core.util.exception.ThrowableUtil;
import org.rhq.server.metrics.ArithmeticMeanCalculator;
import org.rhq.server.metrics.Buckets;
import org.rhq.server.metrics.ComputeAggregate;
import org.rhq.server.metrics.ComputeRawAggregate;
import org.rhq.server.metrics.DateTimeService;
import org.rhq.server.metrics.MetricsConfiguration;
import org.rhq.server.metrics.MetricsDAO;
import org.rhq.server.metrics.RawDataInsertedCallback;
import org.rhq.server.metrics.StorageClientThreadFactory;
import org.rhq.server.metrics.StorageResultSetFuture;
import org.rhq.server.metrics.aggregation.AggregationManager;
import org.rhq.server.metrics.domain.AggregateNumericMetric;
import org.rhq.server.metrics.domain.AggregateNumericMetricMapper;
import org.rhq.server.metrics.domain.Bucket;
import org.rhq.server.metrics.domain.IndexBucket;
import org.rhq.server.metrics.domain.RawNumericMetric;
import org.rhq.server.metrics.invalid.InvalidMetricsManager;

public class MetricsServer {
    private final Log log = LogFactory.getLog(MetricsServer.class);
    private static final int RAW_DATA_AGE_LIMIT_MAX = 5;
    private DateTimeService dateTimeService = new DateTimeService();
    private MetricsDAO dao;
    private MetricsConfiguration configuration;
    private ListeningExecutorService tasks = MoreExecutors.listeningDecorator((ExecutorService)Executors.newCachedThreadPool(new StorageClientThreadFactory("MetricsServerTasks")));
    private InvalidMetricsManager invalidMetricsManager;
    private AggregationManager aggregationManager;
    private Days rawDataAgeLimit = Days.days((int)Math.min(3, Integer.parseInt(System.getProperty("rhq.metrics.data.age-limit", "3"))));

    public void setDAO(MetricsDAO dao) {
        this.dao = dao;
    }

    public void setConfiguration(MetricsConfiguration configuration) {
        this.configuration = configuration;
    }

    public void setDateTimeService(DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public int getRawDataAgeLimit() {
        return this.rawDataAgeLimit.getDays();
    }

    public void setRawDataAgeLimit(int rawDataAgeLimit) {
        if (rawDataAgeLimit > 5) {
            throw new IllegalArgumentException("The requested limit, " + rawDataAgeLimit + ", exceeds the max age " + "limit of " + 5);
        }
        this.rawDataAgeLimit = Days.days((int)rawDataAgeLimit);
    }

    public void setIndexPartitions(int indexPartitions) {
        this.configuration.setIndexPartitions(indexPartitions);
    }

    public void init() {
        this.aggregationManager = new AggregationManager(this.dao, this.dateTimeService, this.configuration);
        this.invalidMetricsManager = new InvalidMetricsManager(this.dateTimeService, this.dao);
    }

    InvalidMetricsManager getInvalidMetricsManager() {
        return this.invalidMetricsManager;
    }

    public AggregationManager getAggregationManager() {
        return this.aggregationManager;
    }

    public void shutdown() {
        this.aggregationManager.shutdown();
        this.invalidMetricsManager.shutdown();
    }

    public RawNumericMetric findLatestValueForResource(int scheduleId) {
        this.log.debug((Object)("Querying for most recent raw metrics for [scheduleId: " + scheduleId + "]"));
        return this.dao.findLatestRawMetric(scheduleId);
    }

    public Iterable<MeasurementDataNumericHighLowComposite> findDataForResource(int scheduleId, long beginTime, long endTime, int numberOfBuckets) {
        Stopwatch stopwatch = new Stopwatch().start();
        try {
            DateTime begin = new DateTime(beginTime);
            if (this.dateTimeService.isInRawDataRange(begin)) {
                List<RawNumericMetric> metrics = this.dao.findRawMetrics(scheduleId, beginTime, endTime);
                List<MeasurementDataNumericHighLowComposite> list = this.createRawComposites(metrics, beginTime, endTime, numberOfBuckets);
                return list;
            }
            List<AggregateNumericMetric> metrics = null;
            if (this.dateTimeService.isIn1HourDataRange(begin)) {
                metrics = this.dao.findAggregateMetrics(scheduleId, Bucket.ONE_HOUR, beginTime, endTime);
                List<MeasurementDataNumericHighLowComposite> list = this.createComposites(metrics, beginTime, endTime, numberOfBuckets);
                return list;
            }
            if (this.dateTimeService.isIn6HourDataRange(begin)) {
                metrics = this.dao.findAggregateMetrics(scheduleId, Bucket.SIX_HOUR, beginTime, endTime);
                List<MeasurementDataNumericHighLowComposite> list = this.createComposites(metrics, beginTime, endTime, numberOfBuckets);
                return list;
            }
            if (this.dateTimeService.isIn24HourDataRange(begin)) {
                metrics = this.dao.findAggregateMetrics(scheduleId, Bucket.TWENTY_FOUR_HOUR, beginTime, endTime);
                List<MeasurementDataNumericHighLowComposite> list = this.createComposites(metrics, beginTime, endTime, numberOfBuckets);
                return list;
            }
            throw new IllegalArgumentException("beginTime[" + beginTime + "] is outside the accepted range.");
        }
        finally {
            stopwatch.stop();
            if (this.log.isDebugEnabled()) {
                this.log.debug((Object)("Finished calculating resource summary aggregate in " + stopwatch.elapsed(TimeUnit.MILLISECONDS) + " ms"));
            }
        }
    }

    public List<MeasurementDataNumericHighLowComposite> findDataForGroup(List<Integer> scheduleIds, long beginTime, long endTime, int numberOfBuckets) {
        DateTime begin;
        if (this.log.isDebugEnabled()) {
            this.log.debug((Object)("Querying for metric data using parameters [scheduleIds: " + scheduleIds + ", beingTime: " + beginTime + ", endTime: " + endTime + ", numberOfBuckets: " + numberOfBuckets + "]"));
        }
        if (this.dateTimeService.isInRawDataRange(begin = new DateTime(beginTime))) {
            Iterable<RawNumericMetric> metrics = this.dao.findRawMetrics(scheduleIds, beginTime, endTime);
            return this.createRawComposites(metrics, beginTime, endTime, numberOfBuckets);
        }
        Bucket bucket = this.getBucket(begin);
        List<AggregateNumericMetric> metrics = this.loadMetrics(scheduleIds, beginTime, endTime, bucket);
        return this.createComposites(metrics, beginTime, endTime, numberOfBuckets);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public AggregateNumericMetric getSummaryAggregate(int scheduleId, long beginTime, long endTime) {
        Stopwatch stopwatch = new Stopwatch().start();
        try {
            DateTime begin = new DateTime(beginTime);
            if (this.dateTimeService.isInRawDataRange(begin)) {
                List<RawNumericMetric> metrics = this.dao.findRawMetrics(scheduleId, beginTime, endTime);
                AggregateNumericMetric aggregateNumericMetric = this.calculateAggregatedRaw(metrics, beginTime);
                return aggregateNumericMetric;
            }
            Bucket bucket = this.getBucket(begin);
            List<AggregateNumericMetric> metrics = this.dao.findAggregateMetrics(scheduleId, bucket, beginTime, endTime);
            AggregateNumericMetric aggregateNumericMetric = this.calculateAggregate(metrics, beginTime, bucket);
            return aggregateNumericMetric;
        }
        finally {
            stopwatch.stop();
            if (this.log.isDebugEnabled()) {
                this.log.debug((Object)("Finished calculating resource summary aggregate for [scheduleId: " + scheduleId + ", beginTime: " + beginTime + ", endTime: " + endTime + "] in " + stopwatch.elapsed(TimeUnit.MILLISECONDS) + " ms"));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ListenableFuture<AggregateNumericMetric> getSummaryAggregateAsync(int scheduleId, long beginTime, long endTime) {
        long start = System.currentTimeMillis();
        try {
            DateTime begin;
            if (this.log.isDebugEnabled()) {
                this.log.debug((Object)("Calculating resource summary aggregate (async) for [scheduleId: " + scheduleId + ", beginTime: " + beginTime + ", endTime: " + endTime + "]"));
            }
            if (this.dateTimeService.isInRawDataRange(begin = new DateTime(beginTime))) {
                StorageResultSetFuture queryFuture = this.dao.findRawMetricsAsync(scheduleId, beginTime, endTime);
                ListenableFuture listenableFuture = Futures.transform((ListenableFuture)queryFuture, (Function)new ComputeRawAggregate(beginTime));
                return listenableFuture;
            }
            Bucket bucket = this.getBucket(begin);
            StorageResultSetFuture queryFuture = this.dao.findAggregateMetricsAsync(scheduleId, bucket, beginTime, endTime);
            ListenableFuture listenableFuture = Futures.transform((ListenableFuture)queryFuture, (Function)new ComputeAggregate(beginTime, bucket));
            return listenableFuture;
        }
        finally {
            long end = System.currentTimeMillis();
            if (this.log.isDebugEnabled()) {
                this.log.debug((Object)("Finished calculating resource summary aggregate (async) in " + (end - start) + " ms"));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public AggregateNumericMetric getSummaryAggregate(List<Integer> scheduleIds, long beginTime, long endTime) {
        Stopwatch stopwatch = new Stopwatch().start();
        try {
            DateTime begin = new DateTime(beginTime);
            if (this.dateTimeService.isInRawDataRange(new DateTime(beginTime))) {
                Iterable<RawNumericMetric> metrics = this.dao.findRawMetrics(scheduleIds, beginTime, endTime);
                AggregateNumericMetric aggregateNumericMetric = this.calculateAggregatedRaw(metrics, beginTime);
                return aggregateNumericMetric;
            }
            Bucket bucket = this.getBucket(begin);
            List<AggregateNumericMetric> metrics = this.loadMetrics(scheduleIds, beginTime, endTime, bucket);
            AggregateNumericMetric aggregateNumericMetric = this.calculateAggregate(metrics, beginTime, bucket);
            return aggregateNumericMetric;
        }
        finally {
            stopwatch.stop();
            if (this.log.isDebugEnabled()) {
                this.log.debug((Object)("Finished calculating group summary aggregate for [scheduleIds: " + scheduleIds + ", beginTime: " + beginTime + ", endTime: " + endTime + "] in " + stopwatch.elapsed(TimeUnit.MILLISECONDS) + " ms"));
            }
        }
    }

    private List<AggregateNumericMetric> loadMetrics(List<Integer> scheduleIds, long begin, long end, Bucket bucket) {
        ArrayList<StorageResultSetFuture> futures = new ArrayList<StorageResultSetFuture>(scheduleIds.size());
        for (Integer scheduleId : scheduleIds) {
            futures.add(this.dao.findAggregateMetricsAsync(scheduleId, bucket, begin, end));
        }
        ListenableFuture resultSetsFuture = Futures.successfulAsList(futures);
        try {
            List resultSets = (List)resultSetsFuture.get();
            AggregateNumericMetricMapper mapper = new AggregateNumericMetricMapper();
            ArrayList<AggregateNumericMetric> metrics = new ArrayList<AggregateNumericMetric>();
            for (ResultSet resultSet : resultSets) {
                metrics.addAll(mapper.mapAll(resultSet));
            }
            return metrics;
        }
        catch (Exception e) {
            this.log.warn((Object)("There was an error while fetching " + (Object)((Object)bucket) + " data for {scheduleIds: " + scheduleIds + ", beginTime: " + begin + ", endTime: " + end + "}"), (Throwable)e);
            return Collections.emptyList();
        }
    }

    protected Bucket getBucket(DateTime begin) {
        Bucket bucket;
        if (this.dateTimeService.isIn1HourDataRange(begin)) {
            bucket = Bucket.ONE_HOUR;
        } else if (this.dateTimeService.isIn6HourDataRange(begin)) {
            bucket = Bucket.SIX_HOUR;
        } else if (this.dateTimeService.isIn24HourDataRange(begin)) {
            bucket = Bucket.TWENTY_FOUR_HOUR;
        } else {
            throw new IllegalArgumentException("beginTime[" + begin.getMillis() + "] is outside the accepted range.");
        }
        return bucket;
    }

    private List<MeasurementDataNumericHighLowComposite> createRawComposites(Iterable<RawNumericMetric> metrics, long beginTime, long endTime, int numberOfBuckets) {
        Buckets buckets = new Buckets(beginTime, endTime, numberOfBuckets);
        for (RawNumericMetric metric : metrics) {
            buckets.insert(metric.getTimestamp(), metric.getValue(), metric.getValue(), metric.getValue());
        }
        ArrayList<MeasurementDataNumericHighLowComposite> data = new ArrayList<MeasurementDataNumericHighLowComposite>();
        for (int i = 0; i < buckets.getNumDataPoints(); ++i) {
            Buckets.Bucket bucket = buckets.get(i);
            data.add(new MeasurementDataNumericHighLowComposite(bucket.getStartTime(), bucket.getAvg(), bucket.getMax(), bucket.getMin()));
        }
        return data;
    }

    private List<MeasurementDataNumericHighLowComposite> createComposites(Iterable<AggregateNumericMetric> metrics, long beginTime, long endTime, int numberOfBuckets) {
        Buckets buckets = new Buckets(beginTime, endTime, numberOfBuckets);
        for (AggregateNumericMetric metric : metrics) {
            if (this.invalidMetricsManager.isInvalidMetric(metric)) {
                this.log.warn((Object)("The " + (Object)((Object)metric.getBucket()) + " metric " + metric + " is invalid. It will be excluded from " + "the results sent to the client and we will attempt to recompute the metric."));
                this.invalidMetricsManager.submit(metric);
                continue;
            }
            buckets.insert(metric.getTimestamp(), metric.getAvg(), metric.getMin(), metric.getMax());
        }
        ArrayList<MeasurementDataNumericHighLowComposite> data = new ArrayList<MeasurementDataNumericHighLowComposite>();
        for (int i = 0; i < buckets.getNumDataPoints(); ++i) {
            Buckets.Bucket bucket = buckets.get(i);
            data.add(new MeasurementDataNumericHighLowComposite(bucket.getStartTime(), bucket.getAvg(), bucket.getMax(), bucket.getMin()));
        }
        return data;
    }

    public void addNumericData(final Set<MeasurementDataNumeric> dataSet, final RawDataInsertedCallback callback) {
        if (this.log.isDebugEnabled()) {
            this.log.debug((Object)("Inserting " + dataSet.size() + " raw metrics"));
        }
        final Stopwatch stopwatch = new Stopwatch().start();
        final AtomicInteger remainingInserts = new AtomicInteger(dataSet.size());
        for (final MeasurementDataNumeric data : dataSet) {
            DateTime collectionTimeSlice = this.dateTimeService.getTimeSlice(new DateTime(data.getTimestamp()), this.configuration.getRawTimeSliceDuration());
            Days days = Days.daysBetween((ReadableInstant)collectionTimeSlice, (ReadableInstant)this.dateTimeService.now());
            if (days.isGreaterThan(this.rawDataAgeLimit)) {
                this.log.info((Object)(data + " is older than the raw data age limit of " + this.rawDataAgeLimit.getDays() + " days. It will not be stored."));
                continue;
            }
            StorageResultSetFuture rawFuture = this.dao.insertRawData(data);
            StorageResultSetFuture indexFuture = this.dao.updateIndex(IndexBucket.RAW, collectionTimeSlice.getMillis(), data.getScheduleId());
            ListenableFuture insertsFuture = Futures.successfulAsList((ListenableFuture[])new ListenableFuture[]{rawFuture, indexFuture});
            Futures.addCallback((ListenableFuture)insertsFuture, (FutureCallback)new FutureCallback<List<ResultSet>>(){

                public void onSuccess(List<ResultSet> result) {
                    callback.onSuccess(data);
                    if (remainingInserts.decrementAndGet() == 0) {
                        stopwatch.stop();
                        if (MetricsServer.this.log.isDebugEnabled()) {
                            MetricsServer.this.log.debug((Object)("Finished inserting " + dataSet.size() + " raw metrics in " + stopwatch.elapsed(TimeUnit.MILLISECONDS) + " ms"));
                        }
                        callback.onFinish();
                    }
                }

                public void onFailure(Throwable t) {
                    if (MetricsServer.this.log.isDebugEnabled()) {
                        MetricsServer.this.log.debug((Object)"An error occurred while inserting raw data", ThrowableUtil.getRootCause((Throwable)t));
                    } else {
                        MetricsServer.this.log.warn((Object)("An error occurred while inserting raw data: " + ThrowableUtil.getRootMessage((Throwable)t)));
                    }
                    callback.onFailure(t);
                }
            }, (Executor)this.tasks);
        }
    }

    public Iterable<AggregateNumericMetric> calculateAggregates() {
        return this.aggregationManager.run();
    }

    private AggregateNumericMetric calculateAggregatedRaw(Iterable<RawNumericMetric> rawMetrics, long timestamp) {
        double min;
        double max = min = Double.NaN;
        int count = 0;
        ArithmeticMeanCalculator mean = new ArithmeticMeanCalculator();
        for (RawNumericMetric metric : rawMetrics) {
            double value = metric.getValue();
            if (count == 0) {
                max = min = value;
            }
            if (value < min) {
                min = value;
            } else if (value > max) {
                max = value;
            }
            mean.add(value);
            ++count;
        }
        return new AggregateNumericMetric(0, Bucket.ONE_HOUR, mean.getArithmeticMean(), min, max, timestamp);
    }

    private AggregateNumericMetric calculateAggregate(Iterable<AggregateNumericMetric> metrics, long timestamp, Bucket bucket) {
        double min;
        double max = min = Double.NaN;
        int count = 0;
        ArithmeticMeanCalculator mean = new ArithmeticMeanCalculator();
        for (AggregateNumericMetric metric : metrics) {
            if (count == 0) {
                min = metric.getMin();
                max = metric.getMax();
            }
            if (metric.getMin() < min) {
                min = metric.getMin();
            }
            if (metric.getMax() > max) {
                max = metric.getMax();
            }
            mean.add(metric.getAvg());
            ++count;
        }
        return new AggregateNumericMetric(0, bucket, mean.getArithmeticMean(), min, max, timestamp);
    }
}

