/*
 * Decompiled with CFR 0.152.
 */
package org.rhq.server.metrics.invalid;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.joda.time.DateTime;
import org.joda.time.ReadableInstant;
import org.rhq.server.metrics.ArithmeticMeanCalculator;
import org.rhq.server.metrics.DateTimeService;
import org.rhq.server.metrics.MetricsConfiguration;
import org.rhq.server.metrics.MetricsDAO;
import org.rhq.server.metrics.domain.AggregateNumericMetric;
import org.rhq.server.metrics.domain.Bucket;
import org.rhq.server.metrics.domain.RawNumericMetric;
import org.rhq.server.metrics.invalid.InvalidMetric;

public class InvalidMetricsManager {
    private static final Log log = LogFactory.getLog(InvalidMetricsManager.class);
    private static final double THRESHOLD = 1.0E-5;
    private DateTimeService dateTimeService;
    private MetricsDAO dao;
    private InvalidMetric current;
    private MetricsConfiguration configuration;
    private DelayQueue<InvalidMetric> queue;
    private ScheduledExecutorService executor;
    private long delay = Long.parseLong(System.getProperty("rhq.metrics.invalid.queue-delay", "600000"));
    private boolean isShutdown;

    public InvalidMetricsManager(DateTimeService dateTimeService, MetricsDAO dao) {
        this(dateTimeService, dao, Integer.parseInt(System.getProperty("rhq.metrics.invalid.poller.initial-delay", "300")), Integer.parseInt(System.getProperty("rhq.metrics.invalid.poller.period", "300")));
    }

    InvalidMetricsManager(DateTimeService dateTimeService, MetricsDAO dao, int pollerDelay, int pollerPeriod) {
        this.dateTimeService = dateTimeService;
        this.dao = dao;
        this.configuration = new MetricsConfiguration();
        this.queue = new DelayQueue();
        this.executor = Executors.newSingleThreadScheduledExecutor();
        this.executor.scheduleAtFixedRate(new InvalidMetricRunnable(), pollerDelay, pollerPeriod, TimeUnit.SECONDS);
    }

    void setDelay(long delay) {
        this.delay = delay;
    }

    public void shutdown() {
        log.info((Object)"Shutting down...");
        this.isShutdown = true;
        this.queue.clear();
        this.executor.shutdown();
    }

    public boolean submit(AggregateNumericMetric metric) {
        DateTime day = this.dateTimeService.getTimeSlice(new DateTime(metric.getTimestamp()), this.configuration.getSixHourTimeSliceDuration());
        InvalidMetric invalidMetric = new InvalidMetric(day, metric, this.delay);
        if (this.isShutdown) {
            log.info((Object)(invalidMetric + " will not be submitted since we are already shutdown."));
            return false;
        }
        if (this.queue.contains(invalidMetric) || this.current != null && this.current.equals(invalidMetric)) {
            log.info((Object)(invalidMetric + " is already in the queue. It will not be resubmitted."));
            return false;
        }
        log.info((Object)("Adding " + invalidMetric + " to queue for processing"));
        this.queue.offer(invalidMetric);
        return true;
    }

    public DelayQueue<InvalidMetric> getQueue() {
        return this.queue;
    }

    int getRemainingInvalidMetrics() {
        return this.current == null ? this.queue.size() : this.queue.size() + 1;
    }

    private void handleInvalidMetric(InvalidMetric invalidMetric) {
        log.info((Object)("Attempting to fix " + invalidMetric + ". This may include updates to 1 hour, 6 hour, and 24 hour metrics."));
        if (invalidMetric.metric.getBucket() == Bucket.TWENTY_FOUR_HOUR) {
            this.update24HourMetric(invalidMetric);
        } else if (invalidMetric.metric.getBucket() == Bucket.SIX_HOUR) {
            if (DateTime.now().isAfter((ReadableInstant)invalidMetric.day.plusDays(1))) {
                this.update24HourMetric(invalidMetric);
            } else {
                this.update6HourMetrics(Arrays.asList(invalidMetric.metric));
            }
        } else {
            DateTime sixHourTimeSlice = this.dateTimeService.getTimeSlice(new DateTime(invalidMetric.metric.getTimestamp()), this.configuration.getOneHourTimeSliceDuration());
            if (DateTime.now().isAfter((ReadableInstant)invalidMetric.day.plusDays(1))) {
                this.update24HourMetric(invalidMetric);
            } else if (DateTime.now().isAfter((ReadableInstant)sixHourTimeSlice.plusHours(6))) {
                List<AggregateNumericMetric> sixHourMetrics = this.dao.findAggregateMetrics(invalidMetric.metric.getScheduleId(), Bucket.SIX_HOUR, invalidMetric.day.getMillis(), invalidMetric.day.plusDays(1).getMillis());
                this.update6HourMetrics(sixHourMetrics);
            } else {
                this.update1HourMetrics(Arrays.asList(invalidMetric.metric));
            }
        }
    }

    private void update24HourMetric(InvalidMetric invalidMetric) {
        List<AggregateNumericMetric> sixHourMetrics = this.dao.findAggregateMetrics(invalidMetric.metric.getScheduleId(), Bucket.SIX_HOUR, invalidMetric.day.getMillis(), invalidMetric.day.plusDays(1).getMillis());
        if (sixHourMetrics.isEmpty()) {
            log.info((Object)("Deleting " + invalidMetric + " since the 6 hour metrics are no longer available."));
            this.remove24HourMetric(invalidMetric.metric);
        } else {
            List<AggregateNumericMetric> updated6HourMetrics = this.update6HourMetrics(sixHourMetrics);
            AggregateNumericMetric recomputed24HourMetric = this.computeAggregate(updated6HourMetrics, invalidMetric.metric.getScheduleId(), invalidMetric.day.getMillis(), Bucket.TWENTY_FOUR_HOUR);
            this.persist24HourMetric(recomputed24HourMetric);
            log.info((Object)(invalidMetric + " has been recomputed with a new value of " + this.getValueText(recomputed24HourMetric)));
        }
    }

    private List<AggregateNumericMetric> update6HourMetrics(List<AggregateNumericMetric> sixHourMetrics) {
        List<AggregateNumericMetric> updated6HourMetrics = this.removeEmpty6HourMetrics(sixHourMetrics);
        List<AggregateNumericMetric> invalid6HourMetrics = this.findInvalidMetrics(updated6HourMetrics);
        for (AggregateNumericMetric invalid6HourMetric : invalid6HourMetrics) {
            List<AggregateNumericMetric> oneHourMetrics = this.dao.findAggregateMetrics(invalid6HourMetric.getScheduleId(), Bucket.ONE_HOUR, invalid6HourMetric.getTimestamp(), new DateTime(invalid6HourMetric.getTimestamp()).plusHours(6).getMillis());
            if (oneHourMetrics.isEmpty()) {
                log.info((Object)("Deleting 6 hour metric " + invalid6HourMetric + " since the 1 hour metrics are no longer available."));
                updated6HourMetrics = this.remove6HourMetric(invalid6HourMetric, sixHourMetrics);
                continue;
            }
            List<AggregateNumericMetric> updated1HourMetrics = this.update1HourMetrics(oneHourMetrics);
            AggregateNumericMetric recomputed6HourMetric = this.computeAggregate(updated1HourMetrics, invalid6HourMetric.getScheduleId(), invalid6HourMetric.getTimestamp(), Bucket.SIX_HOUR);
            updated6HourMetrics = this.replace6HourMetric(invalid6HourMetric, recomputed6HourMetric, sixHourMetrics);
            log.info((Object)("The invalid 6 hour metric " + invalid6HourMetric + " has been recomputed with a new value of " + this.getValueText(recomputed6HourMetric)));
        }
        return updated6HourMetrics;
    }

    private List<AggregateNumericMetric> update1HourMetrics(List<AggregateNumericMetric> oneHourMetrics) {
        List<AggregateNumericMetric> updated1HourMetrics = this.removeEmpty1HourMetrics(oneHourMetrics);
        List<AggregateNumericMetric> invalid1HourMetrics = this.findInvalidMetrics(updated1HourMetrics);
        for (AggregateNumericMetric invalid1HourMetric : invalid1HourMetrics) {
            AggregateNumericMetric recomputed1HourMetric = this.recompute1HourAggregateIfPossible(invalid1HourMetric);
            if (recomputed1HourMetric == null) {
                log.info((Object)("Deleting 1 hour metric " + invalid1HourMetric + " since the raw data is no longer available."));
                updated1HourMetrics = this.remove1HourMetric(invalid1HourMetric, updated1HourMetrics);
                continue;
            }
            updated1HourMetrics = this.replace1HourMetric(invalid1HourMetric, recomputed1HourMetric, updated1HourMetrics);
            log.info((Object)("The invalid 1 hour metric " + invalid1HourMetric + " has been recomputed with a new value of " + this.getValueText(recomputed1HourMetric)));
        }
        return updated1HourMetrics;
    }

    private List<AggregateNumericMetric> removeEmpty6HourMetrics(List<AggregateNumericMetric> metrics) {
        ArrayList<AggregateNumericMetric> nonEmptyMetrics = new ArrayList<AggregateNumericMetric>();
        for (AggregateNumericMetric metric : metrics) {
            if (this.isEmptyMetric(metric)) {
                this.dao.deleteAggregate(metric);
                continue;
            }
            nonEmptyMetrics.add(metric);
        }
        return nonEmptyMetrics;
    }

    private List<AggregateNumericMetric> removeEmpty1HourMetrics(List<AggregateNumericMetric> metrics) {
        ArrayList<AggregateNumericMetric> nonEmptyMetrics = new ArrayList<AggregateNumericMetric>();
        for (AggregateNumericMetric metric : metrics) {
            if (this.isEmptyMetric(metric)) {
                this.dao.deleteAggregate(metric);
                continue;
            }
            nonEmptyMetrics.add(metric);
        }
        return nonEmptyMetrics;
    }

    private boolean isEmptyMetric(AggregateNumericMetric metric) {
        return metric.getMin().equals(Double.NaN) && metric.getMax().equals(Double.NaN) && metric.getAvg().equals(0.0);
    }

    private List<AggregateNumericMetric> findInvalidMetrics(List<AggregateNumericMetric> metrics) {
        ArrayList<AggregateNumericMetric> invalidMetrics = new ArrayList<AggregateNumericMetric>();
        for (AggregateNumericMetric metric : metrics) {
            if (!this.isInvalidMetric(metric)) continue;
            invalidMetrics.add(metric);
        }
        return invalidMetrics;
    }

    public boolean isInvalidMetric(AggregateNumericMetric metric) {
        return metric.getMax() < metric.getAvg() && Math.abs(metric.getMax() - metric.getAvg()) > 1.0E-5 || metric.getMin() > metric.getAvg() && Math.abs(metric.getMin() - metric.getAvg()) > 1.0E-5 || Double.isNaN(metric.getAvg()) || Double.isNaN(metric.getMin()) || Double.isNaN(metric.getMin());
    }

    private AggregateNumericMetric recompute1HourAggregateIfPossible(AggregateNumericMetric metric) {
        List<RawNumericMetric> rawMetrics = this.dao.findRawMetrics(metric.getScheduleId(), metric.getTimestamp(), new DateTime(metric.getTimestamp()).plusHours(1).getMillis());
        if (!rawMetrics.isEmpty()) {
            return this.computeAggregateFromRaw(rawMetrics, metric);
        }
        return null;
    }

    private AggregateNumericMetric computeAggregateFromRaw(List<RawNumericMetric> rawMetrics, AggregateNumericMetric metric) {
        double min;
        double max = min = Double.NaN;
        int count = 0;
        ArithmeticMeanCalculator mean = new ArithmeticMeanCalculator();
        for (RawNumericMetric rawMetric : rawMetrics) {
            double value = rawMetric.getValue();
            if (count == 0) {
                max = min = value;
            }
            if (value < min) {
                min = value;
            } else if (value > max) {
                max = value;
            }
            mean.add(value);
            ++count;
        }
        return new AggregateNumericMetric(metric.getScheduleId(), metric.getBucket(), mean.getArithmeticMean(), min, max, metric.getTimestamp());
    }

    private AggregateNumericMetric computeAggregate(List<AggregateNumericMetric> metrics, int scheduleId, long timestamp, Bucket bucket) {
        double min;
        double max = min = Double.NaN;
        int count = 0;
        ArithmeticMeanCalculator mean = new ArithmeticMeanCalculator();
        for (AggregateNumericMetric metric : metrics) {
            if (count == 0) {
                min = metric.getMin();
                max = metric.getMax();
            }
            if (metric.getMin() < min) {
                min = metric.getMin();
            }
            if (metric.getMax() > max) {
                max = metric.getMax();
            }
            mean.add(metric.getAvg());
            ++count;
        }
        return new AggregateNumericMetric(scheduleId, bucket, mean.getArithmeticMean(), min, max, timestamp);
    }

    private List<AggregateNumericMetric> remove1HourMetric(AggregateNumericMetric metric, List<AggregateNumericMetric> metrics) {
        return this.removeMetric(metric, metrics);
    }

    private List<AggregateNumericMetric> remove6HourMetric(AggregateNumericMetric metric, List<AggregateNumericMetric> metrics) {
        return this.removeMetric(metric, metrics);
    }

    private void remove24HourMetric(AggregateNumericMetric metric) {
        this.dao.deleteAggregate(metric);
    }

    private List<AggregateNumericMetric> replace1HourMetric(AggregateNumericMetric metric, AggregateNumericMetric newMetric, List<AggregateNumericMetric> metrics) {
        return this.replaceMetric(metric, newMetric, metrics, Bucket.ONE_HOUR);
    }

    private List<AggregateNumericMetric> replace6HourMetric(AggregateNumericMetric metric, AggregateNumericMetric newMetric, List<AggregateNumericMetric> metrics) {
        return this.replaceMetric(metric, newMetric, metrics, Bucket.SIX_HOUR);
    }

    private List<AggregateNumericMetric> removeMetric(AggregateNumericMetric metric, List<AggregateNumericMetric> metrics) {
        this.dao.deleteAggregate(metric);
        ArrayList<AggregateNumericMetric> updatedMetrics = new ArrayList<AggregateNumericMetric>(metrics);
        updatedMetrics.remove(metric);
        return updatedMetrics;
    }

    private List<AggregateNumericMetric> replaceMetric(AggregateNumericMetric metric, AggregateNumericMetric newMetric, List<AggregateNumericMetric> metrics, Bucket bucket) {
        switch (bucket) {
            case ONE_HOUR: {
                this.persist1HourMetric(newMetric);
                break;
            }
            case SIX_HOUR: {
                this.persist6HourMetric(newMetric);
                break;
            }
            default: {
                throw new IllegalArgumentException((Object)((Object)bucket) + " cannot be used for this method");
            }
        }
        ArrayList<AggregateNumericMetric> updatedMetrics = new ArrayList<AggregateNumericMetric>(metrics);
        updatedMetrics.remove(metric);
        updatedMetrics.add(newMetric);
        return updatedMetrics;
    }

    private void persist1HourMetric(AggregateNumericMetric metric) {
        this.dao.insert1HourData(metric).get();
    }

    private void persist6HourMetric(AggregateNumericMetric metric) {
        this.dao.insert6HourData(metric).get();
    }

    private void persist24HourMetric(AggregateNumericMetric metric) {
        this.dao.insert24HourData(metric).get();
    }

    private String getValueText(AggregateNumericMetric metric) {
        return "{max: " + metric.getMax() + ", min: " + metric.getMin() + ", avg: " + metric.getAvg() + "}";
    }

    private class InvalidMetricRunnable
    implements Runnable {
        private InvalidMetricRunnable() {
        }

        @Override
        public void run() {
            ArrayList invalidMetrics = new ArrayList(InvalidMetricsManager.this.queue.size());
            InvalidMetricsManager.this.queue.drainTo(invalidMetrics);
            for (InvalidMetric invalidMetric : invalidMetrics) {
                InvalidMetricsManager.this.current = invalidMetric;
                try {
                    InvalidMetricsManager.this.handleInvalidMetric(InvalidMetricsManager.this.current);
                }
                catch (Exception e) {
                    log.warn((Object)("An unexpected occurred while processing invalid metric " + InvalidMetricsManager.this.current), (Throwable)e);
                }
            }
            InvalidMetricsManager.this.current = null;
        }
    }
}

