/*
 * Decompiled with CFR 0.152.
 */
package com.cloudhopper.commons.util;

import com.cloudhopper.commons.util.FileAlreadyExistsException;
import com.cloudhopper.commons.util.FileUtil;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompressionUtil {
    private static final Logger logger = LoggerFactory.getLogger(CompressionUtil.class);

    public static boolean isAlgorithmSupported(String algorithm) {
        return Algorithm.findByName(algorithm) != null;
    }

    public static boolean isFileExtensionSupported(String fileExt) {
        return Algorithm.findByFileExtension(fileExt) != null;
    }

    public static File compress(File sourceFile, String algorithm, boolean deleteSourceFileAfterCompressed) throws FileAlreadyExistsException, IOException {
        return CompressionUtil.compress(sourceFile, sourceFile.getParentFile(), algorithm, deleteSourceFileAfterCompressed);
    }

    public static File compress(File sourceFile, File targetDir, String algorithm, boolean deleteSourceFileAfterCompressed) throws FileAlreadyExistsException, IOException {
        if (!targetDir.isDirectory()) {
            throw new IOException("Cannot compress file since target directory " + targetDir + " neither exists or is a directory");
        }
        Algorithm a = Algorithm.findByName(algorithm);
        if (a == null) {
            throw new IOException("Compression algorithm '" + algorithm + "' is not supported");
        }
        File targetFile = new File(targetDir, sourceFile.getName() + "." + a.getFileExtension());
        CompressionUtil.compress(a, sourceFile, targetFile, deleteSourceFileAfterCompressed);
        return targetFile;
    }

    public static File uncompress(File sourceFile, boolean deleteSourceFileAfterUncompressed) throws FileAlreadyExistsException, IOException {
        return CompressionUtil.uncompress(sourceFile, sourceFile.getParentFile(), deleteSourceFileAfterUncompressed);
    }

    public static File uncompress(File sourceFile, File targetDir, boolean deleteSourceFileAfterUncompressed) throws FileAlreadyExistsException, IOException {
        String fileExt = FileUtil.parseFileExtension(sourceFile.getName());
        if (fileExt == null) {
            throw new IOException("File '" + sourceFile + "' must contain a file extension in order to lookup the compression algorithm");
        }
        Algorithm a = Algorithm.findByFileExtension(fileExt);
        if (a == null) {
            throw new IOException("Unrecognized or unsupported compression algorithm for file extension '" + fileExt + "'");
        }
        if (!targetDir.isDirectory()) {
            throw new IOException("Cannot uncompress file since target directory " + targetDir + " neither exists or is a directory");
        }
        String filename = sourceFile.getName();
        filename = filename.substring(0, filename.length() - (fileExt.length() + 1));
        File targetFile = new File(targetDir, filename);
        CompressionUtil.uncompress(a, sourceFile, targetFile, deleteSourceFileAfterUncompressed);
        return targetFile;
    }

    private static void compress(Algorithm a, File sourceFile, File targetFile, boolean deleteSourceFileAfterCompressed) throws FileAlreadyExistsException, IOException {
        if (!sourceFile.canRead()) {
            throw new IOException("Source file " + sourceFile + " neither exists or can be read");
        }
        if (targetFile.exists()) {
            throw new FileAlreadyExistsException("Target file " + targetFile + " already exists - cannot overwrite!");
        }
        a.getCompressor().compress(sourceFile, targetFile);
        if (deleteSourceFileAfterCompressed) {
            sourceFile.delete();
        }
    }

    private static void uncompress(Algorithm a, File sourceFile, File targetFile, boolean deleteSourceFileAfterUncompressed) throws FileAlreadyExistsException, IOException {
        if (!sourceFile.canRead()) {
            throw new IOException("Source file " + sourceFile + " neither exists or can be read");
        }
        if (targetFile.exists()) {
            throw new FileAlreadyExistsException("Target file " + targetFile + " already exists - cannot overwrite!");
        }
        a.getCompressor().uncompress(sourceFile, targetFile);
        if (deleteSourceFileAfterUncompressed) {
            sourceFile.delete();
        }
    }

    private static void uncompress(Algorithm a, InputStream srcIn, OutputStream destOut) throws IOException {
        a.getCompressor().uncompress(srcIn, destOut);
    }

    private static class ZipCompressor
    implements Compressor {
        private ZipCompressor() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void compress(File srcFile, File destFile) throws IOException {
            FileInputStream in = null;
            ZipOutputStream out = null;
            try {
                int len;
                in = new FileInputStream(srcFile);
                out = new ZipOutputStream(new FileOutputStream(destFile));
                out.putNextEntry(new ZipEntry(srcFile.getName()));
                byte[] buf = new byte[1024];
                while ((len = in.read(buf)) > 0) {
                    out.write(buf, 0, len);
                }
                in.close();
                in = null;
                out.closeEntry();
                out.finish();
                out.close();
                out = null;
            }
            finally {
                if (in != null) {
                    try {
                        in.close();
                    }
                    catch (Exception e) {}
                }
                if (out != null) {
                    logger.warn("Output stream for ZIP compressed file was not null -- indicates error with compression occurred");
                    try {
                        out.close();
                    }
                    catch (Exception e) {}
                }
            }
        }

        public void compress(InputStream srcIn, OutputStream destOut) throws IOException {
            throw new UnsupportedOperationException("Not supported yet.");
        }

        public void uncompress(File srcFile, File destFile) throws IOException {
            FileInputStream in = new FileInputStream(srcFile);
            FileOutputStream out = new FileOutputStream(destFile);
            this.uncompress(in, out);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void uncompress(InputStream srcIn, OutputStream destOut) throws IOException {
            ZipInputStream in = null;
            try {
                int len;
                in = new ZipInputStream(srcIn);
                ZipEntry ze = in.getNextEntry();
                byte[] buf = new byte[1024];
                while ((len = in.read(buf)) > 0) {
                    destOut.write(buf, 0, len);
                }
                if (in.getNextEntry() != null) {
                    throw new IOException("Zip file/inputstream contained more than one entry (this method cannot support)");
                }
                in.closeEntry();
                in.close();
                in = null;
                destOut.close();
                destOut = null;
            }
            finally {
                if (in != null) {
                    try {
                        in.close();
                    }
                    catch (Exception e) {}
                }
                if (destOut != null) {
                    try {
                        destOut.close();
                    }
                    catch (Exception e) {}
                }
            }
        }
    }

    private static class GzipCompressor
    implements Compressor {
        private GzipCompressor() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void compress(File srcFile, File destFile) throws IOException {
            FileInputStream in = null;
            DeflaterOutputStream out = null;
            try {
                int len;
                in = new FileInputStream(srcFile);
                out = new GZIPOutputStream(new FileOutputStream(destFile));
                byte[] buf = new byte[1024];
                while ((len = in.read(buf)) > 0) {
                    ((GZIPOutputStream)out).write(buf, 0, len);
                }
                in.close();
                in = null;
                ((GZIPOutputStream)out).finish();
                out.close();
                out = null;
            }
            finally {
                if (in != null) {
                    try {
                        in.close();
                    }
                    catch (Exception e) {}
                }
                if (out != null) {
                    logger.warn("Output stream for GZIP compressed file was not null -- indicates error with compression occurred");
                    try {
                        out.close();
                    }
                    catch (Exception e) {}
                }
            }
        }

        public void compress(InputStream srcIn, OutputStream destOut) throws IOException {
            throw new UnsupportedOperationException("Not supported yet.");
        }

        public void uncompress(File srcFile, File destFile) throws IOException {
            FileInputStream in = new FileInputStream(srcFile);
            FileOutputStream out = new FileOutputStream(destFile);
            this.uncompress(in, out);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void uncompress(InputStream srcIn, OutputStream destOut) throws IOException {
            GZIPInputStream in = null;
            try {
                int len;
                in = new GZIPInputStream(srcIn);
                byte[] buf = new byte[1024];
                while ((len = in.read(buf)) > 0) {
                    destOut.write(buf, 0, len);
                }
                in.close();
                in = null;
                destOut.close();
                destOut = null;
            }
            finally {
                if (in != null) {
                    try {
                        in.close();
                    }
                    catch (Exception e) {}
                }
                if (destOut != null) {
                    try {
                        destOut.close();
                    }
                    catch (Exception e) {}
                }
            }
        }
    }

    private static interface Compressor {
        public void compress(File var1, File var2) throws IOException;

        public void compress(InputStream var1, OutputStream var2) throws IOException;

        public void uncompress(File var1, File var2) throws IOException;

        public void uncompress(InputStream var1, OutputStream var2) throws IOException;
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    public static enum Algorithm {
        GZIP("gzip", "gz", new GzipCompressor()),
        ZIP("zip", "zip", new ZipCompressor());

        private final String name;
        private final String fileExt;
        private final Compressor compressor;

        private Algorithm(String name, String fileExt, Compressor compressor) {
            this.name = name;
            this.fileExt = fileExt;
            this.compressor = compressor;
        }

        public String getName() {
            return this.name;
        }

        public String getFileExtension() {
            return this.fileExt;
        }

        private Compressor getCompressor() {
            return this.compressor;
        }

        public static Algorithm findByName(String name) {
            for (Algorithm e : Algorithm.values()) {
                if (!e.name.equalsIgnoreCase(name)) continue;
                return e;
            }
            return null;
        }

        public static Algorithm findByFileExtension(String fileExt) {
            for (Algorithm e : Algorithm.values()) {
                if (!e.fileExt.equalsIgnoreCase(fileExt)) continue;
                return e;
            }
            return null;
        }
    }
}

