package org.refcodes.servicebus;

import org.refcodes.matcher.Matcher;
import org.refcodes.matcher.MatcherSugar;

/**
 * Declarative syntactic sugar which may be statically imported in order to
 * allow declarative definitions for the {@link ServiceMatcher} related
 * elements.
 */
public class ServiceMatcherSugar {

	/**
	 * Factory method to create an service matcher by service type.
	 * 
	 * @param <S> The type of the service to be matched
	 * 
	 * @param aServiceType The service type to be matched by this matcher.
	 * 
	 * @return An service matcher by service type.
	 */
	public static <S extends Service<?>> ServiceMatcher<S> isAssignableFrom( Class<?> aServiceType ) {
		Matcher<S> thrAssignableFrom = MatcherSugar.isAssignableFrom( aServiceType );
		return new ServiceMatcherWrapper<S>( thrAssignableFrom );
	}

	/**
	 * Factory method to create an "OR" matcher for the given matchers.
	 * 
	 * @param <S> The type of the service applied to the matcher
	 * 
	 * @param aServiceMatchers The matchers to be combined by an "OR".
	 * 
	 * @return An "OR" matcher.
	 */
	@SafeVarargs
	public static <S extends Service<?>> ServiceMatcher<S> or( ServiceMatcher<S>... aServiceMatchers ) {
		return new ServiceMatcherWrapper<S>( MatcherSugar.or( aServiceMatchers ) );
	}

	/**
	 * Factory method to create an "AND" matcher for the given matchers.
	 * 
	 * @param <S> The type of the service applied to the matcher
	 * 
	 * @param aServiceMatchers The matchers to be combined by an "AND".
	 * 
	 * @return An "AND" matcher.
	 */
	@SafeVarargs
	public static <S extends Service<?>> ServiceMatcher<S> and( ServiceMatcher<S>... aServiceMatchers ) {
		return new ServiceMatcherWrapper<S>( MatcherSugar.and( aServiceMatchers ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Wrapps an service matcher to by of the correct type.
	 *
	 * @param <S> the generic type
	 */
	private static class ServiceMatcherWrapper<S extends Service<?>> implements ServiceMatcher<S> {

		private Matcher<S> serviceMatcher;

		/**
		 * Wrapps the given service matcher to be of the correct type.
		 * 
		 * @param aMatcher the service matcher in question.
		 */
		public ServiceMatcherWrapper( Matcher<S> aMatcher ) {
			assert (aMatcher != null);
			this.serviceMatcher = aMatcher;
		}

		/**
		 * Checks if is matching.
		 *
		 * @param aService the service
		 * @return true, if is matching
		 */
		@Override
		public boolean isMatching( S aService ) {
			return serviceMatcher.isMatching( aService );
		}
	}
}