// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.servicebus;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

/**
 * The {@link ServiceDirectoryImpl} implements the {@link ServiceDirectory}.
 *
 * @param <S> the generic type
 * @param <SCTX> the generic type
 */
public class ServiceDirectoryImpl<S extends Service<?>, SCTX extends ServiceContext<S>> implements ServiceDirectory<S, SCTX> {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Set<ServiceDescriptor<S, SCTX>> _serviceDescriptors = new HashSet<ServiceDescriptor<S, SCTX>>();

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Set<ServiceDescriptor<S, SCTX>> getServiceDescriptors() {
		return Collections.unmodifiableSet( _serviceDescriptors );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasService( ServiceDescriptor<S, SCTX> aServiceDescriptor ) {
		return _serviceDescriptors.contains( aServiceDescriptor );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addService( ServiceDescriptor<S, SCTX> aServiceDescriptor ) throws DuplicateServiceRuntimeException {
		if ( hasService( aServiceDescriptor ) ) {
			throw new DuplicateServiceRuntimeException( aServiceDescriptor.getService(), "The given service descriptor is already known by service type and by service Meta-Data." );
		}
		_serviceDescriptors.add( aServiceDescriptor );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeService( ServiceDescriptor<S, SCTX> aServiceDescriptor ) throws UnknownServiceRuntimeException {
		boolean isRemoved = _serviceDescriptors.remove( aServiceDescriptor );
		if ( !isRemoved ) {
			throw new UnknownServiceRuntimeException( aServiceDescriptor.getService(), "The given service is not know by service type and by service Meta-Data." );
		}
	}
}