// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.servicebus;

import org.refcodes.exception.AbstractRuntimeException;

/**
 * Base exception for {@link ServiceBus} related runtime exceptions.
 */
public abstract class ServiceBusRuntimeException extends AbstractRuntimeException {

	private static final long serialVersionUID = 1L;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new service bus runtime exception.
	 *
	 * @param aMessage the message
	 * @param aErrorCode the error code
	 */
	public ServiceBusRuntimeException( String aMessage, String aErrorCode ) {
		super( aMessage, aErrorCode );
	}

	/**
	 * Instantiates a new service bus runtime exception.
	 *
	 * @param aMessage the message
	 * @param aCause the cause
	 * @param aErrorCode the error code
	 */
	public ServiceBusRuntimeException( String aMessage, Throwable aCause, String aErrorCode ) {
		super( aMessage, aCause, aErrorCode );
	}

	/**
	 * Instantiates a new service bus runtime exception.
	 *
	 * @param message the message
	 * @param cause the cause
	 */
	public ServiceBusRuntimeException( String message, Throwable cause ) {
		super( message, cause );
	}

	/**
	 * Instantiates a new service bus runtime exception.
	 *
	 * @param message the message
	 */
	public ServiceBusRuntimeException( String message ) {
		super( message );
	}

	/**
	 * Instantiates a new service bus runtime exception.
	 *
	 * @param aCause the cause
	 * @param aErrorCode the error code
	 */
	public ServiceBusRuntimeException( Throwable aCause, String aErrorCode ) {
		super( aCause, aErrorCode );
	}

	/**
	 * Instantiates a new service bus runtime exception.
	 *
	 * @param cause the cause
	 */
	public ServiceBusRuntimeException( Throwable cause ) {
		super( cause );
	}

	/**
	 * This exception is the exception for {@link ServiceDescriptor} related
	 * problems.
	 */
	@SuppressWarnings("rawtypes")
	static abstract class ServiceBusServiceDescriptorRuntimeException extends ServiceBusRuntimeException implements ServiceDescriptorAccessor {

		private static final long serialVersionUID = 1L;

		// /////////////////////////////////////////////////////////////////////////
		// VARIABLES:
		// /////////////////////////////////////////////////////////////////////////

		private ServiceDescriptor _serviceDescriptor;

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Instantiates a new service bus service descriptor runtime exception.
		 *
		 * @param aServiceDescriptor the service descriptor
		 * @param aMessage the message
		 * @param aErrorCode the error code
		 */
		public ServiceBusServiceDescriptorRuntimeException( ServiceDescriptor aServiceDescriptor, String aMessage, String aErrorCode ) {
			super( aMessage, aErrorCode );
			_serviceDescriptor = aServiceDescriptor;
		}

		/**
		 * Instantiates a new service bus service descriptor runtime exception.
		 *
		 * @param aServiceDescriptor the service descriptor
		 * @param aMessage the message
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public ServiceBusServiceDescriptorRuntimeException( ServiceDescriptor aServiceDescriptor, String aMessage, Throwable aCause, String aErrorCode ) {
			super( aMessage, aCause, aErrorCode );
			_serviceDescriptor = aServiceDescriptor;
		}

		/**
		 * Instantiates a new service bus service descriptor runtime exception.
		 *
		 * @param aServiceDescriptor the service descriptor
		 * @param message the message
		 * @param cause the cause
		 */
		public ServiceBusServiceDescriptorRuntimeException( ServiceDescriptor aServiceDescriptor, String message, Throwable cause ) {
			super( message, cause );
			_serviceDescriptor = aServiceDescriptor;
		}

		/**
		 * Instantiates a new service bus service descriptor runtime exception.
		 *
		 * @param aServiceDescriptor the service descriptor
		 * @param message the message
		 */
		public ServiceBusServiceDescriptorRuntimeException( ServiceDescriptor aServiceDescriptor, String message ) {
			super( message );
			_serviceDescriptor = aServiceDescriptor;
		}

		/**
		 * Instantiates a new service bus service descriptor runtime exception.
		 *
		 * @param aServiceDescriptor the service descriptor
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public ServiceBusServiceDescriptorRuntimeException( ServiceDescriptor aServiceDescriptor, Throwable aCause, String aErrorCode ) {
			super( aCause, aErrorCode );
			_serviceDescriptor = aServiceDescriptor;

		}

		/**
		 * Instantiates a new service bus service descriptor runtime exception.
		 *
		 * @param aServiceDescriptor the service descriptor
		 * @param cause the cause
		 */
		public ServiceBusServiceDescriptorRuntimeException( ServiceDescriptor aServiceDescriptor, Throwable cause ) {
			super( cause );
			_serviceDescriptor = aServiceDescriptor;
		}

		// /////////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Gets the service descriptor.
		 *
		 * @return the service descriptor
		 */
		@Override
		public ServiceDescriptor getServiceDescriptor() {
			return _serviceDescriptor;
		}
	}

	/**
	 * This exception is the exception for {@link ServiceMatcher} related
	 * problems.
	 */
	static abstract class ServiceBusServiceMatcherRuntimeException extends ServiceBusRuntimeException implements ServiceMatcherAccessor {

		private static final long serialVersionUID = 1L;

		// /////////////////////////////////////////////////////////////////////
		// VARIABLES:
		// /////////////////////////////////////////////////////////////////////

		private ServiceMatcher<?> _serviceMatcher;

		// /////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Instantiates a new service bus service matcher runtime exception.
		 *
		 * @param aServiceMatcher the service matcher
		 * @param aMessage the message
		 * @param aErrorCode the error code
		 */
		public ServiceBusServiceMatcherRuntimeException( ServiceMatcher<?> aServiceMatcher, String aMessage, String aErrorCode ) {
			super( aMessage, aErrorCode );
			_serviceMatcher = aServiceMatcher;
		}

		/**
		 * Instantiates a new service bus service matcher runtime exception.
		 *
		 * @param aServiceMatcher the service matcher
		 * @param aMessage the message
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public ServiceBusServiceMatcherRuntimeException( ServiceMatcher<?> aServiceMatcher, String aMessage, Throwable aCause, String aErrorCode ) {
			super( aMessage, aCause, aErrorCode );
			_serviceMatcher = aServiceMatcher;
		}

		/**
		 * Instantiates a new service bus service matcher runtime exception.
		 *
		 * @param aServiceMatcher the service matcher
		 * @param message the message
		 * @param cause the cause
		 */
		public ServiceBusServiceMatcherRuntimeException( ServiceMatcher<?> aServiceMatcher, String message, Throwable cause ) {
			super( message, cause );
			_serviceMatcher = aServiceMatcher;
		}

		/**
		 * Instantiates a new service bus service matcher runtime exception.
		 *
		 * @param aServiceMatcher the service matcher
		 * @param message the message
		 */
		public ServiceBusServiceMatcherRuntimeException( ServiceMatcher<?> aServiceMatcher, String message ) {
			super( message );
			_serviceMatcher = aServiceMatcher;
		}

		/**
		 * Instantiates a new service bus service matcher runtime exception.
		 *
		 * @param aServiceMatcher the service matcher
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public ServiceBusServiceMatcherRuntimeException( ServiceMatcher<?> aServiceMatcher, Throwable aCause, String aErrorCode ) {
			super( aCause, aErrorCode );
			_serviceMatcher = aServiceMatcher;

		}

		/**
		 * Instantiates a new service bus service matcher runtime exception.
		 *
		 * @param aServiceMatcher the service matcher
		 * @param cause the cause
		 */
		public ServiceBusServiceMatcherRuntimeException( ServiceMatcher<?> aServiceMatcher, Throwable cause ) {
			super( cause );
			_serviceMatcher = aServiceMatcher;
		}

		// /////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Gets the service matcher.
		 *
		 * @return the service matcher
		 */
		@Override
		public ServiceMatcher<?> getServiceMatcher() {
			return _serviceMatcher;
		}
	}

	/**
	 * This exception is the exception for {@link Service} related problems.
	 */
	@SuppressWarnings("rawtypes")
	static abstract class ServiceBusServiceRuntimeException extends ServiceBusRuntimeException implements ServiceAccessor {

		private static final long serialVersionUID = 1L;

		// /////////////////////////////////////////////////////////////////////
		// VARIABLES:
		// /////////////////////////////////////////////////////////////////////

		private Service _service;

		// /////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Instantiates a new service bus service runtime exception.
		 *
		 * @param aService the service
		 * @param aMessage the message
		 * @param aErrorCode the error code
		 */
		public ServiceBusServiceRuntimeException( Service aService, String aMessage, String aErrorCode ) {
			super( aMessage, aErrorCode );
			_service = aService;
		}

		/**
		 * Instantiates a new service bus service runtime exception.
		 *
		 * @param aService the service
		 * @param aMessage the message
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public ServiceBusServiceRuntimeException( Service aService, String aMessage, Throwable aCause, String aErrorCode ) {
			super( aMessage, aCause, aErrorCode );
			_service = aService;
		}

		/**
		 * Instantiates a new service bus service runtime exception.
		 *
		 * @param aService the service
		 * @param message the message
		 * @param cause the cause
		 */
		public ServiceBusServiceRuntimeException( Service aService, String message, Throwable cause ) {
			super( message, cause );
			_service = aService;
		}

		/**
		 * Instantiates a new service bus service runtime exception.
		 *
		 * @param aService the service
		 * @param message the message
		 */
		public ServiceBusServiceRuntimeException( Service aService, String message ) {
			super( message );
			_service = aService;
		}

		/**
		 * Instantiates a new service bus service runtime exception.
		 *
		 * @param aService the service
		 * @param aCause the cause
		 * @param aErrorCode the error code
		 */
		public ServiceBusServiceRuntimeException( Service aService, Throwable aCause, String aErrorCode ) {
			super( aCause, aErrorCode );
			_service = aService;

		}

		/**
		 * Instantiates a new service bus service runtime exception.
		 *
		 * @param aService the service
		 * @param cause the cause
		 */
		public ServiceBusServiceRuntimeException( Service aService, Throwable cause ) {
			super( cause );
			_service = aService;
		}

		// /////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Gets the service.
		 *
		 * @return the service
		 */
		@Override
		public Service getService() {
			return _service;
		}
	}
}
