/*
 * Decompiled with CFR 0.152.
 */
package org.rajawali3d.math;

import android.support.annotation.FloatRange;
import android.support.annotation.IntRange;
import android.support.annotation.NonNull;
import android.support.annotation.Size;
import org.rajawali3d.WorldParameters;
import org.rajawali3d.math.MathUtil;
import org.rajawali3d.math.Matrix4;
import org.rajawali3d.math.vector.Vector3;

public final class Quaternion
implements Cloneable {
    public static final double NORMALIZATION_TOLERANCE = 1.0E-6;
    public static final double PARALLEL_TOLERANCE = 1.0E-6;
    public double w;
    public double x;
    public double y;
    public double z;
    @NonNull
    private Vector3 mTmpVec1 = new Vector3();
    @NonNull
    private Vector3 mTmpVec2 = new Vector3();
    @NonNull
    private Vector3 mTmpVec3 = new Vector3();
    @NonNull
    private static final Quaternion sTmp1 = new Quaternion(0.0, 0.0, 0.0, 0.0);
    @NonNull
    private static final Quaternion sTmp2 = new Quaternion(0.0, 0.0, 0.0, 0.0);

    public Quaternion() {
        this.identity();
    }

    public Quaternion(double w, double x, double y, double z) {
        this.setAll(w, x, y, z);
    }

    public Quaternion(@NonNull Quaternion quat) {
        this.setAll(quat);
    }

    public Quaternion(@NonNull Vector3 axis, double angle) {
        this.fromAngleAxis(axis, angle);
    }

    @NonNull
    public Quaternion setAll(double w, double x, double y, double z) {
        this.w = w;
        this.x = x;
        this.y = y;
        this.z = z;
        return this;
    }

    @NonNull
    public Quaternion setAll(Quaternion quat) {
        return this.setAll(quat.w, quat.x, quat.y, quat.z);
    }

    @NonNull
    public Quaternion fromAngleAxis(@NonNull Vector3.Axis axis, double angle) {
        this.fromAngleAxis(Vector3.getAxisVector(axis), angle);
        return this;
    }

    @NonNull
    public Quaternion fromAngleAxis(@NonNull Vector3 axis, double angle) {
        if (axis.isZero()) {
            return this.identity();
        }
        this.mTmpVec1.setAll(axis);
        if (!this.mTmpVec1.isUnit()) {
            this.mTmpVec1.normalize();
        }
        double radian = MathUtil.degreesToRadians(angle);
        double halfAngle = radian * 0.5;
        double halfAngleSin = Math.sin(halfAngle);
        this.w = Math.cos(halfAngle);
        this.x = halfAngleSin * this.mTmpVec1.x;
        this.y = halfAngleSin * this.mTmpVec1.y;
        this.z = halfAngleSin * this.mTmpVec1.z;
        return this;
    }

    @NonNull
    public Quaternion fromAngleAxis(double x, double y, double z, double angle) {
        return this.fromAngleAxis(new Vector3(x, y, z), angle);
    }

    @NonNull
    public Quaternion fromAxes(@NonNull Vector3 xAxis, @NonNull Vector3 yAxis, @NonNull Vector3 zAxis) {
        return this.fromAxes(xAxis.x, xAxis.y, xAxis.z, yAxis.x, yAxis.y, yAxis.z, zAxis.x, zAxis.y, zAxis.z);
    }

    @NonNull
    public Quaternion fromAxes(double xx, double xy, double xz, double yx, double yy, double yz, double zx, double zy, double zz) {
        double z;
        double y;
        double x;
        double w;
        double m00 = xx;
        double m01 = xy;
        double m02 = xz;
        double m10 = yx;
        double m11 = yy;
        double m12 = yz;
        double m20 = zx;
        double m21 = zy;
        double m22 = zz;
        double t = m00 + m11 + m22;
        if (t >= 0.0) {
            double s = Math.sqrt(t + 1.0);
            w = 0.5 * s;
            s = 0.5 / s;
            x = (m21 - m12) * s;
            y = (m02 - m20) * s;
            z = (m10 - m01) * s;
        } else if (m00 > m11 && m00 > m22) {
            double s = Math.sqrt(1.0 + m00 - m11 - m22);
            x = s * 0.5;
            s = 0.5 / s;
            y = (m10 + m01) * s;
            z = (m02 + m20) * s;
            w = (m21 - m12) * s;
        } else if (m11 > m22) {
            double s = Math.sqrt(1.0 + m11 - m00 - m22);
            y = s * 0.5;
            s = 0.5 / s;
            x = (m10 + m01) * s;
            z = (m21 + m12) * s;
            w = (m02 - m20) * s;
        } else {
            double s = Math.sqrt(1.0 + m22 - m00 - m11);
            z = s * 0.5;
            s = 0.5 / s;
            x = (m02 + m20) * s;
            y = (m21 + m12) * s;
            w = (m10 - m01) * s;
        }
        return this.setAll(w, x, y, z);
    }

    @NonNull
    public Quaternion fromMatrix(@NonNull Matrix4 matrix) {
        double[] value = new double[16];
        matrix.toArray(value);
        this.fromAxes(value[0], value[1], value[2], value[4], value[5], value[6], value[8], value[9], value[10]);
        return this;
    }

    @NonNull
    public Quaternion fromMatrix(@NonNull @Size(min=16L) double[] matrix) {
        this.fromAxes(matrix[0], matrix[1], matrix[2], matrix[4], matrix[5], matrix[6], matrix[8], matrix[9], matrix[10]);
        return this;
    }

    @NonNull
    public Quaternion fromEuler(double yaw, double pitch, double roll) {
        yaw = Math.toRadians(yaw);
        pitch = Math.toRadians(pitch);
        roll = Math.toRadians(roll);
        double hr = roll * 0.5;
        double shr = Math.sin(hr);
        double chr = Math.cos(hr);
        double hp = pitch * 0.5;
        double shp = Math.sin(hp);
        double chp = Math.cos(hp);
        double hy = yaw * 0.5;
        double shy = Math.sin(hy);
        double chy = Math.cos(hy);
        double chy_shp = chy * shp;
        double shy_chp = shy * chp;
        double chy_chp = chy * chp;
        double shy_shp = shy * shp;
        this.x = chy_shp * chr + shy_chp * shr;
        this.y = shy_chp * chr - chy_shp * shr;
        this.z = chy_chp * shr - shy_shp * chr;
        this.w = chy_chp * chr + shy_shp * shr;
        return this;
    }

    @NonNull
    public Quaternion fromRotationBetween(@NonNull Vector3 u, @NonNull Vector3 v) {
        double dot = u.dot(v);
        double dotError = 1.0 - Math.abs(MathUtil.clamp(dot, -1.0, 1.0));
        if (dotError <= 1.0E-6) {
            if (dot < 0.0) {
                this.mTmpVec3.crossAndSet(WorldParameters.RIGHT_AXIS, u);
                if (this.mTmpVec3.length() < 1.0E-6) {
                    this.mTmpVec3.crossAndSet(WorldParameters.UP_AXIS, u);
                }
                this.mTmpVec3.normalize();
                return this.fromAngleAxis(this.mTmpVec3, 180.0);
            }
            return this.identity();
        }
        this.mTmpVec3.crossAndSet(u, v).normalize();
        this.x = this.mTmpVec3.x;
        this.y = this.mTmpVec3.y;
        this.z = this.mTmpVec3.z;
        this.w = 1.0 + dot;
        this.normalize();
        return this;
    }

    @NonNull
    public Quaternion fromRotationBetween(double x1, double y1, double z1, double x2, double y2, double z2) {
        this.mTmpVec1.setAll(x1, y1, z1).normalize();
        this.mTmpVec2.setAll(x2, y2, z2).normalize();
        return this.fromRotationBetween(this.mTmpVec1, this.mTmpVec2);
    }

    @NonNull
    public static Quaternion createFromRotationBetween(@NonNull Vector3 u, @NonNull Vector3 v) {
        Quaternion q = new Quaternion();
        q.fromRotationBetween(u, v);
        return q;
    }

    @NonNull
    public Quaternion add(@NonNull Quaternion quat) {
        this.w += quat.w;
        this.x += quat.x;
        this.y += quat.y;
        this.z += quat.z;
        return this;
    }

    @NonNull
    public Quaternion subtract(@NonNull Quaternion quat) {
        this.w -= quat.w;
        this.x -= quat.x;
        this.y -= quat.y;
        this.z -= quat.z;
        return this;
    }

    @NonNull
    public Quaternion multiply(double scalar) {
        this.w *= scalar;
        this.x *= scalar;
        this.y *= scalar;
        this.z *= scalar;
        return this;
    }

    @NonNull
    public Quaternion multiply(@NonNull Quaternion quat) {
        double tW = this.w;
        double tX = this.x;
        double tY = this.y;
        double tZ = this.z;
        this.w = tW * quat.w - tX * quat.x - tY * quat.y - tZ * quat.z;
        this.x = tW * quat.x + tX * quat.w + tY * quat.z - tZ * quat.y;
        this.y = tW * quat.y + tY * quat.w + tZ * quat.x - tX * quat.z;
        this.z = tW * quat.z + tZ * quat.w + tX * quat.y - tY * quat.x;
        return this;
    }

    @NonNull
    public Vector3 multiply(@NonNull Vector3 vector) {
        this.mTmpVec3.setAll(this.x, this.y, this.z);
        this.mTmpVec1.crossAndSet(this.mTmpVec3, vector);
        this.mTmpVec2.crossAndSet(this.mTmpVec3, this.mTmpVec1);
        this.mTmpVec1.multiply(2.0 * this.w);
        this.mTmpVec2.multiply(2.0);
        this.mTmpVec1.add(this.mTmpVec2);
        this.mTmpVec1.add(vector);
        return this.mTmpVec1;
    }

    @NonNull
    public Quaternion multiplyLeft(@NonNull Quaternion quat) {
        double newW = quat.w * this.w - quat.x * this.x - quat.y * this.y - quat.z * this.z;
        double newX = quat.w * this.x + quat.x * this.w + quat.y * this.z - quat.z * this.y;
        double newY = quat.w * this.y + quat.y * this.w + quat.z * this.x - quat.x * this.z;
        double newZ = quat.w * this.z + quat.z * this.w + quat.x * this.y - quat.y * this.x;
        return this.setAll(newW, newX, newY, newZ);
    }

    public double normalize() {
        double len = this.length2();
        if (len != 0.0 && Math.abs(len - 1.0) > 1.0E-6) {
            double factor = 1.0 / Math.sqrt(len);
            this.multiply(factor);
        }
        return len;
    }

    @NonNull
    public Quaternion conjugate() {
        this.x = -this.x;
        this.y = -this.y;
        this.z = -this.z;
        return this;
    }

    @NonNull
    public Quaternion inverse() {
        double norm = this.length2();
        double invNorm = 1.0 / norm;
        return this.setAll(this.w * invNorm, -this.x * invNorm, -this.y * invNorm, -this.z * invNorm);
    }

    @NonNull
    public Quaternion invertAndCreate() {
        double norm = this.length2();
        double invNorm = 1.0 / norm;
        return new Quaternion(this.w * invNorm, -this.x * invNorm, -this.y * invNorm, -this.z * invNorm);
    }

    @NonNull
    public Quaternion computeW() {
        double t = 1.0 - this.x * this.x - this.y * this.y - this.z * this.z;
        this.w = t < 0.0 ? 0.0 : -Math.sqrt(t);
        return this;
    }

    @NonNull
    public Vector3 getXAxis() {
        double fTy = 2.0 * this.y;
        double fTz = 2.0 * this.z;
        double fTwy = fTy * this.w;
        double fTwz = fTz * this.w;
        double fTxy = fTy * this.x;
        double fTxz = fTz * this.x;
        double fTyy = fTy * this.y;
        double fTzz = fTz * this.z;
        return new Vector3(1.0 - (fTyy + fTzz), fTxy + fTwz, fTxz - fTwy);
    }

    @NonNull
    public Vector3 getYAxis() {
        double fTx = 2.0 * this.x;
        double fTy = 2.0 * this.y;
        double fTz = 2.0 * this.z;
        double fTwx = fTx * this.w;
        double fTwz = fTz * this.w;
        double fTxx = fTx * this.x;
        double fTxy = fTy * this.x;
        double fTyz = fTz * this.y;
        double fTzz = fTz * this.z;
        return new Vector3(fTxy - fTwz, 1.0 - (fTxx + fTzz), fTyz + fTwx);
    }

    @NonNull
    public Vector3 getZAxis() {
        double fTx = 2.0 * this.x;
        double fTy = 2.0 * this.y;
        double fTz = 2.0 * this.z;
        double fTwx = fTx * this.w;
        double fTwy = fTy * this.w;
        double fTxx = fTx * this.x;
        double fTxz = fTz * this.x;
        double fTyy = fTy * this.y;
        double fTyz = fTz * this.y;
        return new Vector3(fTxz + fTwy, fTyz - fTwx, 1.0 - (fTxx + fTyy));
    }

    @NonNull
    public Vector3 getAxis(@NonNull Vector3.Axis axis) {
        if (axis == Vector3.Axis.X) {
            return this.getXAxis();
        }
        if (axis == Vector3.Axis.Y) {
            return this.getYAxis();
        }
        return this.getZAxis();
    }

    public double length() {
        return Math.sqrt(this.length2());
    }

    public double length2() {
        return this.w * this.w + this.x * this.x + this.y * this.y + this.z * this.z;
    }

    public double dot(@NonNull Quaternion other) {
        return this.w * other.w + this.x * other.x + this.y * other.y + this.z * other.z;
    }

    @NonNull
    public Quaternion identity() {
        this.w = 1.0;
        this.x = 0.0;
        this.y = 0.0;
        this.z = 0.0;
        return this;
    }

    @NonNull
    public static Quaternion getIdentity() {
        return new Quaternion(1.0, 0.0, 0.0, 0.0);
    }

    @NonNull
    public Quaternion exp() {
        double vNorm = Math.sqrt(this.x * this.x + this.y * this.y + this.z * this.z);
        double sin = Math.sin(vNorm);
        this.w = Math.cos(vNorm);
        double coeff = sin / vNorm;
        this.x = coeff * this.x;
        this.y = coeff * this.y;
        this.z = coeff * this.z;
        return this;
    }

    @NonNull
    public Quaternion expAndCreate() {
        Quaternion result = new Quaternion(this);
        result.exp();
        return result;
    }

    @NonNull
    public Quaternion log() {
        double qNorm = this.length();
        if (qNorm > 0.0) {
            double vNorm = Math.sqrt(this.x * this.x + this.y * this.y + this.z * this.z);
            double coeff = Math.acos(this.w / qNorm) / vNorm;
            this.w = Math.log(qNorm);
            this.x = coeff * this.x;
            this.y = coeff * this.y;
            this.z = coeff * this.z;
        }
        return this;
    }

    @NonNull
    public Quaternion logAndCreate() {
        Quaternion result = new Quaternion(this);
        result.log();
        return result;
    }

    @NonNull
    public Quaternion pow(double p) {
        return this.log().multiply(p).exp();
    }

    @NonNull
    public Quaternion pow(@NonNull Quaternion p) {
        return this.log().multiply(p).exp();
    }

    @NonNull
    public Quaternion powAndCreate(double p) {
        return new Quaternion(this).pow(p);
    }

    @NonNull
    public Quaternion powAndCreate(@NonNull Quaternion p) {
        return new Quaternion(this).pow(p);
    }

    @NonNull
    public Quaternion slerp(@NonNull Quaternion end, @FloatRange(from=0.0, to=1.0) double t) {
        return this.slerp(this, end, t, true);
    }

    @NonNull
    public Quaternion slerp(@NonNull Quaternion q1, @NonNull Quaternion q2, @FloatRange(from=0.0, to=1.0) double t) {
        return this.slerp(q1, q2, t, true);
    }

    @NonNull
    public Quaternion slerp(@NonNull Quaternion start, @NonNull Quaternion end, @FloatRange(from=0.0, to=1.0) double t, boolean shortestPath) {
        if (this.equals(end)) {
            return this;
        }
        double result = start.dot(end);
        if (shortestPath && result < 0.0) {
            end.x = -end.x;
            end.y = -end.y;
            end.z = -end.z;
            end.w = -end.w;
            result = -result;
        }
        double scale0 = 1.0 - t;
        double scale1 = t;
        if (!shortestPath || 1.0 - result > 0.1) {
            double theta = Math.acos(result);
            double invSinTheta = 1.0 / Math.sin(theta);
            scale0 = Math.sin((1.0 - t) * theta) * invSinTheta;
            scale1 = Math.sin(t * theta) * invSinTheta;
        }
        this.x = scale0 * start.x + scale1 * end.x;
        this.y = scale0 * start.y + scale1 * end.y;
        this.z = scale0 * start.z + scale1 * end.z;
        this.w = scale0 * start.w + scale1 * end.w;
        this.normalize();
        return this;
    }

    @NonNull
    public static Quaternion lerp(@NonNull Quaternion start, @NonNull Quaternion end, @FloatRange(from=0.0, to=1.0) double t, boolean shortestPath) {
        if (start.equals(end)) {
            return sTmp1.setAll(end);
        }
        sTmp1.setAll(start);
        sTmp2.setAll(end);
        double fCos = sTmp1.dot(sTmp2);
        if (fCos < 0.0 && shortestPath) {
            sTmp2.inverse();
            sTmp2.subtract(sTmp1);
            sTmp2.multiply(t);
            sTmp1.add(sTmp2);
        } else {
            sTmp2.subtract(sTmp1);
            sTmp2.multiply(t);
            sTmp1.add(sTmp2);
        }
        return sTmp1;
    }

    @NonNull
    public static Quaternion nlerp(@NonNull Quaternion q1, @NonNull Quaternion q2, @FloatRange(from=0.0, to=1.0) double t, boolean shortestPath) {
        Quaternion result = Quaternion.lerp(q1, q2, t, shortestPath);
        result.normalize();
        return result;
    }

    @IntRange(from=-1L, to=1L)
    public int getGimbalPole() {
        double t = this.y * this.x + this.z * this.w;
        return t > 0.499 ? 1 : (t < -0.499 ? -1 : 0);
    }

    public double getRotationX() {
        int pole = this.getGimbalPole();
        return pole == 0 ? Math.asin(MathUtil.clamp(2.0 * (this.w * this.x - this.z * this.y), -1.0, 1.0)) : (double)pole * Math.PI * 0.5;
    }

    public double getRotationY() {
        return this.getGimbalPole() == 0 ? Math.atan2(2.0 * (this.y * this.w + this.x * this.z), 1.0 - 2.0 * (this.y * this.y + this.x * this.x)) : 0.0;
    }

    public double getRotationZ() {
        int pole = this.getGimbalPole();
        return pole == 0 ? Math.atan2(2.0 * (this.w * this.z + this.y * this.x), 1.0 - 2.0 * (this.x * this.x + this.z * this.z)) : (double)pole * 2.0 * Math.atan2(this.y, this.w);
    }

    @NonNull
    public Matrix4 toRotationMatrix() {
        Matrix4 matrix = new Matrix4();
        this.toRotationMatrix(matrix);
        return matrix;
    }

    @NonNull
    public Matrix4 toRotationMatrix(@NonNull Matrix4 matrix) {
        this.toRotationMatrix(matrix.getDoubleValues());
        return matrix;
    }

    public void toRotationMatrix(@NonNull @Size(min=16L) double[] matrix) {
        double x2 = this.x * this.x;
        double y2 = this.y * this.y;
        double z2 = this.z * this.z;
        double xy = this.x * this.y;
        double xz = this.x * this.z;
        double yz = this.y * this.z;
        double wx = this.w * this.x;
        double wy = this.w * this.y;
        double wz = this.w * this.z;
        matrix[0] = 1.0 - 2.0 * (y2 + z2);
        matrix[1] = 2.0 * (xy - wz);
        matrix[2] = 2.0 * (xz + wy);
        matrix[3] = 0.0;
        matrix[4] = 2.0 * (xy + wz);
        matrix[5] = 1.0 - 2.0 * (x2 + z2);
        matrix[6] = 2.0 * (yz - wx);
        matrix[7] = 0.0;
        matrix[8] = 2.0 * (xz - wy);
        matrix[9] = 2.0 * (yz + wx);
        matrix[10] = 1.0 - 2.0 * (x2 + y2);
        matrix[11] = 0.0;
        matrix[12] = 0.0;
        matrix[13] = 0.0;
        matrix[14] = 0.0;
        matrix[15] = 1.0;
    }

    @NonNull
    public Quaternion lookAt(@NonNull Vector3 lookAt, @NonNull Vector3 upDirection) {
        this.mTmpVec1.setAll(lookAt);
        this.mTmpVec2.setAll(upDirection);
        double dotProduct = Vector3.dot(lookAt, upDirection);
        double dotError = Math.abs(Math.abs(dotProduct) - lookAt.length() * upDirection.length());
        if (dotError <= 1.0E-6) {
            this.mTmpVec2.normalize();
            if (dotProduct < 0.0) {
                this.mTmpVec1.inverse();
            }
            this.fromRotationBetween(WorldParameters.FORWARD_AXIS, this.mTmpVec1);
            return this;
        }
        Vector3.orthoNormalize(this.mTmpVec1, this.mTmpVec2);
        this.mTmpVec3.crossAndSet(this.mTmpVec2, this.mTmpVec1);
        this.fromAxes(this.mTmpVec3, this.mTmpVec2, this.mTmpVec1);
        return this;
    }

    @NonNull
    public static Quaternion lookAtAndCreate(@NonNull Vector3 lookAt, @NonNull Vector3 upDirection) {
        Quaternion ret = new Quaternion();
        return ret.lookAt(lookAt, upDirection);
    }

    public double angleBetween(@NonNull Quaternion other) {
        Quaternion inv = this.clone().inverse();
        Quaternion res = inv.multiplyLeft(other);
        return 2.0 * Math.acos(res.w);
    }

    @NonNull
    public Quaternion clone() {
        return new Quaternion(this.w, this.x, this.y, this.z);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Quaternion)) {
            return false;
        }
        Quaternion comp = (Quaternion)o;
        return this.x == comp.x && this.y == comp.y && this.z == comp.z && this.w == comp.w;
    }

    public boolean equals(@NonNull Quaternion other, double tolerance) {
        double fCos = this.dot(other);
        if (fCos > 1.0 && fCos - 1.0 < tolerance) {
            return true;
        }
        double angle = Math.acos(fCos);
        return Math.abs(angle) <= tolerance || MathUtil.realEqual(angle, Math.PI, tolerance);
    }

    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append("Quaternion <w, x, y, z>: <").append(this.w).append(", ").append(this.x).append(", ").append(this.y).append(", ").append(this.z).append(">");
        return sb.toString();
    }
}

