/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.objectweb.util.monolog.wrapper.javaLog;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.logging.LogManager;

import org.objectweb.util.monolog.Monolog;
import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.LogInfo;
import org.objectweb.util.monolog.file.api.Pattern;
import org.objectweb.util.monolog.wrapper.common.AbstractFactory;

/**
 * is the LoggerFactory for the wrapper to java.util.logging
 *
 * @author S.Chassande-Barrioz
 * @author Mohammed Boukada (Log extension)
 */
public class LoggerFactory extends AbstractFactory {

    /**
     * the LogManager of java.util.logging
     */
    protected static LogManager manager = null;

    /**
     * the root logger
     */
    protected static Logger rootLogger = null;


    /**
     * Keep a reference to loggers.
     * @see java.util.logging.LogManager addLogger javadoc
     */
    private ArrayList loggers = new ArrayList();

    /**
     * Contains extensions
     */
    private Map<Character, LogInfo> logExtensions = new HashMap<Character, LogInfo>();

    /**
     * This static code initialize the value of the variable defined into
     * BasicLevel.
     */
    static {
        debug("Initializing " + LoggerFactory.class.getName());
        debug("set default level values");
        BasicLevel.INHERIT = -1;
        debug("\t-INHERIT= " + BasicLevel.INHERIT);
        BasicLevel.DEBUG = java.util.logging.Level.FINEST.intValue();
        debug("\t-DEBUG= " + BasicLevel.DEBUG);
        BasicLevel.INFO = java.util.logging.Level.INFO.intValue();
        debug("\t-INFO= " + BasicLevel.INFO);
        BasicLevel.WARN = java.util.logging.Level.WARNING.intValue();
        debug("\t-WARN= " + BasicLevel.WARN);
        BasicLevel.ERROR = java.util.logging.Level.SEVERE.intValue();
        debug("\t-ERROR= " + BasicLevel.ERROR);
        BasicLevel.FATAL = JDKLevelImpl.FATAL.intValue();
        debug("\t-FATAL= " + BasicLevel.FATAL);

        BasicLevel.LEVEL_INHERIT = new LevelImpl("INHERIT", BasicLevel.INHERIT);
        BasicLevel.LEVEL_DEBUG = new LevelImpl("DEBUG", BasicLevel.DEBUG);
        BasicLevel.LEVEL_INFO = new LevelImpl("INFO", BasicLevel.INFO);
        BasicLevel.LEVEL_WARN = new LevelImpl("WARN", BasicLevel.WARN);
        BasicLevel.LEVEL_ERROR = new LevelImpl("ERROR", BasicLevel.ERROR);
        BasicLevel.LEVEL_FATAL = new LevelImpl("FATAL", BasicLevel.FATAL);

        manager = LogManager.getLogManager();
        if (classLoaderIsoltion) {
            debug("class loader isolation activated");
            int i = 0;
            while (rootLogger == null) {
                rootLoggerName = "root" + i;
                synchronized (manager) {
                    // the manager object is shared between the multiple instances of this class
                    if (manager.getLogger(rootLoggerName) == null) {
                        rootLogger = new Logger(rootLoggerName, null);
                        manager.addLogger(rootLogger);
                    } else {
                        i ++;
                    }
                }
            }
            rootLogger.setUseParentHandlers(false);
            Monolog.debug("Instanciate the root logger " + rootLoggerName);
            rootLoggerPrefix = rootLoggerName + '.';
        } else {
            rootLogger = new Logger(manager.getLogger(""));
        }
    }

    /**
     * This constant is used to initialize the factory with the configure method
     */
    public final static String JAVALOG_CONFIGURATION = "javaLogConfiguration";

    /**
     * This constant means that this java log system must be initialize with
     * the default configuration
     */
    public final static String DEFAULT = "default";

    /**
     * This constant means that this java log system must be initialize with
     * a property file
     */
    public final static String PROPERTY = "property";

    /**
     * This constant means that this java log system must be initialize with
     * a xml file
     */
    public final static String CLASS = "class";

    /**
     * This constant is the properties file name with wich the java log system
     * must be initialized.
     */
    public final static String JAVALOG_CONFIGURATION_FILE
            = "javaLogConfigurationFile";

    /**
     * This constant is the properties class name with wich the java log system
     * must be initialized.
     */
    public final static String JAVALOG_CONFIGURATION_CLASS
            = "javaLogConfigurationClass";

    public LoggerFactory() {
        super();
    }

    public String getWrapperName() {
        return "javaLog";
    }

    protected String[][] getDefaultHandlerType2className() {
        if (handlerTypes != null) {
            return new String[][] {
                    {handlerTypes[0], "org.objectweb.util.monolog.wrapper.javaLog.GenericHandler"},
                    {handlerTypes[1], "org.objectweb.util.monolog.wrapper.javaLog.GenericHandler"},
                    {handlerTypes[2], "org.objectweb.util.monolog.wrapper.javaLog.GenericHandler"},
                    {handlerTypes[3], "org.objectweb.util.monolog.wrapper.javaLog.GenericHandler"},
                    {handlerTypes[4], "org.objectweb.util.monolog.wrapper.javaLog.JMXGenericHandler"},
                    {handlerTypes[5], "org.objectweb.util.monolog.wrapper.javaLog.GenericHandler"},
                    {handlerTypes[6], "org.objectweb.util.monolog.wrapper.javaLog.GenericHandler"}

            };
        } else {
            return new String[][] {};
        }
    }

    /**
     * This method allocates org.objectweb.util.monolog.wrapper.javaLog.Logger
     * objects whic are also java.util.logging.Logger and
     * org.objectweb.util.monolog.api.Logger.
     */
    protected synchronized Logger getMonoLogger(String name, String resName) {
        if (name == null)
            throw new IllegalArgumentException(
                    "The specified Logger name is null");
        if (name.equals("root") || name.length()==0) {
            return rootLogger;
        }
        // isolates the logger hierarchy
        name = monoLoggerName(name);
        // Search if the logger already exist.
        Object o = null;
        if (manager != null) {
            o = manager.getLogger(name);
        }
        if (o == null) {
            // It doesn't exist => creates and adds it
            Logger result = new Logger(name, resName);
            if (manager != null) {
                manager.addLogger(result);
            }
            Monolog.debug("Instanciate the logger " + name);

            // In the javadoc of the LogManager, it is said that the caller
            // need to keep a reference to this logger to avoid GC.
            // Because LogManager could use some weak reference on it
            loggers.add(result);
            return result;
        } else if (o instanceof Logger) {
            Monolog.debug("return existing logger " + name);
            return (Logger) o;
        } else {
            // This is not a monolog logger, but still a java.util.log.Logger
            // So we can wrap it safely
            // TODO cache ?
            return new Logger((java.util.logging.Logger) o);
        }
    }


    // IMPLEMENTATION OF THE Configurable INTERFACE //
    //----------------------------------------------//

    /**
     * This method permits to configure the factory.
     * The properties parameter must contains the JAVALOG_CONFIGURATION property.
     * Its value can be DEFAULT, PROPERTY or XML.
     * In the PROPERTY case of the properties parameter must also contain the
     * JAVALOG_CONFIGURATION_FILE property which the value is the configuration
     *  file name.
     * In the CLASS case of the properties parameter must also contain the
     * JAVALOG_CONFIGURATION_CLASS property which the value is the configuration
     *  class name which will initialized the java log system..
     *
     */
    public void configure(Properties prop) throws Exception {
        if (prop == null)
            return;
        String confMode = prop.getProperty(JAVALOG_CONFIGURATION, null);
        if (confMode == null)
            return;

        String param = null;
        if (confMode.equals(PROPERTY)) {
            param = prop.getProperty(JAVALOG_CONFIGURATION_FILE, null);
            if (param != null)
                System.setProperty("java.util.logging.config.file", param);
            manager.readConfiguration();
        } else if (confMode.equals(CLASS)) {
            param = prop.getProperty(JAVALOG_CONFIGURATION_CLASS, null);
            if (param != null)
                System.setProperty("java.util.logging.config.class", param);
            manager.readConfiguration();
        }
    }


    // IMPLEMENTATION OF LoggerFactory INTERFACE //
    //-------------------------------------------//

    public org.objectweb.util.monolog.api.Logger getLogger(String key) {
        return getMonoLogger(key, resourceBundleName);
    }

    public org.objectweb.util.monolog.api.Logger getLogger(String key, String rbn) {
        return getMonoLogger(key, rbn);
    }

    /**
     * It retrieves a list of all loggers.
     */
    public org.objectweb.util.monolog.api.Logger[] getLoggers() {
        ArrayList res = new ArrayList();
        for (Enumeration e = manager.getLoggerNames(); e.hasMoreElements();) {
             Object o = manager.getLogger((String) e.nextElement());
             if (o instanceof Logger) {
                 res.add(o);
             }
        }
         return (Logger[]) res.toArray(new Logger[0]);
     }

    /**
     * Add an extension
     * @param pattern Character corresponding to the info to add
     * @param logInfoProvider instance of info provider
     */
    public void addLogInfo(Character pattern, LogInfo logInfoProvider) throws Exception {
        if (this.logExtensions.get(pattern) != null) {
            throw new Exception("Token '" + pattern + "' aleady used");
        }

        for (char token : Pattern.TOKENS) {
            if (pattern == token) {
                throw new Exception("Token '" + pattern + "' aleady used");
            }
        }
        this.logExtensions.put(pattern, logInfoProvider);
    }

    /**
     * Remove an extension
     * @param pattern log info pattern
     */
    public void removeLogInfo (char pattern) {
        this.logExtensions.remove(pattern);
    }

    /**
     * Gets the provider of the extension
     * @param pattern
     * @return LogInfo provider
     */
    public LogInfo getLogInfo(Character pattern) {
        return this.logExtensions.get(pattern);
    }
}
