/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.objectweb.util.monolog.wrapper.javaLog;

import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.FileHandler;
import java.util.logging.Filter;
import java.util.logging.Formatter;
import java.util.logging.LogRecord;

import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Handler;
import org.objectweb.util.monolog.api.MonologFactory;
import org.objectweb.util.monolog.wrapper.common.RelatifEnvironmentPathGetter;

/**
 * Is a generic handler implementation used to wrapper java.util.logging.Handler
 * instance.
 * @author S.Chassande-Barrioz
 */
public class GenericHandler
        extends java.util.logging.Handler
        implements Handler {

    /**
     * The real handler
     */
    public java.util.logging.Handler handler = null;

    /**
     * the type of the handler (see org.objectweb.util.monolog.api.Handler for
     * the possible values)
     */
    protected String type;

    /**
     * The name of the handler
     */
    protected String name;

    /**
     * Attributes which has been assigned on the handler
     */
    Map attributes = null;

	public GenericHandler() {
		attributes = new HashMap();
	}

	public GenericHandler(String name) {
		this();
		this.name = name;
	}

    /**
     * Builds a generic handler with its name and the type. The real handler
     * will be instanciated after the configuration step.
     * @param name is the name of the handler
     * @param type is the type of the handler
     */
    public GenericHandler(String name, String type) {
        this(name);
		this.type = type;
    }

    /**
     * builds a generic handler since a real handler.
     * @param name is the name of the handler
     * @param h is the real handler
     */
    public GenericHandler(String name, java.util.logging.Handler h) {
        this(name);
		handler = h;
        if (h instanceof FileHandler) {
            type = "file";
        } else if (h instanceof ConsoleHandler) {
            type = "console";
        } else if (h instanceof JMXHandler) {
        	type = "jmx";
        }
    }

    /**
     * It retrieves the name of the handler
     */
    public String getName() {
        return name;
    }

    /**
     * It assigns the name of the handler
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * It retrieves the Handler type
     */
    public String getType() {
        return type;
    }

    /**
     * It retrieves the attributes of the handler
     */
    public String[] getAttributeNames() {
        return (String[]) attributes.keySet().toArray(new String[0]);
    }

    /**
     * It retrieves the value of an attribute value of the handler.
     * @param name is an attribute name
     */
    public Object getAttribute(String n) {
        return attributes.get(n);
    }

    /**
     * It assigns an attributte to the handler.
     * @param _name is the attribute name
     * @param value is the attribute value
     * @return the old value is the attribute was already defined
     */
    public Object setAttribute(String _name, Object value) {
        if (!_name.equalsIgnoreCase("activation")) {
            return attributes.put(_name, value);
        }
		if (type == null) {
			type = (String) attributes.get("handlertype");
		}
        MonologFactory mf = (MonologFactory) value;
        String output = (String) attributes.get(Handler.OUTPUT_ATTRIBUTE);
        output = RelatifEnvironmentPathGetter.getRealPath(output);
        String pattern = (String) attributes.get(Handler.PATTERN_ATTRIBUTE);
        String level = (String) attributes.get(Handler.LEVEL_ATTRIBUTE);
        String append = (String) attributes.get(Handler.APPEND_MODE_ATTRIBUTE);
        String nbfile = (String) attributes.get(Handler.FILE_NUMBER_ATTRIBUTE);
        String fileSize = (String) attributes.get(Handler.MAX_SIZE_ATTRIBUTE);
        boolean appendVal = true;
        if (append != null && append.length() > 0) {
            appendVal = Boolean.getBoolean(append);
        }
        int levelVal = BasicLevel.DEBUG;
        if (level != null && level.length() > 0) {
            levelVal = org.objectweb.util.monolog.wrapper.common.LevelImpl.evaluate(level, mf);
        }
        if ("console".equalsIgnoreCase(type)) {
            handler = new ConsoleHandler();
			if (output != null && output.length() >0) {
				if (output.equalsIgnoreCase("System.err")) {
					((ConsoleHandler) handler).setOutput(System.err);
				} else if (output.equalsIgnoreCase("switch")) {
					((ConsoleHandler) handler).activateSwitching();
				} else if (output.equalsIgnoreCase("System.out")) {
					((ConsoleHandler) handler).setOutput(System.out);
				}
			}
        } else if ("file".equalsIgnoreCase(type)
            || "rollingfile".equalsIgnoreCase(type)) {
            int limit = 0;
            if (fileSize != null && fileSize.length() > 0) {
                limit = Integer.parseInt(fileSize);
            }
            int count = 1;
            if (nbfile != null && nbfile.length() > 0) {
                count = Integer.parseInt(nbfile);
            }
            try {
                handler = new FileHandler(output, limit, count, appendVal);
            } catch (Exception e) {
                throw new IllegalStateException("Error when building the handler '"
					+ name + "': " + e.getMessage());
            }
        } else if ("jmx".equalsIgnoreCase(type)) {
 	       handler = new JMXHandler();
        } else {
			throw new IllegalStateException("Error when building the handler '"
				+ name + "': unknwon type: " + type);
		}
        handler.setFormatter(new MonologFormatter(pattern));
        handler.setLevel(LevelImpl.int2Level(levelVal));
        return null;
    }


    // OVERRIDE THE java.util.logging.Handler CLASS //
    //----------------------------------------------//

    /**
     * Close the Handler and free all associated resources.
     */
    public void close() {
        handler.close();
    }

    /**
     * Flush any buffered output.
     */
    public void flush() {
        handler.flush();
    }

    /**
     * Return the character encoding for this Handler.
     */
    public String getEncoding() {
        return handler.getEncoding();
    }

    /**
     * Get the current Filter for this Handler.
     */
    public Filter getFilter() {
        return handler.getFilter();
    }

    /**
     * Return the Formatter for this Handler.
     */
    public Formatter getFormatter() {
        return handler.getFormatter();
    }

    /**
     * Get the log level specifying which messages will be logged by this Handler.
     */
    public java.util.logging.Level getLevel() {
        System.out.println("handler("+ name + ").getLevel(): " + handler.getLevel().getName());
        return handler.getLevel();
    }

    /**
     * Check if this Handler would actually log a given LogRecord.
     */
    public boolean isLoggable(LogRecord record) {
        //System.out.println("handler("+ name + ").isLoggable(" + record.getLevel().getName() + "): " + handler.isLoggable(record));
        return handler.isLoggable(record);
    }

    /**
     * Publish a LogRecord.
     */
    public void publish(LogRecord record) {
        //System.out.println("handler("+ name + ").publish(" + record.getLevel().getName() + "): " + handler.isLoggable(record));
        handler.publish(record);
    }

    /**
     * Set the character encoding used by this Handler.
     */
    public void setEncoding(String encoding)
            throws SecurityException, UnsupportedEncodingException {
        handler.setEncoding(encoding);
    }

    /**
     * Set the most recent IO exception.
     */
    protected void setException(Exception exception) {
    }

    /**
     * Set a Filter to control output on this Handler.
     */
    public void setFilter(Filter newFilter) {
        handler.setFilter(newFilter);
    }

    /**
     * Set a Formatter.
     */
    public void setFormatter(Formatter newFormatter) {
        handler.setFormatter(newFormatter);
    }

    /**
     * Set the log level specifying which message levels will
     * be logged by this Handler.
     */
    public void setLevel(java.util.logging.Level newLevel) {
        handler.setLevel(newLevel);
    }

}
