/*
 * Decompiled with CFR 0.152.
 */
package org.osiam.client;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import java.io.IOException;
import java.net.URI;
import org.osiam.bundled.javax.ws.rs.ProcessingException;
import org.osiam.bundled.javax.ws.rs.client.Entity;
import org.osiam.bundled.javax.ws.rs.client.WebTarget;
import org.osiam.bundled.javax.ws.rs.core.Form;
import org.osiam.bundled.javax.ws.rs.core.MediaType;
import org.osiam.bundled.javax.ws.rs.core.Response;
import org.osiam.bundled.javax.ws.rs.core.UriBuilder;
import org.osiam.bundled.javax.ws.rs.core.UriBuilderException;
import org.osiam.client.OsiamConnector;
import org.osiam.client.exception.ConflictException;
import org.osiam.client.exception.ConnectionInitializationException;
import org.osiam.client.exception.OAuthErrorMessage;
import org.osiam.client.exception.OsiamClientException;
import org.osiam.client.exception.UnauthorizedException;
import org.osiam.client.oauth.AccessToken;
import org.osiam.client.oauth.GrantType;
import org.osiam.client.oauth.Scope;

class AuthService {
    private static final String BEARER = "Bearer ";
    private static final String TOKEN_ENDPOINT = "/oauth/token";
    private final String endpoint;
    private final String clientId;
    private final String clientSecret;
    private final String clientRedirectUri;
    private final WebTarget targetEndpoint;

    private AuthService(Builder builder) {
        this.endpoint = builder.endpoint;
        this.clientId = builder.clientId;
        this.clientSecret = builder.clientSecret;
        this.clientRedirectUri = builder.clientRedirectUri;
        this.targetEndpoint = OsiamConnector.getClient().target(this.endpoint);
    }

    public AccessToken retrieveAccessToken(Scope ... scopes) {
        String content;
        Response.StatusType status;
        this.ensureClientCredentialsAreSet();
        String formattedScopes = this.getScopesAsString(scopes);
        Form form = new Form();
        form.param("scope", formattedScopes);
        form.param("grant_type", GrantType.CLIENT_CREDENTIALS.getUrlParam());
        try {
            Response response = this.targetEndpoint.path(TOKEN_ENDPOINT).request("application/json").property("jersey.config.client.http.auth.username", this.clientId).property("jersey.config.client.http.auth.password", this.clientSecret).post(Entity.entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));
            status = response.getStatusInfo();
            content = response.readEntity(String.class);
        }
        catch (ProcessingException e) {
            throw this.createGeneralConnectionInitializationException(e);
        }
        this.checkAndHandleResponse(content, status);
        return this.getAccessToken(content);
    }

    public AccessToken retrieveAccessToken(String userName, String password, Scope ... scopes) {
        String content;
        Response.StatusType status;
        this.ensureClientCredentialsAreSet();
        String formattedScopes = this.getScopesAsString(scopes);
        Form form = new Form();
        form.param("scope", formattedScopes);
        form.param("grant_type", GrantType.RESOURCE_OWNER_PASSWORD_CREDENTIALS.getUrlParam());
        form.param("username", userName);
        form.param("password", password);
        try {
            Response response = this.targetEndpoint.path(TOKEN_ENDPOINT).request("application/json").property("jersey.config.client.http.auth.username", this.clientId).property("jersey.config.client.http.auth.password", this.clientSecret).post(Entity.entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));
            status = response.getStatusInfo();
            content = response.readEntity(String.class);
        }
        catch (ProcessingException e) {
            throw this.createGeneralConnectionInitializationException(e);
        }
        this.checkAndHandleResponse(content, status);
        return this.getAccessToken(content);
    }

    public AccessToken retrieveAccessToken(String authCode) {
        String content;
        Response.StatusType status;
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)authCode) ? 1 : 0) != 0, (Object)"The given authentication code can't be null.");
        this.ensureClientCredentialsAreSet();
        Form form = new Form();
        form.param("code", authCode);
        form.param("grant_type", "authorization_code");
        form.param("redirect_uri", this.clientRedirectUri);
        try {
            Response response = this.targetEndpoint.path(TOKEN_ENDPOINT).request("application/json").property("jersey.config.client.http.auth.username", this.clientId).property("jersey.config.client.http.auth.password", this.clientSecret).post(Entity.entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));
            status = response.getStatusInfo();
            content = response.readEntity(String.class);
        }
        catch (ProcessingException e) {
            throw this.createGeneralConnectionInitializationException(e);
        }
        if (status.getStatusCode() == Response.Status.BAD_REQUEST.getStatusCode()) {
            String errorMessage = this.extractErrorMessage(content, status);
            throw new ConflictException(errorMessage);
        }
        this.checkAndHandleResponse(content, status);
        return this.getAccessToken(content);
    }

    private String getScopesAsString(Scope ... scopes) {
        StringBuilder scopeBuilder = new StringBuilder();
        for (Scope scope : scopes) {
            scopeBuilder.append(scope.toString()).append(" ");
        }
        return scopeBuilder.toString().trim();
    }

    public AccessToken refreshAccessToken(AccessToken accessToken, Scope ... scopes) {
        String content;
        Response.StatusType status;
        Preconditions.checkArgument((accessToken != null ? 1 : 0) != 0, (Object)"The given accessToken code can't be null.");
        Preconditions.checkArgument((accessToken.getRefreshToken() != null ? 1 : 0) != 0, (Object)"Unable to perform a refresh_token_grant request without refresh token.");
        this.ensureClientCredentialsAreSet();
        String formattedScopes = this.getScopesAsString(scopes);
        Form form = new Form();
        form.param("scope", formattedScopes);
        form.param("grant_type", GrantType.REFRESH_TOKEN.getUrlParam());
        form.param("refresh_token", accessToken.getRefreshToken());
        try {
            Response response = this.targetEndpoint.path(TOKEN_ENDPOINT).request("application/json").property("jersey.config.client.http.auth.username", this.clientId).property("jersey.config.client.http.auth.password", this.clientSecret).post(Entity.entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));
            status = response.getStatusInfo();
            content = response.readEntity(String.class);
        }
        catch (ProcessingException e) {
            throw this.createGeneralConnectionInitializationException(e);
        }
        if (status.getStatusCode() == Response.Status.BAD_REQUEST.getStatusCode()) {
            throw new ConflictException(this.extractErrorMessage(content, status));
        }
        this.checkAndHandleResponse(content, status);
        return this.getAccessToken(content);
    }

    public URI getAuthorizationUri(Scope ... scopes) {
        Preconditions.checkState((!Strings.isNullOrEmpty((String)this.clientRedirectUri) ? 1 : 0) != 0, (Object)"Can't create the login uri: redirect URI was not set.");
        try {
            String formattedScopes = this.getScopesAsString(scopes);
            return UriBuilder.fromUri(this.endpoint).path("/oauth/authorize").queryParam("client_id", this.clientId).queryParam("response_type", "code").queryParam("redirect_uri", this.clientRedirectUri).queryParam("scope", formattedScopes).build(new Object[0]);
        }
        catch (IllegalArgumentException | UriBuilderException e) {
            throw new OsiamClientException("Unable to create redirect URI", e);
        }
    }

    public AccessToken validateAccessToken(AccessToken tokenToValidate) {
        String content;
        Response.StatusType status;
        Preconditions.checkNotNull((Object)tokenToValidate, (Object)"The tokenToValidate must not be null.");
        try {
            Response response = this.targetEndpoint.path("/token/validation").request("application/json").property("jersey.config.client.http.auth.username", this.clientId).property("jersey.config.client.http.auth.password", this.clientSecret).header("Authorization", BEARER + tokenToValidate.getToken()).post(null);
            status = response.getStatusInfo();
            content = response.readEntity(String.class);
        }
        catch (ProcessingException e) {
            throw this.createGeneralConnectionInitializationException(e);
        }
        this.checkAndHandleResponse(content, status);
        return this.getAccessToken(content);
    }

    public void revokeAccessToken(AccessToken tokenToRevoke) {
        String content;
        Response.StatusType status;
        try {
            Response response = this.targetEndpoint.path("/token/revocation").request("application/json").property("jersey.config.client.http.auth.username", this.clientId).property("jersey.config.client.http.auth.password", this.clientSecret).header("Authorization", BEARER + tokenToRevoke.getToken()).post(null);
            status = response.getStatusInfo();
            content = response.readEntity(String.class);
        }
        catch (ProcessingException e) {
            throw this.createGeneralConnectionInitializationException(e);
        }
        this.checkAndHandleResponse(content, status);
    }

    public void revokeAllAccessTokens(String id, AccessToken accessToken) {
        String content;
        Response.StatusType status;
        try {
            Response response = this.targetEndpoint.path("/token/revocation").path(id).request("application/json").property("jersey.config.client.http.auth.username", this.clientId).property("jersey.config.client.http.auth.password", this.clientSecret).header("Authorization", BEARER + accessToken.getToken()).post(null);
            status = response.getStatusInfo();
            content = response.readEntity(String.class);
        }
        catch (ProcessingException e) {
            throw this.createGeneralConnectionInitializationException(e);
        }
        this.checkAndHandleResponse(content, status);
    }

    private void checkAndHandleResponse(String content, Response.StatusType status) {
        if (status.getStatusCode() == Response.Status.OK.getStatusCode()) {
            return;
        }
        String errorMessage = this.extractErrorMessage(content, status);
        if (status.getStatusCode() == Response.Status.BAD_REQUEST.getStatusCode()) {
            throw new ConnectionInitializationException(errorMessage);
        }
        if (status.getStatusCode() == Response.Status.UNAUTHORIZED.getStatusCode()) {
            throw new UnauthorizedException(errorMessage);
        }
        throw new ConnectionInitializationException(errorMessage);
    }

    private String extractErrorMessage(String content, Response.StatusType status) {
        try {
            OAuthErrorMessage error = (OAuthErrorMessage)new ObjectMapper().readValue(content, OAuthErrorMessage.class);
            return error.getDescription();
        }
        catch (IOException e) {
            String errorMessage = String.format("Could not deserialize the error response for the HTTP status '%s'.", status.getReasonPhrase());
            if (content != null) {
                errorMessage = errorMessage + String.format(" Original response: %s", content);
            }
            return errorMessage;
        }
    }

    private AccessToken getAccessToken(String content) {
        try {
            return (AccessToken)new ObjectMapper().readValue(content, AccessToken.class);
        }
        catch (IOException e) {
            throw new OsiamClientException(String.format("Unable to parse access token: %s", content), e);
        }
    }

    private void ensureClientCredentialsAreSet() {
        Preconditions.checkState((!Strings.isNullOrEmpty((String)this.clientId) ? 1 : 0) != 0, (Object)"The client id can't be null or empty.");
        Preconditions.checkState((!Strings.isNullOrEmpty((String)this.clientSecret) ? 1 : 0) != 0, (Object)"The client secrect can't be null or empty.");
    }

    private ConnectionInitializationException createGeneralConnectionInitializationException(Throwable e) {
        return new ConnectionInitializationException("Unable to retrieve access token.", e);
    }

    public static class Builder {
        private String clientId;
        private String clientSecret;
        private String endpoint;
        private String clientRedirectUri;

        public Builder(String endpoint) {
            this.endpoint = endpoint;
        }

        public Builder setClientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public Builder setClientRedirectUri(String clientRedirectUri) {
            this.clientRedirectUri = clientRedirectUri;
            return this;
        }

        public Builder setClientSecret(String clientSecret) {
            this.clientSecret = clientSecret;
            return this;
        }

        public AuthService build() {
            return new AuthService(this);
        }
    }
}

