/*
 * Copyright 2024 the original author or authors.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.openrewrite.remote.java;

import lombok.EqualsAndHashCode;
import lombok.Value;
import org.jspecify.annotations.Nullable;
import org.openrewrite.*;
import org.openrewrite.config.RecipeDescriptor;
import org.openrewrite.remote.RemotingExecutionContextView;

import static java.util.Objects.requireNonNull;

@Value
@EqualsAndHashCode(callSuper = false)
class RemotingRecipe extends Recipe {

    RecipeDescriptor recipeDescriptor;

    @Override
    public String getDisplayName() {
        return recipeDescriptor.getDisplayName();
    }

    @Override
    public String getDescription() {
        return recipeDescriptor.getDescription();
    }

    @Override
    public TreeVisitor<?, ExecutionContext> getVisitor() {
        return new TreeVisitor<Tree, ExecutionContext>() {
            @Override
            public @Nullable Tree visit(@Nullable Tree tree, ExecutionContext ctx) {
                if (tree instanceof SourceFile) {
                    SourceFile sourceFile = (SourceFile) tree;
                    RemotingClient client = RemotingExecutionContextView.view(ctx)
                            .getMessage(RemotingClient.REMOTING_CLIENT);
                    return requireNonNull(client).runRecipe(recipeDescriptor, sourceFile);
                }
                return tree;
            }
        };
    }
}
