/*
 * Copyright 2024 the original author or authors.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.openrewrite.remote.java;

import com.fasterxml.jackson.dataformat.cbor.CBORFactory;
import lombok.Value;
import lombok.experimental.NonFinal;
import org.jspecify.annotations.Nullable;
import org.openrewrite.ExecutionContext;
import org.openrewrite.InMemoryExecutionContext;
import org.openrewrite.SourceFile;
import org.openrewrite.config.OptionDescriptor;
import org.openrewrite.config.RecipeDescriptor;
import org.openrewrite.remote.AbstractRemotingClient;
import org.openrewrite.remote.RemotingContext;
import org.openrewrite.remote.RemotingExecutionContextView;
import org.openrewrite.remote.RemotingMessenger;

import java.net.Socket;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

@Value
public class RemotingClient extends AbstractRemotingClient {

    static final String REMOTING_CLIENT = "org.openrewrite.remote.remotingClient";

    private static final boolean debug = false;

    @NonFinal
    @Nullable
    SourceFile remoteState;

    Map<RecipeDescriptor, Integer> remoteRecipes = new HashMap<>();

    public RemotingClient(Supplier<Socket> socketSupplier, RemotingContext context, RemotingMessenger messenger) {
        super(socketSupplier, context, messenger);
    }

    public static RemotingClient create(ExecutionContext ctx,
                                        Class<?> contextClass,
                                        Supplier<Socket> socketChannelSupplier) {
        RemotingExecutionContextView view = RemotingExecutionContextView.view(ctx);
        RemotingContext context = view.getRemotingContext();
        if (context == null) {
            context = new RemotingContext(contextClass.getClassLoader(), debug);
        }

        return new RemotingClient(
                socketChannelSupplier,
                context,
                new RemotingMessenger((CBORFactory) context.objectMapper().getFactory(), Collections.emptyMap(),
                        (m) -> new InMemoryExecutionContext())
        );
    }

    static RemotingClient create(RemotingContext context,
                                 RemotingMessenger messenger,
                                 Socket openSocket) {
        RemotingClient client = new RemotingClient(() -> openSocket, context, messenger);
        client.socket = openSocket;

        return client;
    }

    @SuppressWarnings("unused")
    public <T extends SourceFile> T runRecipe(RecipeDescriptor recipe, T sourceFile) {
        if (remoteState != null && !remoteState.equals(sourceFile)) {
            remoteState = null;
        }
        remoteState = runUsingSocket((socket, messenger) -> runRecipe0(recipe, sourceFile, remoteState, socket));
        //noinspection unchecked
        return (T) remoteState;
    }

    private <T extends SourceFile> T runRecipe0(RecipeDescriptor recipe, T sourceFile, @Nullable T remoteState, Socket socket) {
        int recipeId = sendLoadRecipe(recipe, socket);
        return requireNonNull(messenger.sendRequest(
                generator -> {
                    generator.writeString("run-recipe-visitor");
                    generator.writeNumber(recipeId);
                    RemotingMessenger.sendTree(context, generator, sourceFile, remoteState);
                },
                parser -> RemotingMessenger.receiveTree(context, parser, sourceFile),
                socket
        ));
    }

    private int sendLoadRecipe(RecipeDescriptor recipe, Socket socket) {
        if (remoteRecipes.isEmpty()) {
            // FIXME remove this
//            sendReset();
        }
        if (!remoteRecipes.containsKey(recipe)) {
            Integer recipeId = messenger.sendRequest(
                    generator -> {
                        generator.writeString("load-recipe");
                        generator.writeString(recipe.getName());
                        generator.writeStartObject();
                        List<OptionDescriptor> options = recipe.getOptions();
                        for (OptionDescriptor option : options) {
                            generator.writeFieldName(option.getName());
                            generator.writeObject(option.getValue());
                        }
                        generator.writeEndObject();
                    },
                    parser -> {
                        parser.nextToken();
                        return parser.getIntValue();
                    },
                    socket
            );
            remoteRecipes.put(recipe, recipeId);
        }
        return remoteRecipes.get(recipe);
    }

    public <T> T runUsingSocket(BiFunction<Socket, RemotingMessenger, T> command) {
        return withActiveSocket(command);
    }

    private void sendReset() {
        runUsingSocket((socket, messenger) -> {
            messenger.sendReset(socket);
            return null;
        });
        remoteRecipes.clear();
    }
}
