/*
 * Copyright 2024 the original author or authors.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.openrewrite.remote.java;

import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.dataformat.cbor.CBORGenerator;
import com.fasterxml.jackson.dataformat.cbor.CBORParser;
import lombok.SneakyThrows;
import org.jspecify.annotations.Nullable;
import org.openrewrite.*;
import org.openrewrite.config.OptionDescriptor;
import org.openrewrite.config.RecipeDescriptor;
import org.openrewrite.internal.InMemoryLargeSourceSet;
import org.openrewrite.remote.RemotingContext;
import org.openrewrite.remote.RemotingMessenger;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Objects.requireNonNull;

public class CommonHandler {

    public static Map<String, Supplier<RemotingMessenger.RequestHandler<?>>> createHandlersMapping(
            RemotingContext context,
            List<Recipe> recipes) {
        return new HashMap<String, Supplier<RemotingMessenger.RequestHandler<?>>>() {
            {
                put("reset",
                        () -> RemotingMessenger.RequestHandler.of((parser, ctx) -> null,
                                (ignore, generator, ctx) -> {
                                    context.reset();
                                    recipes.clear();
                                }));
                put("load-recipe",
                        () -> RemotingMessenger.RequestHandler.of((parser, ctx) -> {
                            parser.nextToken();
                            String recipeName = parser.getValueAsString();
                            Recipe recipe;
                            try {
//                    Environment env = Environment.builder().scanClassLoader(mapper.getTypeFactory().getClassLoader()).build();
                                Class<? extends Recipe> recipeClass =
                                        context.findClass(recipeName);
                                Constructor<? extends Recipe> constructor =
                                        recipeClass.getConstructor();
                                constructor.setAccessible(true);
                                recipe = constructor.newInstance();
                                parser.nextToken();
                                TreeNode recipeOptions = parser.readValueAsTree();
                            } catch (ClassNotFoundException | NoSuchMethodException |
                                     InstantiationException | IllegalAccessException |
                                     InvocationTargetException e) {
                                throw new RuntimeException(e);
                            }
                            return recipe;
                        }, (recipe, generator, ctx) -> {
                            recipes.add(recipe);
                            generator.writeNumber(recipes.size() - 1);
                        }));
                put("run-recipe-visitor",
                        () -> new RemotingMessenger.RequestHandler<SourceFile>() {
                            @Nullable
                            private SourceFile sourceFile;

                            @Override
                            public @Nullable SourceFile receiveRequest(CBORParser parser,
                                                                       ExecutionContext ctx) throws IOException {
                                parser.nextToken();
                                Recipe recipe = recipes.get(parser.getIntValue());
                                sourceFile = RemotingMessenger.receiveTree(context, parser, null);
                                return (SourceFile) requireNonNull(recipe).getVisitor().visit(sourceFile, ctx);
                            }

                            @SneakyThrows
                            @Override
                            public void sendResponse(@Nullable SourceFile updated,
                                                     CBORGenerator generator,
                                                     ExecutionContext ctx) {
                                RemotingMessenger.sendTree(context, generator, updated, sourceFile);
                            }
                        });

                put("run-recipe",
                        () -> new RemotingMessenger.RequestHandler<RecipeRun>() {

                            private final List<SourceFile> sourceFiles =
                                    new ArrayList<>();

                            @Override
                            public RecipeRun receiveRequest(CBORParser parser,
                                                            ExecutionContext ctx) throws IOException {
                                parser.nextToken();
                                String recipeName = parser.getValueAsString();
                                parser.nextToken();
                                HashMap<String, Object> options =
                                        parser.readValueAs(new TypeReference<Map<String, Object>>() {
                                        });
                                List<OptionDescriptor> recipeOptions = new ArrayList<>();
                                for (Map.Entry<String, Object> entry : options.entrySet()) {
                                    recipeOptions.add(new OptionDescriptor(entry.getKey(),
                                            "",
                                            null,
                                            null,
                                            null,
                                            null,
                                            true,
                                            null));
                                }
                                RecipeDescriptor recipeDescriptor = new RecipeDescriptor(
                                        recipeName,
                                        "",
                                        "",
                                        "",
                                        emptySet(),
                                        null,
                                        recipeOptions,
                                        emptyList(),
                                        emptyList(),
                                        emptyList(),
                                        emptyList(),
                                        emptyList(),
                                        URI.create("recipe://" + recipeName));
                                parser.nextToken();
                                int fileCount = parser.getIntValue();
                                for (int i = 0; i < fileCount; i++) {
                                    sourceFiles.add(RemotingMessenger.receiveTree(context, parser, null));
                                }
                                InMemoryLargeSourceSet largeSourceSet =
                                        new InMemoryLargeSourceSet(sourceFiles);
                                RemotingRecipe recipe =
                                        new RemotingRecipe(recipeDescriptor);
                                return recipe.run(largeSourceSet, ctx);
                            }

                            @SneakyThrows
                            @Override
                            public void sendResponse(@Nullable RecipeRun recipeRun,
                                                     CBORGenerator generator,
                                                     ExecutionContext ctx) {
                                if (recipeRun.getChangeset()
                                            .size() == 0) {
                                    for (SourceFile sourceFile : sourceFiles) {
                                        RemotingMessenger.sendTree(context, generator, sourceFile, sourceFile);
                                    }
                                } else {
                                    List<Result> results = recipeRun.getChangeset()
                                            .getAllResults();
                                    for (Result result : results) {
                                        RemotingMessenger.sendTree(context, generator, result.getAfter(), result.getBefore());
                                    }
                                }
                            }
                        });
                put("print", () -> RemotingMessenger.RequestHandler.of(
                        (parser, ctx) -> {
                            Tree received = RemotingMessenger.receiveTree(context, parser, null);
                            return context.getProvider(received.getClass()).newValidator().validate(received, ctx);
                        },
                        (tree, generator, ctx) -> {
                            String print = requireNonNull(tree).print(new Cursor(null, Cursor.ROOT_VALUE));
                            generator.writeString(print);
                        }));
            }
        };
    }

//    private static  <T extends Tree> T receiveTree(RemotingContext context, CBORParser parser) {
//        return ReceiverContext.receiveTree(null, context.newReceiver(parser), context);
//    }
//
//    private static  <T extends Tree> void sendTree(RemotingContext context, T after,
//		    @Nullable T before,
//		    CBORGenerator generator) {
//        SenderContext.sendTree(after, before, context.newSender(generator));
//    }

}
