/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.model.impl;

import java.io.Serializable;

import org.openrdf.vocabulary.RDF;
import org.openrdf.vocabulary.RDFS;

import org.openrdf.model.GraphException;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

import org.openrdf.sesame.sail.StatementIterator;

/**
 * An implementation of the URI interface.
 **/
public class URIImpl implements URI, Serializable {

/*------------------------------------------------------+
| Some static constants defining rdf and rdfs resources |
+------------------------------------------------------*/

	// RDF
	/** Constant URI for rdf:type **/
	public static final URI RDF_TYPE = new URIImpl(RDF.TYPE);
	/** Constant URI for rdf:Property **/
	public static final URI RDF_PROPERTY = new URIImpl(RDF.PROPERTY);
	/** Constant URI for rdf:XMLLiteral **/
	public static final URI RDF_XMLLITERAL = new URIImpl(RDF.XMLLITERAL);
	/** Constant URI for rdf:subject **/
	public static final URI RDF_SUBJECT = new URIImpl(RDF.SUBJECT);
	/** Constant URI for rdf:predicate **/
	public static final URI RDF_PREDICATE = new URIImpl(RDF.PREDICATE);
	/** Constant URI for rdf:object **/
	public static final URI RDF_OBJECT = new URIImpl(RDF.OBJECT);
	/** Constant URI for rdf:Statement **/
	public static final URI RDF_STATEMENT = new URIImpl(RDF.STATEMENT);
	/** Constant URI for rdf:Alt **/
	public static final URI RDF_ALT = new URIImpl(RDF.ALT);
	/** Constant URI for rdf:Bag **/
	public static final URI RDF_BAG = new URIImpl(RDF.BAG);
	/** Constant URI for rdf:Seq **/
	public static final URI RDF_SEQ = new URIImpl(RDF.SEQ);
	/** Constant URI for rdf:value **/
	public static final URI RDF_VALUE = new URIImpl(RDF.VALUE);
	/** Constant URI for rdf:List **/
	public static final URI RDF_LIST = new URIImpl(RDF.LIST);
	/** Constant URI for rdf:first **/
	public static final URI RDF_FIRST = new URIImpl(RDF.FIRST);
	/** Constant URI for rdf:rest **/
	public static final URI RDF_REST = new URIImpl(RDF.REST);
	/** Constant URI for rdf:nil **/
	public static final URI RDF_NIL = new URIImpl(RDF.NIL);
	
	// RDF Schema
	/** Constant URI for rdfs:Resource **/
	public static final URI RDFS_RESOURCE = new URIImpl(RDFS.RESOURCE);
	/** Constant URI for rdfs:Class **/
	public static final URI RDFS_CLASS = new URIImpl(RDFS.CLASS);
	/** Constant URI for rdfs:Literal **/
	public static final URI RDFS_LITERAL = new URIImpl(RDFS.LITERAL);
	/** Constant URI for rdfs:subClassOf **/
	public static final URI RDFS_SUBCLASSOF = new URIImpl(RDFS.SUBCLASSOF);
	/** Constant URI for rdfs:subPropertyOf **/
	public static final URI RDFS_SUBPROPERTYOF = new URIImpl(RDFS.SUBPROPERTYOF);
	/** Constant URI for rdfs:domain **/
	public static final URI RDFS_DOMAIN = new URIImpl(RDFS.DOMAIN);
	/** Constant URI for rdfs:range **/
	public static final URI RDFS_RANGE = new URIImpl(RDFS.RANGE);
	/** Constant URI for rdfs:comment **/
	public static final URI RDFS_COMMENT = new URIImpl(RDFS.COMMENT);
	/** Constant URI for rdfs:label **/
	public static final URI RDFS_LABEL = new URIImpl(RDFS.LABEL);
	/** Constant URI for rdfs:isDefinedBy **/
	public static final URI RDFS_ISDEFINEDBY = new URIImpl(RDFS.ISDEFINEDBY);
	/** Constant URI for rdfs:seeAlso **/
	public static final URI RDFS_SEEALSO = new URIImpl(RDFS.SEEALSO);
	/** Constant URI for rdfs:Datatype **/
	public static final URI RDFS_DATATYPE = new URIImpl(RDFS.DATATYPE);
	/** Constant URI for rdfs:Container **/
	public static final URI RDFS_CONTAINER = new URIImpl(RDFS.CONTAINER);
	/** Constant URI for rdfs:member **/
	public static final URI RDFS_MEMBER = new URIImpl(RDFS.MEMBER);
	/** Constant URI for rdfs:ContainerMembershipProperty **/
	public static final URI RDFS_CONTAINERMEMBERSHIPPROPERTY = new URIImpl(RDFS.CONTAINERMEMBERSHIPPROPERTY);
	
/*---------------------------------------+
| Variables                              |
+---------------------------------------*/

	/** The namespace. **/
    private String _namespace;

	/** local name **/
    private String _localName;

/*---------------------------------------+
| Constructors                           |
+---------------------------------------*/

	/**
	 * Creates a new URI from the supplied string.
	 *
	 * @param uri A String representing a legal, absolute URI.
	 * @throws IllegalArgumentException If the supplied URI is not a
	 * legal (absolute) URI.
	 **/
	public URIImpl(String uri) {
		// Find the place to split the uri
		int i = uri.length() - 1;
		while (i >= 0) {
			char c = uri.charAt(i);
			if (c == '#' || c == ':' || c == '/') {
				break;
			}
			i--;
		}

		if (i > 0) {
			// Split the uri
			_namespace = uri.substring(0, i + 1);
			_localName = uri.substring(i + 1);
		}
		else {
			throw new IllegalArgumentException("'"+uri+"' is not a legal (absolute) URI");
		}
	}

	/**
	 * Creates a new URI that will get the supplied namespace and local name.
	 *
	 * @param namespace A namespace.
	 * @param localName A legal local name. A legal local name adheres to the definition of an NCName as specified at
	 * <a href="http://www.w3.org/TR/REC-xml-names/#NT-NCName">http://www.w3.org/TR/REC-xml-names/#NT-NCName</a>.
	 **/
	public URIImpl(String namespace, String localName) {
		if (namespace == null) {
			throw new IllegalArgumentException("namespace must not be null");
		}
		if (localName == null) {
			throw new IllegalArgumentException("localName must not be null");
		}

		_namespace = namespace;
		_localName = localName;
	}

/*---------------------------------------+
| Methods                                |
+---------------------------------------*/

	// inherit comments
	public String getNamespace() {
		return _namespace;
	}

	// inherit comments
	public String getLocalName() {
		return _localName;
	}

	// inherit comments
	public String getURI() {
		// This code is (much) more efficient then just concatenating the two strings.
		char[] result = new char[_namespace.length() + _localName.length()];
		_namespace.getChars(0, _namespace.length(), result, 0);
		_localName.getChars(0, _localName.length(), result, _namespace.length());
		return new String(result);
	}

	// implements URI.equals(Object)
    public boolean equals(Object o) {
		if (this == o) {
			return true;
		}

        if (o instanceof URI) {
			URI other = (URI)o;

			return
				_localName.equals(other.getLocalName()) &&
				_namespace.equals(other.getNamespace());
		}

		return false;
	}

	// Implements URI.hashCode()
	public int hashCode() {
		return _namespace.hashCode() ^ _localName.hashCode();
	}

	/**
	 * Compares this URI with the supplied object.
	 *
	 * @throws ClassCastException if o is not of type Value
	 * @see java.lang.Comparable
	 **/
	public int compareTo(Object o) {
		if (this == o) {
			return 0;
		}

		Value other = (Value)o;

		if (other instanceof URI) {
			return getURI().compareTo( ((URI)other).getURI() );
		}
		else {
			// URI < BNode and URI < Literal
			return -1;
		}
	}

	/**
	 * Returns the String-representation of this URI.
	 **/
	public String toString() {
		return getURI();
	}

	// Implements URI.getPredicateStatements()
	public StatementIterator getPredicateStatements()
		throws GraphException
	{
		throw new GraphException("no backing store associated");
	}

	// Implements Resource.addProperty(URI, Value)
	public void addProperty(URI property, Value value)
		throws GraphException
	{
		throw new GraphException("no backing store associated");		
	}

	// Implements Resource.getSubjectStatements()
	public StatementIterator getSubjectStatements()
		throws GraphException
	{
		throw new GraphException("no backing store associated");
	}

	// Implements Value.getObjectStatements()
	public StatementIterator getObjectStatements()
		throws GraphException
	{
		throw new GraphException("no backing store associated");
	}
}
