/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.netconf.test.tool;

import com.google.common.collect.Lists;
import java.io.IOException;
import java.net.Authenticator;
import java.net.PasswordAuthentication;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.Callable;
import java.util.concurrent.Semaphore;
import java.util.stream.Collectors;
import org.opendaylight.netconf.test.tool.PayloadCreator;
import org.opendaylight.netconf.test.tool.TesttoolParameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class Execution
implements Callable<Void> {
    private static final Logger LOG = LoggerFactory.getLogger(Execution.class);
    private static final String NETCONF_TOPOLOGY_DESTINATION = "http://%s:%s/rests/data/network-topology:network-topology/topology=topology-netconf";
    private final HttpClient httpClient;
    private final String destination;
    private final List<Integer> openDevices;
    private final TesttoolParameters params;
    private final Semaphore semaphore;
    private final int throttle;
    private final boolean isAsync;

    Execution(List<Integer> openDevices, final TesttoolParameters params) {
        this.httpClient = HttpClient.newBuilder().authenticator(new Authenticator(this){

            @Override
            protected PasswordAuthentication getPasswordAuthentication() {
                return new PasswordAuthentication(params.controllerAuthUsername, params.controllerAuthPassword.toCharArray());
            }
        }).version(HttpClient.Version.HTTP_1_1).build();
        this.destination = String.format(Locale.ROOT, NETCONF_TOPOLOGY_DESTINATION, params.controllerIp, params.controllerPort);
        this.openDevices = openDevices;
        this.params = params;
        this.throttle = params.throttle / params.threadAmount;
        this.isAsync = params.async;
        if (params.async && params.threadAmount > 1) {
            LOG.info("Throttling per thread: {}", (Object)this.throttle);
        }
        this.semaphore = new Semaphore(this.throttle);
    }

    @Override
    public Void call() {
        List<HttpRequest> requests = this.prepareRequests();
        if (this.isAsync) {
            this.sendAsync(requests);
        } else {
            this.sendSync(requests);
        }
        return null;
    }

    private List<HttpRequest> prepareRequests() {
        List batches = Lists.partition(this.openDevices, (int)this.params.generateConfigBatchSize);
        return batches.stream().map(b -> PayloadCreator.createStringPayload(b, this.params)).map(this::prepareRequest).collect(Collectors.toList());
    }

    private void sendAsync(List<HttpRequest> requests) {
        LOG.info("Begin sending async requests");
        for (HttpRequest request : requests) {
            try {
                this.semaphore.acquire();
            }
            catch (InterruptedException e) {
                LOG.warn("Semaphore acquire interrupted");
            }
            this.httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString()).whenComplete((response, error) -> {
                if (response.statusCode() != 200) {
                    LOG.warn("Unexpected status code: {} for request to uri: {} with body: {}", new Object[]{response.statusCode(), request.uri(), response.body()});
                }
                this.semaphore.release();
            });
        }
        LOG.info("Requests sent, waiting for responses");
        try {
            this.semaphore.acquire(this.throttle);
        }
        catch (InterruptedException e) {
            LOG.warn("Semaphore acquire interrupted");
        }
        LOG.info("Responses received, ending...");
    }

    private void sendSync(List<HttpRequest> requests) {
        LOG.info("Begin sending sync requests");
        for (HttpRequest request : requests) {
            try {
                HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
                if (response.statusCode() == 200) continue;
                LOG.warn("Unexpected status code: {} for request to uri: {} with body: {}", new Object[]{response.statusCode(), request.uri(), response.body()});
            }
            catch (IOException | InterruptedException e) {
                LOG.error("Failed to execute request: {}", (Object)request, (Object)e);
                throw new IllegalStateException("Failed to execute request", e);
            }
        }
        LOG.info("End sending sync requests");
    }

    private HttpRequest prepareRequest(String payload) {
        LOG.info("Creating request to: {} with payload: {}", (Object)this.destination, (Object)payload);
        return HttpRequest.newBuilder(URI.create(this.destination)).method("PATCH", HttpRequest.BodyPublishers.ofString(payload, StandardCharsets.UTF_8)).header("Content-Type", "application/json").header("Accept", "application/json").build();
    }
}

