/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.netconf.server;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.netconf.api.DocumentedException;
import org.opendaylight.netconf.api.NetconfSession;
import org.opendaylight.netconf.api.NetconfSessionListener;
import org.opendaylight.netconf.api.NetconfTerminationReason;
import org.opendaylight.netconf.api.messages.NetconfMessage;
import org.opendaylight.netconf.api.messages.NotificationMessage;
import org.opendaylight.netconf.api.xml.XmlUtil;
import org.opendaylight.netconf.server.NetconfServerSession;
import org.opendaylight.netconf.server.SendErrorExceptionUtil;
import org.opendaylight.netconf.server.api.monitoring.NetconfMonitoringService;
import org.opendaylight.netconf.server.api.monitoring.SessionEvent;
import org.opendaylight.netconf.server.api.monitoring.SessionListener;
import org.opendaylight.netconf.server.osgi.NetconfOperationRouterImpl;
import org.opendaylight.netconf.server.spi.SubtreeFilter;
import org.opendaylight.yangtools.yang.common.ErrorSeverity;
import org.opendaylight.yangtools.yang.common.ErrorTag;
import org.opendaylight.yangtools.yang.common.ErrorType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

public class NetconfServerSessionListener
implements NetconfSessionListener<NetconfServerSession> {
    private static final Logger LOG = LoggerFactory.getLogger(NetconfServerSessionListener.class);
    private final SessionListener monitoringSessionListener;
    private final NetconfOperationRouterImpl operationRouter;
    private final AutoCloseable onSessionDownCloseable;

    NetconfServerSessionListener(NetconfOperationRouterImpl operationRouter, NetconfMonitoringService monitoringService, AutoCloseable onSessionDownCloseable) {
        this.operationRouter = Objects.requireNonNull(operationRouter);
        this.monitoringSessionListener = monitoringService.getSessionListener();
        this.onSessionDownCloseable = onSessionDownCloseable;
    }

    public void onSessionUp(NetconfServerSession netconfNetconfServerSession) {
        this.monitoringSessionListener.onSessionUp(netconfNetconfServerSession);
    }

    public void onSessionDown(NetconfServerSession netconfNetconfServerSession, Exception cause) {
        LOG.debug("Session {} down, reason: {}", (Object)netconfNetconfServerSession, (Object)cause.getMessage());
        this.onDown(netconfNetconfServerSession);
    }

    public void onDown(NetconfServerSession netconfNetconfServerSession) {
        this.monitoringSessionListener.onSessionDown(netconfNetconfServerSession);
        try {
            this.operationRouter.close();
        }
        catch (Exception closingEx) {
            LOG.debug("Ignoring exception while closing operationRouter", (Throwable)closingEx);
        }
        try {
            this.onSessionDownCloseable.close();
        }
        catch (Exception ex) {
            LOG.debug("Ignoring exception while closing onSessionDownCloseable", (Throwable)ex);
        }
    }

    public void onSessionTerminated(NetconfServerSession netconfNetconfServerSession, NetconfTerminationReason netconfTerminationReason) {
        LOG.debug("Session {} terminated, reason: {}", (Object)netconfNetconfServerSession, (Object)netconfTerminationReason.getErrorMessage());
        this.onDown(netconfNetconfServerSession);
    }

    public void onMessage(NetconfServerSession session, NetconfMessage netconfMessage) {
        try {
            Preconditions.checkState((this.operationRouter != null ? 1 : 0) != 0, (Object)"Cannot handle message, session up was not yet received");
            NetconfMessage message = this.processDocument(netconfMessage, session);
            LOG.debug("Responding with message {}", (Object)message);
            session.sendMessage(message).addListener(future -> {
                Throwable cause = future.cause();
                if (cause != null) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Failed to send response {}", (Object)NetconfServerSessionListener.getMessageId(message), (Object)cause);
                    }
                    session.counters().incOutRpcErrors();
                    this.monitoringSessionListener.onSessionEvent(SessionEvent.outRpcError(session));
                } else if (LOG.isDebugEnabled()) {
                    LOG.debug("Finished sending response {}", (Object)NetconfServerSessionListener.getMessageId(message));
                }
            });
            this.monitoringSessionListener.onSessionEvent(SessionEvent.inRpcSuccess(session));
        }
        catch (RuntimeException e) {
            LOG.error("Unexpected exception", (Throwable)e);
            session.counters().incInBadRpcs();
            this.monitoringSessionListener.onSessionEvent(SessionEvent.inRpcFail(session));
            throw new IllegalStateException("Unable to process incoming message " + String.valueOf(netconfMessage), e);
        }
        catch (DocumentedException e) {
            LOG.trace("Error occurred while processing message", (Throwable)e);
            session.counters().incOutRpcErrors();
            session.counters().incInBadRpcs();
            this.monitoringSessionListener.onSessionEvent(SessionEvent.inRpcFail(session));
            this.monitoringSessionListener.onSessionEvent(SessionEvent.outRpcError(session));
            SendErrorExceptionUtil.sendErrorMessage((NetconfSession)session, e, netconfMessage);
        }
    }

    private static Attr getMessageId(NetconfMessage message) {
        return message.getDocument().getDocumentElement().getAttributeNode("message-id");
    }

    public void onError(NetconfServerSession session, Exception failure) {
        session.counters().incInBadRpcs();
        this.monitoringSessionListener.onSessionEvent(SessionEvent.inRpcFail(session));
        throw new IllegalStateException("Unable to process incoming message", failure);
    }

    public void onNotification(NetconfServerSession session, NotificationMessage notification) {
        this.monitoringSessionListener.onSessionEvent(SessionEvent.notification(session));
    }

    private NetconfMessage processDocument(NetconfMessage netconfMessage, NetconfServerSession session) throws DocumentedException {
        Document incomingDocument = netconfMessage.getDocument();
        Element rootNode = incomingDocument.getDocumentElement();
        if ("rpc".equals(rootNode.getLocalName())) {
            Document responseDocument = XmlUtil.newDocument();
            NetconfServerSessionListener.checkMessageId(rootNode);
            Document rpcReply = this.operationRouter.onNetconfMessage(incomingDocument, session);
            rpcReply = SubtreeFilter.applyRpcSubtreeFilter(incomingDocument, rpcReply);
            session.counters().incInRpcs();
            responseDocument.appendChild(responseDocument.importNode(rpcReply.getDocumentElement(), true));
            return new NetconfMessage(responseDocument);
        }
        throw new DocumentedException("Unknown tag " + rootNode.getNodeName() + " in message:\n" + String.valueOf(netconfMessage), ErrorType.PROTOCOL, ErrorTag.UNKNOWN_ELEMENT, ErrorSeverity.ERROR, (Map)ImmutableMap.of((Object)"bad-element", (Object)rootNode.getNodeName()));
    }

    private static void checkMessageId(Node rootNode) throws DocumentedException {
        NamedNodeMap attributes = rootNode.getAttributes();
        if (attributes.getNamedItemNS("urn:ietf:params:xml:ns:netconf:base:1.0", "message-id") != null) {
            return;
        }
        if (attributes.getNamedItem("message-id") != null) {
            return;
        }
        throw new DocumentedException("Missing attribute " + rootNode.getNodeName(), ErrorType.RPC, ErrorTag.MISSING_ATTRIBUTE, ErrorSeverity.ERROR, (Map)ImmutableMap.of((Object)"bad-attribute", (Object)"message-id", (Object)"bad-element", (Object)"rpc"));
    }
}

