/**
 * Copyright (C) 2018 Cambridge Systematics, Inc.
 *
 * <p>Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of the License at
 *
 * <p>http://www.apache.org/licenses/LICENSE-2.0
 *
 * <p>Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.onebusaway.gtfs_transformer.impl;

import java.util.List;
import org.onebusaway.gtfs.model.AgencyAndId;
import org.onebusaway.gtfs.model.Stop;
import org.onebusaway.gtfs.services.GtfsMutableRelationalDao;
import org.onebusaway.gtfs_transformer.services.GtfsTransformStrategy;
import org.onebusaway.gtfs_transformer.services.TransformContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

// based on the control file, we are changing from the "new" id to the "old" id
// in addition to changing the id, also update the stop code
public class UpdateStopIdsFromFile implements GtfsTransformStrategy {

  private final Logger _log = LoggerFactory.getLogger(UpdateStopIdsFromFile.class);

  private static final int NEW_STOP_ID = 1;
  private static final int OLD_STOP_ID = 2;
  private static final int NEW_STOP_CODE = 3;

  @Override
  public String getName() {
    return this.getClass().getSimpleName();
  }

  @Override
  public void run(TransformContext context, GtfsMutableRelationalDao dao) {

    List<String> controlLines =
        new InputLibrary().readList((String) context.getParameter("controlFile"));
    int matched = 0;
    int unmatched = 0;
    int duplicate = 0;

    AgencyAndId agencyAndId = dao.getAllStops().iterator().next().getId();

    for (String controlLine : controlLines) {
      String[] controlArray = controlLine.split(",");
      if (controlArray == null || controlArray.length < 2) {
        _log.info("bad control line {}", controlLine);
        continue;
      }
      String oldId = controlArray[OLD_STOP_ID];
      String newId = controlArray[NEW_STOP_ID];
      String newCode = controlArray[NEW_STOP_CODE];

      Stop stop = dao.getStopForId(new AgencyAndId(agencyAndId.getAgencyId(), newId));

      if (stop == null) {
        if (!newId.equals("0")) {
          _log.info("missing stop for new id {}", newId);
        } else {
          _log.error("No stop found for id {}", newId);
        }
        unmatched++;
        continue;
      }

      matched++;
      _log.info("Setting existing new id {} to old id {}", newId, oldId);
      stop.setId(new AgencyAndId(stop.getId().getAgencyId(), oldId));
      stop.setCode(newCode);
    }
    _log.info(
        "Complete with {} matched and {} unmatched and {} duplicates",
        matched,
        unmatched,
        duplicate);
  }
}
