/*
 * Copyright (c) 2002-2023 "Neo4j,"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.neo4j.ogm.drivers.bolt.driver;

import org.neo4j.driver.Driver;
import org.neo4j.ogm.Neo4jOgmVersion;

/**
 * A similar version of this class written by the same author appeared in SDN 6.1.11 for the first time.
 * Representation of a user agent containing sensible information to identify queries generated by or executed via Neo4j-OGM.
 *
 * @author Michael J. Simons
 */
public enum UserAgent {

	INSTANCE(
			getVersionOf(Driver.class),
			Neo4jOgmVersion.resolve()
	);

	private final String driverVersion;

	private final String neo4jOgmVersion;

	private final String representation;

	UserAgent(String driverVersion, String neo4jOgmVersion) {
		int idxOfDash = driverVersion == null ? -1 : driverVersion.indexOf('-');
		this.driverVersion = driverVersion == null ?
				null :
				driverVersion.substring(0, idxOfDash > 0 ? idxOfDash : driverVersion.length());
		this.neo4jOgmVersion = neo4jOgmVersion;

		String unknown = "-";
		this.representation = String.format("Java/%s (%s %s %s) neo4j-java/%s neo4j-ogm/%s",
				System.getProperty("java.version"),
				System.getProperty("java.vm.vendor"),
				System.getProperty("java.vm.name"),
				System.getProperty("java.vm.version"),
				this.driverVersion == null ? unknown : this.driverVersion,
				this.neo4jOgmVersion == null ? unknown : this.neo4jOgmVersion
		);
	}

	public String getDriverVersion() {
		return driverVersion;
	}

	public String getNeo4jOgmVersion() {
		return neo4jOgmVersion;
	}

    private static String getVersionOf(Class<?> type) {

        Package p = type.getPackage();
        String version = p.getImplementationVersion();
        if (!(version == null || version.trim().isEmpty())) {
            return version;
        }
        return null;
    }

    @Override
	public String toString() {
		return this.representation;
	}
}
