/*
 * Copyright (c) 2002-2020 "Neo4j,"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.neo4j.logging;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.function.Function;

/**
 * A {@link LogProvider} implementation that duplicates all messages to other LogProvider instances
 */
public class DuplicatingLogProvider extends AbstractLogProvider<DuplicatingLog>
{
    private final CopyOnWriteArraySet<LogProvider> logProviders;
    private final Map<DuplicatingLog,Map<LogProvider,Log>> duplicatingLogCache =
            Collections.synchronizedMap( new WeakHashMap<>() );

    /**
     * @param logProviders A list of {@link LogProvider} instances that messages should be duplicated to
     */
    public DuplicatingLogProvider( LogProvider... logProviders )
    {
        this.logProviders = new CopyOnWriteArraySet<>( Arrays.asList( logProviders ) );
    }

    @Override
    protected DuplicatingLog buildLog( final Class<?> loggingClass )
    {
        return buildLog( logProvider -> logProvider.getLog( loggingClass ) );
    }

    @Override
    protected DuplicatingLog buildLog( final String name )
    {
        return buildLog( logProvider -> logProvider.getLog( name ) );
    }

    private DuplicatingLog buildLog( Function<LogProvider, Log> logConstructor )
    {
        List<Log> logs = new ArrayList<>( logProviders.size() );
        Map<LogProvider, Log> providedLogs = new HashMap<>();
        for ( LogProvider logProvider : logProviders )
        {
            Log log = logConstructor.apply( logProvider );
            providedLogs.put( logProvider, log );
            logs.add( log );
        }
        DuplicatingLog duplicatingLog = new DuplicatingLog( logs );
        duplicatingLogCache.put( duplicatingLog, providedLogs );
        return duplicatingLog;
    }
}
