/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.fs;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.NoSuchFileException;
import java.nio.file.StandardCopyOption;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.hamcrest.core.Is;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.neo4j.function.Predicates;
import org.neo4j.io.fs.FileHandle;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.OpenMode;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.fs.watcher.FileWatcher;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.TestDirectoryExtension;
import org.neo4j.test.matchers.ByteArrayMatcher;
import org.neo4j.test.rule.TestDirectory;

@ExtendWith(value={TestDirectoryExtension.class})
public abstract class FileSystemAbstractionTest {
    @Inject
    TestDirectory testDirectory;
    private int recordSize = 9;
    private int maxPages = 20;
    private int pageCachePageSize = 32;
    private int recordsPerFilePage = this.pageCachePageSize / this.recordSize;
    private int recordCount = 25 * this.maxPages * this.recordsPerFilePage;
    protected FileSystemAbstraction fsa;
    protected File path;

    @BeforeEach
    void before() {
        this.fsa = this.buildFileSystemAbstraction();
        this.path = new File(this.testDirectory.directory(), UUID.randomUUID().toString());
    }

    @AfterEach
    void tearDown() throws Exception {
        this.fsa.close();
    }

    protected abstract FileSystemAbstraction buildFileSystemAbstraction();

    @Test
    void shouldCreatePath() throws Exception {
        this.fsa.mkdirs(this.path);
        Assertions.assertTrue((boolean)this.fsa.fileExists(this.path));
    }

    @Test
    void shouldCreateDeepPath() throws Exception {
        this.path = new File(this.path, UUID.randomUUID() + "/" + UUID.randomUUID());
        this.fsa.mkdirs(this.path);
        Assertions.assertTrue((boolean)this.fsa.fileExists(this.path));
    }

    @Test
    void shouldCreatePathThatAlreadyExists() throws Exception {
        this.fsa.mkdirs(this.path);
        Assertions.assertTrue((boolean)this.fsa.fileExists(this.path));
        this.fsa.mkdirs(this.path);
        Assertions.assertTrue((boolean)this.fsa.fileExists(this.path));
    }

    @Test
    void shouldCreatePathThatPointsToFile() throws Exception {
        this.fsa.mkdirs(this.path);
        Assertions.assertTrue((boolean)this.fsa.fileExists(this.path));
        this.path = new File(this.path, "some_file");
        try (StoreChannel channel = this.fsa.create(this.path);){
            MatcherAssert.assertThat((Object)channel, (Matcher)Is.is((Matcher)Matchers.not((Matcher)Matchers.nullValue())));
            this.fsa.mkdirs(this.path);
            Assertions.assertTrue((boolean)this.fsa.fileExists(this.path));
        }
    }

    @Test
    void moveToDirectoryMustMoveFile() throws Exception {
        File source = new File(this.path, "source");
        File target = new File(this.path, "target");
        File file = new File(source, "file");
        File fileAfterMove = new File(target, "file");
        this.fsa.mkdirs(source);
        this.fsa.mkdirs(target);
        this.fsa.create(file).close();
        Assertions.assertTrue((boolean)this.fsa.fileExists(file));
        Assertions.assertFalse((boolean)this.fsa.fileExists(fileAfterMove));
        this.fsa.moveToDirectory(file, target);
        Assertions.assertFalse((boolean)this.fsa.fileExists(file));
        Assertions.assertTrue((boolean)this.fsa.fileExists(fileAfterMove));
    }

    @Test
    void copyToDirectoryCopiesFile() throws IOException {
        File source = new File(this.path, "source");
        File target = new File(this.path, "target");
        File file = new File(source, "file");
        File fileAfterCopy = new File(target, "file");
        this.fsa.mkdirs(source);
        this.fsa.mkdirs(target);
        this.fsa.create(file).close();
        Assertions.assertTrue((boolean)this.fsa.fileExists(file));
        Assertions.assertFalse((boolean)this.fsa.fileExists(fileAfterCopy));
        this.fsa.copyToDirectory(file, target);
        Assertions.assertTrue((boolean)this.fsa.fileExists(file));
        Assertions.assertTrue((boolean)this.fsa.fileExists(fileAfterCopy));
    }

    @Test
    void copyToDirectoryReplaceExistingFile() throws Exception {
        File source = new File(this.path, "source");
        File target = new File(this.path, "target");
        File file = new File(source, "file");
        File targetFile = new File(target, "file");
        this.fsa.mkdirs(source);
        this.fsa.mkdirs(target);
        this.fsa.create(file).close();
        this.writeIntegerIntoFile(targetFile);
        this.fsa.copyToDirectory(file, target);
        Assertions.assertTrue((boolean)this.fsa.fileExists(file));
        Assertions.assertTrue((boolean)this.fsa.fileExists(targetFile));
        Assertions.assertEquals((long)0L, (long)this.fsa.getFileSize(targetFile));
    }

    @Test
    void deleteRecursivelyMustDeleteAllFilesInDirectory() throws Exception {
        this.fsa.mkdirs(this.path);
        File a = new File(this.path, "a");
        this.fsa.create(a).close();
        File b = new File(this.path, "b");
        this.fsa.create(b).close();
        File c = new File(this.path, "c");
        this.fsa.create(c).close();
        File d = new File(this.path, "d");
        this.fsa.create(d).close();
        this.fsa.deleteRecursively(this.path);
        Assertions.assertFalse((boolean)this.fsa.fileExists(a));
        Assertions.assertFalse((boolean)this.fsa.fileExists(b));
        Assertions.assertFalse((boolean)this.fsa.fileExists(c));
        Assertions.assertFalse((boolean)this.fsa.fileExists(d));
    }

    @Test
    void deleteRecursivelyMustDeleteGivenDirectory() throws Exception {
        this.fsa.mkdirs(this.path);
        this.fsa.deleteRecursively(this.path);
        Assertions.assertFalse((boolean)this.fsa.fileExists(this.path));
    }

    @Test
    void deleteRecursivelyMustDeleteGivenFile() throws Exception {
        this.fsa.mkdirs(this.path);
        File file = new File(this.path, "file");
        this.fsa.create(file).close();
        this.fsa.deleteRecursively(file);
        Assertions.assertFalse((boolean)this.fsa.fileExists(file));
    }

    @Test
    void fileWatcherCreation() throws IOException {
        try (FileWatcher fileWatcher = this.fsa.fileWatcher();){
            Assertions.assertNotNull((Object)fileWatcher.watch(this.testDirectory.directory("testDirectory")));
        }
    }

    @Test
    void readAndWriteMustTakeBufferPositionIntoAccount() throws Exception {
        byte[] bytes = new byte[]{1, 2, 3, 4, 5};
        ByteBuffer buf = ByteBuffer.wrap(bytes);
        buf.position(1);
        this.fsa.mkdirs(this.path);
        File file = new File(this.path, "file");
        try (StoreChannel channel = this.fsa.open(file, OpenMode.READ_WRITE);){
            MatcherAssert.assertThat((Object)channel.write(buf), (Matcher)Is.is((Object)4));
        }
        var5_5 = null;
        try (InputStream stream = this.fsa.openAsInputStream(file);){
            MatcherAssert.assertThat((Object)stream.read(), (Matcher)Is.is((Object)2));
            MatcherAssert.assertThat((Object)stream.read(), (Matcher)Is.is((Object)3));
            MatcherAssert.assertThat((Object)stream.read(), (Matcher)Is.is((Object)4));
            MatcherAssert.assertThat((Object)stream.read(), (Matcher)Is.is((Object)5));
            MatcherAssert.assertThat((Object)stream.read(), (Matcher)Is.is((Object)-1));
        }
        catch (Throwable throwable) {
            var5_5 = throwable;
            throw throwable;
        }
        Arrays.fill(bytes, (byte)0);
        buf.position(1);
        channel = this.fsa.open(file, OpenMode.READ_WRITE);
        var5_5 = null;
        try {
            MatcherAssert.assertThat((Object)channel.read(buf), (Matcher)Is.is((Object)4));
            buf.clear();
            MatcherAssert.assertThat((Object)buf.get(), (Matcher)Is.is((Object)0));
            MatcherAssert.assertThat((Object)buf.get(), (Matcher)Is.is((Object)2));
            MatcherAssert.assertThat((Object)buf.get(), (Matcher)Is.is((Object)3));
            MatcherAssert.assertThat((Object)buf.get(), (Matcher)Is.is((Object)4));
            MatcherAssert.assertThat((Object)buf.get(), (Matcher)Is.is((Object)5));
        }
        catch (Throwable throwable) {
            var5_5 = throwable;
            throw throwable;
        }
        finally {
            if (channel != null) {
                if (var5_5 != null) {
                    try {
                        channel.close();
                    }
                    catch (Throwable throwable) {
                        var5_5.addSuppressed(throwable);
                    }
                } else {
                    channel.close();
                }
            }
        }
    }

    @Test
    void streamFilesRecursiveMustBeEmptyForEmptyBaseDirectory() throws Exception {
        File dir = this.existingDirectory("dir");
        MatcherAssert.assertThat((Object)this.fsa.streamFilesRecursive(dir).count(), (Matcher)Matchers.is((Object)0L));
    }

    @Test
    void streamFilesRecursiveMustListAllFilesInBaseDirectory() throws Exception {
        File a = this.existingFile("a");
        File b = this.existingFile("b");
        File c = this.existingFile("c");
        Stream stream = this.fsa.streamFilesRecursive(a.getParentFile());
        List filepaths = stream.map(FileHandle::getFile).collect(Collectors.toList());
        MatcherAssert.assertThat(filepaths, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{a.getCanonicalFile(), b.getCanonicalFile(), c.getCanonicalFile()}));
    }

    @Test
    void streamFilesRecursiveMustListAllFilesInSubDirectories() throws Exception {
        File sub1 = this.existingDirectory("sub1");
        File sub2 = this.existingDirectory("sub2");
        File a = this.existingFile("a");
        File b = new File(sub1, "b");
        File c = new File(sub2, "c");
        this.ensureExists(b);
        this.ensureExists(c);
        Stream stream = this.fsa.streamFilesRecursive(a.getParentFile());
        List filepaths = stream.map(FileHandle::getFile).collect(Collectors.toList());
        MatcherAssert.assertThat(filepaths, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{a.getCanonicalFile(), b.getCanonicalFile(), c.getCanonicalFile()}));
    }

    @Test
    void streamFilesRecursiveMustNotListSubDirectories() throws Exception {
        File sub1 = this.existingDirectory("sub1");
        File sub2 = this.existingDirectory("sub2");
        File sub2sub1 = new File(sub2, "sub1");
        this.ensureDirectoryExists(sub2sub1);
        this.existingDirectory("sub3");
        File a = this.existingFile("a");
        File b = new File(sub1, "b");
        File c = new File(sub2, "c");
        this.ensureExists(b);
        this.ensureExists(c);
        Stream stream = this.fsa.streamFilesRecursive(a.getParentFile());
        List filepaths = stream.map(FileHandle::getFile).collect(Collectors.toList());
        MatcherAssert.assertThat(filepaths, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{a.getCanonicalFile(), b.getCanonicalFile(), c.getCanonicalFile()}));
    }

    @Test
    void streamFilesRecursiveFilePathsMustBeCanonical() throws Exception {
        File sub = this.existingDirectory("sub");
        File a = new File(new File(new File(sub, ".."), "sub"), "a");
        this.ensureExists(a);
        Stream stream = this.fsa.streamFilesRecursive(sub.getParentFile());
        List filepaths = stream.map(FileHandle::getFile).collect(Collectors.toList());
        MatcherAssert.assertThat(filepaths, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{a.getCanonicalFile()}));
    }

    @Test
    void streamFilesRecursiveMustBeAbleToGivePathRelativeToBase() throws Exception {
        File sub = this.existingDirectory("sub");
        File a = this.existingFile("a");
        File b = new File(sub, "b");
        this.ensureExists(b);
        File base = a.getParentFile();
        Set set = this.fsa.streamFilesRecursive(base).map(FileHandle::getRelativeFile).collect(Collectors.toSet());
        MatcherAssert.assertThat((String)("Files relative to base directory " + base), set, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{new File("a"), new File("sub" + File.separator + "b")}));
    }

    @Test
    void streamFilesRecursiveMustListSingleFileGivenAsBase() throws Exception {
        this.existingDirectory("sub");
        this.existingFile("sub/x");
        File a = this.existingFile("a");
        Stream stream = this.fsa.streamFilesRecursive(a);
        List filepaths = stream.map(FileHandle::getFile).collect(Collectors.toList());
        MatcherAssert.assertThat(filepaths, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{a}));
    }

    @Test
    void streamFilesRecursiveListedSingleFileMustHaveCanonicalPath() throws Exception {
        File sub = this.existingDirectory("sub");
        this.existingFile("sub/x");
        File a = this.existingFile("a");
        File queryForA = new File(new File(sub, ".."), "a");
        Stream stream = this.fsa.streamFilesRecursive(queryForA);
        List filepaths = stream.map(FileHandle::getFile).collect(Collectors.toList());
        MatcherAssert.assertThat(filepaths, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{a.getCanonicalFile()}));
    }

    @Test
    void streamFilesRecursiveMustReturnEmptyStreamForNonExistingBasePath() throws Exception {
        File nonExisting = new File("nonExisting");
        Assertions.assertFalse((boolean)this.fsa.streamFilesRecursive(nonExisting).anyMatch(Predicates.alwaysTrue()));
    }

    @Test
    void streamFilesRecursiveMustRenameFiles() throws Exception {
        File a = this.existingFile("a");
        File b = this.nonExistingFile("b");
        File base = a.getParentFile();
        this.fsa.streamFilesRecursive(base).forEach(FileHandle.handleRename((File)b));
        List filepaths = this.fsa.streamFilesRecursive(base).map(FileHandle::getFile).collect(Collectors.toList());
        MatcherAssert.assertThat(filepaths, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{b.getCanonicalFile()}));
    }

    @Test
    void streamFilesRecursiveMustDeleteFiles() throws Exception {
        File a = this.existingFile("a");
        File b = this.existingFile("b");
        File c = this.existingFile("c");
        File base = a.getParentFile();
        this.fsa.streamFilesRecursive(base).forEach(FileHandle.HANDLE_DELETE);
        Assertions.assertFalse((boolean)this.fsa.fileExists(a));
        Assertions.assertFalse((boolean)this.fsa.fileExists(b));
        Assertions.assertFalse((boolean)this.fsa.fileExists(c));
    }

    @Test
    void streamFilesRecursiveMustThrowWhenDeletingNonExistingFile() throws Exception {
        File a = this.existingFile("a");
        FileHandle handle = (FileHandle)this.fsa.streamFilesRecursive(a).findAny().get();
        this.fsa.deleteFile(a);
        Assertions.assertThrows(NoSuchFileException.class, () -> ((FileHandle)handle).delete());
    }

    @Test
    void streamFilesRecursiveMustThrowWhenTargetFileOfRenameAlreadyExists() throws Exception {
        File a = this.existingFile("a");
        File b = this.existingFile("b");
        FileHandle handle = (FileHandle)this.fsa.streamFilesRecursive(a).findAny().get();
        Assertions.assertThrows(FileAlreadyExistsException.class, () -> handle.rename(b, new CopyOption[0]));
    }

    @Test
    void streamFilesRecursiveMustNotThrowWhenTargetFileOfRenameAlreadyExistsAndUsingReplaceExisting() throws Exception {
        File a = this.existingFile("a");
        File b = this.existingFile("b");
        FileHandle handle = (FileHandle)this.fsa.streamFilesRecursive(a).findAny().get();
        handle.rename(b, new CopyOption[]{StandardCopyOption.REPLACE_EXISTING});
    }

    @Test
    void streamFilesRecursiveMustDeleteSubDirectoriesEmptiedByFileRename() throws Exception {
        File sub = this.existingDirectory("sub");
        File x = new File(sub, "x");
        this.ensureExists(x);
        File target = this.nonExistingFile("target");
        this.fsa.streamFilesRecursive(sub).forEach(FileHandle.handleRename((File)target));
        Assertions.assertFalse((boolean)this.fsa.isDirectory(sub));
        Assertions.assertFalse((boolean)this.fsa.fileExists(sub));
    }

    @Test
    void streamFilesRecursiveMustDeleteMultipleLayersOfSubDirectoriesIfTheyBecomeEmptyByRename() throws Exception {
        File sub = this.existingDirectory("sub");
        File subsub = new File(sub, "subsub");
        this.ensureDirectoryExists(subsub);
        File x = new File(subsub, "x");
        this.ensureExists(x);
        File target = this.nonExistingFile("target");
        this.fsa.streamFilesRecursive(sub).forEach(FileHandle.handleRename((File)target));
        Assertions.assertFalse((boolean)this.fsa.isDirectory(subsub));
        Assertions.assertFalse((boolean)this.fsa.fileExists(subsub));
        Assertions.assertFalse((boolean)this.fsa.isDirectory(sub));
        Assertions.assertFalse((boolean)this.fsa.fileExists(sub));
    }

    @Test
    void streamFilesRecursiveMustNotDeleteDirectoriesAboveBaseDirectoryIfTheyBecomeEmptyByRename() throws Exception {
        File sub = this.existingDirectory("sub");
        File subsub = new File(sub, "subsub");
        File subsubsub = new File(subsub, "subsubsub");
        this.ensureDirectoryExists(subsub);
        this.ensureDirectoryExists(subsubsub);
        File x = new File(subsubsub, "x");
        this.ensureExists(x);
        File target = this.nonExistingFile("target");
        this.fsa.streamFilesRecursive(subsub).forEach(FileHandle.handleRename((File)target));
        Assertions.assertFalse((boolean)this.fsa.fileExists(subsubsub));
        Assertions.assertFalse((boolean)this.fsa.isDirectory(subsubsub));
        Assertions.assertFalse((boolean)this.fsa.fileExists(subsub));
        Assertions.assertFalse((boolean)this.fsa.isDirectory(subsub));
        Assertions.assertTrue((boolean)this.fsa.fileExists(sub));
        Assertions.assertTrue((boolean)this.fsa.isDirectory(sub));
    }

    @Test
    void streamFilesRecursiveMustDeleteSubDirectoriesEmptiedByFileDelete() throws Exception {
        File sub = this.existingDirectory("sub");
        File x = new File(sub, "x");
        this.ensureExists(x);
        this.fsa.streamFilesRecursive(sub).forEach(FileHandle.HANDLE_DELETE);
        Assertions.assertFalse((boolean)this.fsa.isDirectory(sub));
        Assertions.assertFalse((boolean)this.fsa.fileExists(sub));
    }

    @Test
    void streamFilesRecursiveMustDeleteMultipleLayersOfSubDirectoriesIfTheyBecomeEmptyByDelete() throws Exception {
        File sub = this.existingDirectory("sub");
        File subsub = new File(sub, "subsub");
        this.ensureDirectoryExists(subsub);
        File x = new File(subsub, "x");
        this.ensureExists(x);
        this.fsa.streamFilesRecursive(sub).forEach(FileHandle.HANDLE_DELETE);
        Assertions.assertFalse((boolean)this.fsa.isDirectory(subsub));
        Assertions.assertFalse((boolean)this.fsa.fileExists(subsub));
        Assertions.assertFalse((boolean)this.fsa.isDirectory(sub));
        Assertions.assertFalse((boolean)this.fsa.fileExists(sub));
    }

    @Test
    void streamFilesRecursiveMustNotDeleteDirectoriesAboveBaseDirectoryIfTheyBecomeEmptyByDelete() throws Exception {
        File sub = this.existingDirectory("sub");
        File subsub = new File(sub, "subsub");
        File subsubsub = new File(subsub, "subsubsub");
        this.ensureDirectoryExists(subsub);
        this.ensureDirectoryExists(subsubsub);
        File x = new File(subsubsub, "x");
        this.ensureExists(x);
        this.fsa.streamFilesRecursive(subsub).forEach(FileHandle.HANDLE_DELETE);
        Assertions.assertFalse((boolean)this.fsa.fileExists(subsubsub));
        Assertions.assertFalse((boolean)this.fsa.isDirectory(subsubsub));
        Assertions.assertFalse((boolean)this.fsa.fileExists(subsub));
        Assertions.assertFalse((boolean)this.fsa.isDirectory(subsub));
        Assertions.assertTrue((boolean)this.fsa.fileExists(sub));
        Assertions.assertTrue((boolean)this.fsa.isDirectory(sub));
    }

    @Test
    void streamFilesRecursiveMustCreateMissingPathDirectoriesImpliedByFileRename() throws Exception {
        File a = this.existingFile("a");
        File sub = new File(this.path, "sub");
        File target = new File(sub, "b");
        FileHandle handle = (FileHandle)this.fsa.streamFilesRecursive(a).findAny().get();
        handle.rename(target, new CopyOption[0]);
        Assertions.assertTrue((boolean)this.fsa.isDirectory(sub));
        Assertions.assertTrue((boolean)this.fsa.fileExists(target));
    }

    @Test
    void streamFilesRecursiveMustNotSeeFilesLaterCreatedBaseDirectory() throws Exception {
        File a = this.existingFile("a");
        Stream stream = this.fsa.streamFilesRecursive(a.getParentFile());
        File b = this.existingFile("b");
        Set files = stream.map(FileHandle::getFile).collect(Collectors.toSet());
        MatcherAssert.assertThat(files, (Matcher)Matchers.contains((Object[])new File[]{a}));
        MatcherAssert.assertThat(files, (Matcher)Matchers.not((Matcher)Matchers.contains((Object[])new File[]{b})));
    }

    @Test
    void streamFilesRecursiveMustNotSeeFilesRenamedIntoBaseDirectory() throws Exception {
        File a = this.existingFile("a");
        File sub = this.existingDirectory("sub");
        File x = new File(sub, "x");
        this.ensureExists(x);
        File target = this.nonExistingFile("target");
        HashSet observedFiles = new HashSet();
        this.fsa.streamFilesRecursive(a.getParentFile()).forEach(fh -> {
            File file = fh.getFile();
            observedFiles.add(file);
            if (file.equals(x)) {
                FileHandle.handleRename((File)target).accept(fh);
            }
        });
        MatcherAssert.assertThat(observedFiles, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{a, x}));
    }

    @Test
    void streamFilesRecursiveMustNotSeeFilesRenamedIntoSubDirectory() throws Exception {
        File a = this.existingFile("a");
        File sub = this.existingDirectory("sub");
        File target = new File(sub, "target");
        HashSet observedFiles = new HashSet();
        this.fsa.streamFilesRecursive(a.getParentFile()).forEach(fh -> {
            File file = fh.getFile();
            observedFiles.add(file);
            if (file.equals(a)) {
                FileHandle.handleRename((File)target).accept(fh);
            }
        });
        MatcherAssert.assertThat(observedFiles, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{a}));
    }

    @Test
    void streamFilesRecursiveRenameMustCanonicaliseSourceFile() throws Exception {
        File a = new File(new File(this.existingFile("a"), "poke"), "..");
        File b = this.nonExistingFile("b");
        FileHandle handle = (FileHandle)this.fsa.streamFilesRecursive(a).findAny().get();
        handle.rename(b, new CopyOption[0]);
    }

    @Test
    void streamFilesRecursiveRenameMustCanonicaliseTargetFile() throws Exception {
        File a = this.existingFile("a");
        File b = new File(new File(new File(this.path, "b"), "poke"), "..");
        FileHandle handle = (FileHandle)this.fsa.streamFilesRecursive(a).findAny().get();
        handle.rename(b, new CopyOption[0]);
    }

    @Test
    void streamFilesRecursiveRenameTargetFileMustBeRenamed() throws Exception {
        File a = this.existingFile("a");
        File b = this.nonExistingFile("b");
        FileHandle handle = (FileHandle)this.fsa.streamFilesRecursive(a).findAny().get();
        handle.rename(b, new CopyOption[0]);
        Assertions.assertTrue((boolean)this.fsa.fileExists(b));
    }

    @Test
    void streamFilesRecursiveSourceFileMustNotBeMappableAfterRename() throws Exception {
        File a = this.existingFile("a");
        File b = this.nonExistingFile("b");
        FileHandle handle = (FileHandle)this.fsa.streamFilesRecursive(a).findAny().get();
        handle.rename(b, new CopyOption[0]);
        Assertions.assertFalse((boolean)this.fsa.fileExists(a));
    }

    @Test
    void streamFilesRecursiveRenameMustNotChangeSourceFileContents() throws Exception {
        File a = this.existingFile("a");
        File b = this.nonExistingFile("b");
        this.generateFileWithRecords(a, this.recordCount);
        FileHandle handle = (FileHandle)this.fsa.streamFilesRecursive(a).findAny().get();
        handle.rename(b, new CopyOption[0]);
        this.verifyRecordsInFile(b, this.recordCount);
    }

    @Test
    void streamFilesRecursiveRenameMustNotChangeSourceFileContentsWithReplaceExisting() throws Exception {
        File a = this.existingFile("a");
        File b = this.existingFile("b");
        this.generateFileWithRecords(a, this.recordCount);
        this.generateFileWithRecords(b, this.recordCount + this.recordsPerFilePage);
        try (StoreChannel channel = this.fsa.open(b, OpenMode.READ_WRITE);){
            ThreadLocalRandom rng = ThreadLocalRandom.current();
            int fileSize = (int)channel.size();
            ByteBuffer buffer = ByteBuffer.allocate(fileSize);
            for (int i = 0; i < fileSize; ++i) {
                buffer.put(i, (byte)rng.nextInt());
            }
            buffer.rewind();
            channel.writeAll(buffer);
        }
        FileHandle handle = (FileHandle)this.fsa.streamFilesRecursive(a).findAny().get();
        handle.rename(b, new CopyOption[]{StandardCopyOption.REPLACE_EXISTING});
        this.verifyRecordsInFile(b, this.recordCount);
    }

    @Test
    void lastModifiedOfNonExistingFileIsZero() {
        MatcherAssert.assertThat((Object)this.fsa.lastModifiedTime(this.nonExistingFile("blabla")), (Matcher)Is.is((Object)0L));
    }

    @Test
    void shouldHandlePathThatLooksVeryDifferentWhenCanonicalized() throws Exception {
        File dir = this.existingDirectory("/././home/.././././home/././.././././././././././././././././././home/././");
        File a = this.existingFile("/home/a");
        List filepaths = this.fsa.streamFilesRecursive(dir).map(FileHandle::getRelativeFile).collect(Collectors.toList());
        MatcherAssert.assertThat(filepaths, (Matcher)Matchers.containsInAnyOrder((Object[])new File[]{new File(a.getName())}));
    }

    private void generateFileWithRecords(File file, int recordCount) throws IOException {
        try (StoreChannel channel = this.fsa.open(file, OpenMode.READ_WRITE);){
            ByteBuffer buf = ByteBuffer.allocate(this.recordSize);
            for (int i = 0; i < recordCount; ++i) {
                FileSystemAbstractionTest.generateRecordForId(i, buf);
                int rem = buf.remaining();
                while ((rem -= channel.write(buf)) > 0) {
                }
            }
        }
    }

    private void verifyRecordsInFile(File file, int recordCount) throws IOException {
        try (StoreChannel channel = this.fsa.open(file, OpenMode.READ);){
            ByteBuffer buf = ByteBuffer.allocate(this.recordSize);
            ByteBuffer observation = ByteBuffer.allocate(this.recordSize);
            for (int i = 0; i < recordCount; ++i) {
                FileSystemAbstractionTest.generateRecordForId(i, buf);
                observation.position(0);
                channel.read(observation);
                this.assertRecord(i, observation, buf);
            }
        }
    }

    private void assertRecord(long pageId, ByteBuffer actualPageContents, ByteBuffer expectedPageContents) {
        byte[] actualBytes = actualPageContents.array();
        byte[] expectedBytes = expectedPageContents.array();
        int estimatedPageId = this.estimateId(actualBytes);
        MatcherAssert.assertThat((String)("Page id: " + pageId + " (based on record data, it should have been " + estimatedPageId + ", a difference of " + Math.abs(pageId - (long)estimatedPageId) + ")"), (Object)actualBytes, (Matcher)ByteArrayMatcher.byteArray((byte[])expectedBytes));
    }

    private int estimateId(byte[] record) {
        return ByteBuffer.wrap(record).getInt() - 1;
    }

    private static void generateRecordForId(long id, ByteBuffer buf) {
        buf.position(0);
        int x = (int)(id + 1L);
        buf.putInt(x);
        while (buf.position() < buf.limit()) {
            buf.put((byte)(++x & 0xFF));
        }
        buf.position(0);
    }

    private File existingFile(String fileName) throws IOException {
        File file = new File(this.path, fileName);
        this.fsa.mkdirs(this.path);
        this.fsa.create(file).close();
        return file;
    }

    private File nonExistingFile(String fileName) {
        return new File(this.path, fileName);
    }

    private File existingDirectory(String dir) throws IOException {
        File directory = new File(this.path, dir);
        this.fsa.mkdirs(directory);
        return directory;
    }

    private void ensureExists(File file) throws IOException {
        this.fsa.mkdirs(file.getParentFile());
        this.fsa.create(file).close();
    }

    private void ensureDirectoryExists(File directory) throws IOException {
        this.fsa.mkdirs(directory);
    }

    private void writeIntegerIntoFile(File targetFile) throws IOException {
        StoreChannel storeChannel = this.fsa.create(targetFile);
        ByteBuffer byteBuffer = ByteBuffer.allocate(32).putInt(7);
        byteBuffer.flip();
        storeChannel.writeAll(byteBuffer);
        storeChannel.close();
    }
}

