/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.commandline.dbms;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Collection;
import org.neo4j.cli.AbstractAdminCommand;
import org.neo4j.cli.CommandFailedException;
import org.neo4j.cli.Converters;
import org.neo4j.cli.ExecutionContext;
import org.neo4j.configuration.BootloaderSettings;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.ConfigUtils;
import org.neo4j.configuration.GraphDatabaseInternalSettings;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.dbms.MemoryRecommendation;
import org.neo4j.graphdb.config.Configuration;
import org.neo4j.graphdb.config.Setting;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.layout.Neo4jLayout;
import org.neo4j.io.os.OsBeanUtil;
import org.neo4j.kernel.api.index.IndexDirectoryStructure;
import picocli.CommandLine;

@CommandLine.Command(name="memory-recommendation", header={"Print Neo4j heap and pagecache memory settings recommendations."}, description={"Print heuristic memory setting recommendations for the Neo4j JVM heap and pagecache. The heuristic is based on the total memory of the system the command is running on, or on the amount of memory specified with the --memory argument. The heuristic assumes that the system is dedicated to running Neo4j. If this is not the case, then use the --memory argument to specify how much memory can be expected to be dedicated to Neo4j. The output is formatted such that it can be copy-pasted into the neo4j.conf file."})
public class MemoryRecommendationsCommand
extends AbstractAdminCommand {
    @CommandLine.Option(names={"--memory"}, paramLabel="<size>", converter={Converters.ByteUnitConverter.class}, description={"Recommend memory settings with respect to the given amount of memory, instead of the total memory of the system running the command. Valid units are: k, K, m, M, g, G."})
    private Long memory;
    @CommandLine.Option(names={"--docker"}, fallbackValue="true", description={"The recommended memory settings are produced in the form of environment variables that can be directly passed to a Neo4j docker container. The recommended use is to save the generated environment variables to a file and pass the file to a docker container using the '--env-file' docker option."})
    private boolean dockerOutput;

    public MemoryRecommendationsCommand(ExecutionContext ctx) {
        super(ctx);
    }

    protected void execute() throws IOException {
        if (this.memory == null) {
            this.memory = OsBeanUtil.getTotalPhysicalMemory();
        }
        Path configFile = this.ctx.confDir().resolve("neo4j.conf");
        Config config = this.getConfig(configFile);
        long offHeapMemory = MemoryRecommendation.recommendTxStateMemory(config, this.memory);
        String os = MemoryRecommendation.bytesToString(MemoryRecommendation.recommendOsMemory(this.memory));
        String heap = MemoryRecommendation.bytesToString(MemoryRecommendation.recommendHeapMemory(this.memory));
        String pageCache = MemoryRecommendation.bytesToString(MemoryRecommendation.recommendPageCacheMemory(this.memory, offHeapMemory));
        String txState = MemoryRecommendation.bytesToString(offHeapMemory);
        Path databasesRoot = (Path)config.get(GraphDatabaseInternalSettings.databases_root_path);
        Neo4jLayout storeLayout = Neo4jLayout.of((Configuration)config);
        Collection layouts = storeLayout.databaseLayouts();
        long pageCacheSize = this.pageCacheSize(layouts);
        long luceneSize = this.luceneSize(layouts);
        this.print("# Memory settings recommendation:");
        this.print("#");
        this.print("# Assuming the system is dedicated to running Neo4j and has " + ByteUnit.bytesToString((long)this.memory) + " of memory,");
        this.print("# we recommend a heap size of around " + heap + ", and a page cache of around " + pageCache + ",");
        this.print("# and that about " + os + " is left for the operating system, and the native memory");
        this.print("# needed by Lucene and Netty.");
        this.print("#");
        this.print("# Tip: If the indexing storage use is high, e.g. there are many indexes or most");
        this.print("# data indexed, then it might advantageous to leave more memory for the");
        this.print("# operating system.");
        this.print("#");
        this.print("# Tip: The more concurrent transactions your workload has and the more updates");
        this.print("# they do, the more heap memory you will need. However, don't allocate more");
        this.print("# than 31g of heap, since this will disable pointer compression, also known as");
        this.print("# \"compressed oops\", in the JVM and make less effective use of the heap.");
        this.print("#");
        this.print("# Tip: Setting the initial and the max heap size to the same value means the");
        this.print("# JVM will never need to change the heap size. Changing the heap size otherwise");
        this.print("# involves a full GC, which is desirable to avoid.");
        this.print("#");
        this.print("# Based on the above, the following memory settings are recommended:");
        this.printSetting(BootloaderSettings.initial_heap_size, heap);
        this.printSetting(BootloaderSettings.max_heap_size, heap);
        this.printSetting(GraphDatabaseSettings.pagecache_memory, pageCache);
        if (offHeapMemory != 0L) {
            this.printSetting(GraphDatabaseSettings.tx_state_max_off_heap_memory, txState);
        }
        this.print("#");
        this.print("# It is also recommended turning out-of-memory errors into full crashes,");
        this.print("# instead of allowing a partially crashed database to continue running:");
        this.printSetting(BootloaderSettings.additional_jvm, "-XX:+ExitOnOutOfMemoryError");
        this.print("#");
        this.print("# The numbers below have been derived based on your current databases located at: '" + databasesRoot + "'.");
        this.print("# They can be used as an input into more detailed memory analysis.");
        this.print("# Total size of lucene indexes in all databases: " + MemoryRecommendation.bytesToString(luceneSize));
        this.print("# Total size of data and native indexes in all databases: " + MemoryRecommendation.bytesToString(pageCacheSize));
    }

    private void printSetting(Setting<?> setting, String value) {
        if (!this.dockerOutput) {
            this.print(setting.name() + "=" + value);
        } else {
            String nameWithFixedUnderscores = setting.name().replaceAll("_", "__");
            String nameWithFixedUnderscoresAndDots = nameWithFixedUnderscores.replace('.', '_');
            this.print("NEO4J_" + nameWithFixedUnderscoresAndDots + "='" + value + "'");
        }
    }

    private long pageCacheSize(Collection<DatabaseLayout> layouts) throws IOException {
        long sum = 0L;
        for (DatabaseLayout layout : layouts) {
            sum += this.getDatabasePageCacheSize(layout, this.ctx.fs());
        }
        return sum;
    }

    private long getDatabasePageCacheSize(DatabaseLayout layout, FileSystemAbstraction fs) throws IOException {
        return MemoryRecommendation.sumStoreFiles(layout, fs) + MemoryRecommendation.sumIndexFiles(IndexDirectoryStructure.baseSchemaIndexFolder((Path)layout.databaseDirectory()), MemoryRecommendation.getNativeIndexFileFilter(layout.databaseDirectory(), false, fs), fs);
    }

    private long luceneSize(Collection<DatabaseLayout> layouts) throws IOException {
        long sum = 0L;
        for (DatabaseLayout layout : layouts) {
            sum += this.getDatabaseLuceneSize(layout, this.ctx.fs());
        }
        return sum;
    }

    private long getDatabaseLuceneSize(DatabaseLayout databaseLayout, FileSystemAbstraction fs) throws IOException {
        Path databaseDirectory = databaseLayout.databaseDirectory();
        return MemoryRecommendation.sumIndexFiles(IndexDirectoryStructure.baseSchemaIndexFolder((Path)databaseDirectory), MemoryRecommendation.getNativeIndexFileFilter(databaseDirectory, true, fs), fs);
    }

    private Config getConfig(Path configFile) {
        if (!this.ctx.fs().fileExists(configFile)) {
            throw new CommandFailedException("Unable to find config file, tried: " + configFile.toAbsolutePath());
        }
        Config config = Config.newBuilder().fromFile(configFile).set(GraphDatabaseSettings.neo4j_home, (Object)this.ctx.homeDir().toAbsolutePath()).commandExpansion(this.allowCommandExpansion).build();
        ConfigUtils.disableAllConnectors((Config)config);
        return config;
    }

    private void print(String text) {
        this.ctx.out().println(text);
    }
}

