/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.commandline.dbms;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.StringJoiner;
import java.util.stream.Collectors;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.neo4j.cli.AbstractAdminCommand;
import org.neo4j.cli.CommandFailedException;
import org.neo4j.cli.Converters;
import org.neo4j.cli.ExecutionContext;
import org.neo4j.cloud.storage.SchemeFileSystemAbstraction;
import org.neo4j.commandline.Util;
import org.neo4j.commandline.dbms.LoadDumpExecutor;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.configuration.helpers.DatabaseNamePattern;
import org.neo4j.dbms.archive.DumpFormatSelector;
import org.neo4j.dbms.archive.Loader;
import org.neo4j.dbms.archive.backup.BackupDescription;
import org.neo4j.dbms.archive.backup.BackupFormatSelector;
import org.neo4j.function.ThrowingSupplier;
import org.neo4j.internal.helpers.Exceptions;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.database.NormalizedDatabaseName;
import org.neo4j.logging.InternalLogProvider;
import org.neo4j.logging.log4j.Log4jLogProvider;
import picocli.CommandLine;

@CommandLine.Command(name="load", header={"Load a database from an archive created with the dump command or from full Neo4j Enterprise backup."}, description={"Load a database from an archive. <archive-path> must be a directory containing an archive(s). Archive can be a database dump created with the dump command, or can be a full backup artifact created by the backup command from Neo4j Enterprise. If neither --from-path or --from-stdin is supplied `server.directories.dumps.root` setting will be searched for the archive. Existing databases can be replaced by specifying --overwrite-destination. It is not possible to replace a database that is mounted in a running Neo4j server. If --info is specified, then the database is not loaded, but information (i.e. file count, byte count, and format of load file) about the archive is printed instead."})
public class LoadCommand
extends AbstractAdminCommand {
    @CommandLine.Parameters(arity="1", description={"Name of the database to load. Can contain * and ? for globbing. Note that * and ? have special meaning in some shells and might need to be escaped or used with quotes."}, converter={Converters.DatabaseNamePatternConverter.class})
    private DatabaseNamePattern database;
    @CommandLine.ArgGroup
    private SourceOption source = new SourceOption();
    @CommandLine.Option(names={"--overwrite-destination"}, arity="0..1", paramLabel="true|false", fallbackValue="true", showDefaultValue=CommandLine.Help.Visibility.ALWAYS, description={"If an existing database should be replaced."})
    private boolean force;
    @CommandLine.Option(names={"--info"}, fallbackValue="true", description={"Print meta-data information about the archive file, instead of loading the contained database."})
    private boolean info;
    static final String SYSTEM_ERR_MESSAGE = "WARNING! You are loading a dump of Neo4j's internal system database.%nThis system database dump may contain unwanted metadata for the DBMS it was taken from;%nLoading it should only be done after consulting the Neo4j Operations Manual.%n";
    public static final String FULL_BACKUP_DESCRIPTION = "Neo4j Full Backup";
    public static final String DIFFERENTIAL_BACKUP_DESCRIPTION = "Neo4j Differential Backup";
    public static final String ZSTD_DUMP_DESCRIPTION = "Neo4j ZSTD Dump.";
    public static final String GZIP_DUMP_DESCRIPTION = "TAR+GZIP.";
    public static final String UNKNOWN_COUNT = "?";

    public LoadCommand(ExecutionContext ctx) {
        super(ctx);
    }

    protected Optional<String> commandConfigName() {
        return Optional.of("database-load");
    }

    protected Loader createLoader(FileSystemAbstraction fs) {
        return new Loader(fs, this.ctx.err());
    }

    public void execute() {
        Config config = this.buildConfig();
        try (Log4jLogProvider logProvider = Util.configuredLogProvider((OutputStream)this.ctx.out(), (boolean)this.verbose);
             SchemeFileSystemAbstraction fs = new SchemeFileSystemAbstraction(this.ctx.fs(), config, (InternalLogProvider)logProvider);){
            Path sourcePath = null;
            if (this.source.path != null && !fs.isDirectory(sourcePath = this.normalizeAndValidateIfStoragePathDirectory(fs.resolve(this.source.path)))) {
                throw new CommandFailedException(this.source.path + " is not an existing directory");
            }
            if (this.database.containsPattern() && this.source.stdIn) {
                throw new CommandFailedException("Globbing in database name can not be used in combination with standard input. Specify a directory as source or a single target database");
            }
            if (sourcePath == null && !this.source.stdIn) {
                Path defaultDumpsPath = (Path)config.get(GraphDatabaseSettings.database_dumps_root_path);
                if (!fs.isDirectory(defaultDumpsPath)) {
                    throw new CommandFailedException("The root location for storing dumps ('" + GraphDatabaseSettings.database_dumps_root_path.name() + "'=" + String.valueOf(defaultDumpsPath) + ") doesn't contain any dumps yet. Specify another directory with --from-path.");
                }
                sourcePath = fs.resolve(defaultDumpsPath.toString());
            }
            if (this.info) {
                this.inspectDump((FileSystemAbstraction)fs, sourcePath);
            } else {
                this.loadDump((FileSystemAbstraction)fs, sourcePath, config);
            }
        }
        catch (IOException e) {
            Util.wrapIOException((IOException)e);
        }
    }

    private void inspectDump(FileSystemAbstraction fs, Path sourcePath) {
        Set<DumpInfo> dbNames = this.getDbNames(fs, sourcePath, true);
        Loader loader = this.createLoader(fs);
        ArrayList<FailedLoad> failedLoads = new ArrayList<FailedLoad>();
        for (DumpInfo dumpInfo : dbNames) {
            if (dumpInfo.stdIn) {
                this.inspectOne(dumpInfo.dbName, (ThrowingSupplier<InputStream, IOException>)((ThrowingSupplier)() -> ((ExecutionContext)this.ctx).in()), loader, failedLoads, "reading from stdin");
                continue;
            }
            for (Path path : dumpInfo.archives) {
                this.inspectOne(dumpInfo.dbName, LoadCommand.streamSupplierFor(fs, path), loader, failedLoads, path.toString());
            }
        }
        this.checkFailure(failedLoads, "Print metadata failed for databases: '");
    }

    private void inspectOne(String dbName, ThrowingSupplier<InputStream, IOException> archiveInputStreamSupplier, Loader loader, List<FailedLoad> failedLoads, String streamDescription) {
        try {
            MutableBoolean backup = new MutableBoolean(false);
            MutableBoolean fullBackup = new MutableBoolean(false);
            Loader.DumpMetaData metaData = loader.getMetaData(archiveInputStreamSupplier, streamSupplier -> DumpFormatSelector.decompressWithBackupSupport((ThrowingSupplier<InputStream, IOException>)streamSupplier, bd -> {
                backup.setTrue();
                fullBackup.setValue(bd.isFull());
            }));
            String archiveFormat = this.getArchiveFormat(backup.booleanValue(), fullBackup.booleanValue(), metaData.compressed());
            Loader.SizeMeta sizeMeta = metaData.sizeMeta();
            this.printArchiveInfo(dbName, archiveFormat, sizeMeta);
        }
        catch (Exception e) {
            this.ctx.err().printf("Failed to get metadata for archive '%s': %s%n", streamDescription, e.getMessage());
            failedLoads.add(new FailedLoad(dbName, e));
        }
    }

    private void printArchiveInfo(String dbName, String archiveFormat, Loader.SizeMeta sizeMeta) {
        this.ctx.out().println("Database: " + dbName);
        this.ctx.out().println("Format: " + archiveFormat);
        this.ctx.out().println("Files: " + String.valueOf(sizeMeta != null ? Long.valueOf(sizeMeta.files()) : UNKNOWN_COUNT));
        this.ctx.out().println("Bytes: " + String.valueOf(sizeMeta != null ? Long.valueOf(sizeMeta.bytes()) : UNKNOWN_COUNT));
        this.ctx.out().println();
    }

    private String getArchiveFormat(boolean backup, boolean fullBackup, boolean compressed) {
        if (backup) {
            if (fullBackup) {
                return FULL_BACKUP_DESCRIPTION;
            }
            return DIFFERENTIAL_BACKUP_DESCRIPTION;
        }
        if (compressed) {
            return ZSTD_DUMP_DESCRIPTION;
        }
        return GZIP_DUMP_DESCRIPTION;
    }

    private void loadDump(FileSystemAbstraction fs, Path sourcePath, Config config) throws IOException {
        Set<DumpInfo> dbNames = this.getDbNames(fs, sourcePath, false);
        this.loadDump(dbNames, config, fs);
    }

    protected void loadDump(Set<DumpInfo> dbNames, Config config, FileSystemAbstraction fs) throws IOException {
        LoadDumpExecutor loadDumpExecutor = new LoadDumpExecutor(config, fs, this.ctx.err(), this.ctx.out(), this.createLoader(fs), LoadCommand::decompress);
        ArrayList<FailedLoad> failedLoads = new ArrayList<FailedLoad>();
        for (DumpInfo dbName : dbNames) {
            try {
                String dumpInputDescription;
                if (dbName.dbName.equals("system")) {
                    this.ctx.err().printf(SYSTEM_ERR_MESSAGE, new Object[0]);
                }
                Path dumpPath = null;
                if (!dbName.stdIn) {
                    if (dbName.archives.size() > 1) {
                        throw new CommandFailedException("Multiple archives match:\n" + dbName.archives.stream().map(Path::toString).collect(Collectors.joining("\n")) + "\nRemove ambiguity by leaving only one of the above, or use --from-stdin option and pipe desired archive.");
                    }
                    if (dbName.archives.isEmpty()) {
                        throw new CommandFailedException("No matching archives found");
                    }
                    dumpPath = dbName.archives.getFirst();
                    if (!fs.fileExists(dumpPath)) {
                        throw new CommandFailedException("Archive does not exist: " + String.valueOf(dumpPath));
                    }
                }
                String string = dumpInputDescription = dbName.stdIn ? "reading from stdin" : dumpPath.toString();
                ThrowingSupplier dumpInputStreamSupplier = dbName.stdIn ? () -> ((ExecutionContext)this.ctx).in() : LoadCommand.streamSupplierFor(fs, dumpPath);
                loadDumpExecutor.execute(new LoadDumpExecutor.DumpInput((ThrowingSupplier<InputStream, IOException>)dumpInputStreamSupplier, dumpInputDescription), dbName.dbName, this.force);
            }
            catch (Exception e) {
                this.ctx.err().printf("Failed to load database '%s': %s%n", dbName.dbName, e.getMessage());
                failedLoads.add(new FailedLoad(dbName.dbName, e));
            }
        }
        this.checkFailure(failedLoads, "Load failed for databases: '");
    }

    private static ThrowingSupplier<InputStream, IOException> streamSupplierFor(FileSystemAbstraction fs, Path dumpPath) {
        return () -> fs.openAsInputStream(dumpPath);
    }

    private void checkFailure(List<FailedLoad> failedLoads, String prefix) {
        if (!failedLoads.isEmpty()) {
            StringJoiner failedDbs = new StringJoiner("', '", prefix, "'");
            Exception exceptions = null;
            for (FailedLoad failedLoad : failedLoads) {
                failedDbs.add(failedLoad.dbName);
                exceptions = (Exception)Exceptions.chain(exceptions, (Throwable)failedLoad.e);
            }
            this.ctx.err().println(failedDbs);
            throw new CommandFailedException(failedDbs.toString(), exceptions);
        }
    }

    private Set<DumpInfo> getDbNames(FileSystemAbstraction fs, Path sourcePath, boolean includeDiff) {
        if (this.source.stdIn) {
            return Set.of(new DumpInfo(this.database.getDatabaseName(), true, Collections.emptyList()));
        }
        Map<String, List<Path>> dbsToArchives = this.listArchivesMatching(fs, sourcePath, this.database, includeDiff);
        if (!this.database.containsPattern()) {
            List<Path> archives = dbsToArchives.getOrDefault(this.database.getNormalizedDatabaseName(), Collections.emptyList());
            return Set.of(new DumpInfo(this.database.getDatabaseName(), false, archives));
        }
        Set<DumpInfo> dbNames = dbsToArchives.entrySet().stream().map(DumpInfo::new).collect(Collectors.toSet());
        if (dbNames.isEmpty()) {
            throw new CommandFailedException("Pattern '" + this.database.getDatabaseName() + "' did not match any archive file in " + String.valueOf(sourcePath));
        }
        return dbNames;
    }

    private Map<String, List<Path>> listArchivesMatching(FileSystemAbstraction fs, Path sourcePath, DatabaseNamePattern pattern, boolean includeDiff) {
        try {
            HashMap<String, List<Path>> result = new HashMap<String, List<Path>>();
            for (Path path : fs.listFiles(sourcePath)) {
                String fileName = path.getFileName().toString();
                if (fs.isDirectory(path)) continue;
                if (fileName.endsWith(".dump")) {
                    String dbName = new NormalizedDatabaseName(fileName.substring(0, fileName.length() - ".dump".length())).name();
                    if (!pattern.matches(dbName)) continue;
                    result.computeIfAbsent(dbName, name -> new ArrayList()).add(path);
                    continue;
                }
                if (!fileName.endsWith(".backup")) continue;
                try (InputStream inputStream = fs.openAsInputStream(path);){
                    BackupDescription backupDescription = BackupFormatSelector.readDescription(inputStream);
                    String dbName = backupDescription.getDatabaseName();
                    if (!pattern.matches(dbName) || !includeDiff && !backupDescription.isFull()) continue;
                    result.computeIfAbsent(dbName, name -> new ArrayList()).add(path);
                }
            }
            return result;
        }
        catch (IOException e) {
            throw new CommandFailedException("Failed to list archive files", (Throwable)e);
        }
    }

    protected Config buildConfig() {
        return this.createPrefilledConfigBuilder().build();
    }

    private static InputStream decompress(ThrowingSupplier<InputStream, IOException> streamSupplier) throws IOException {
        return DumpFormatSelector.decompressWithBackupSupport(streamSupplier, bd -> {
            if (!bd.isFull()) {
                throw new CommandFailedException("Loading of differential Neo4j backup is not supported. Use restore database instead.");
            }
        });
    }

    private static class SourceOption {
        @CommandLine.Option(names={"--from-path"}, paramLabel="<path>", description={"Path to directory containing archive(s).\nIt is possible to load databases from AWS S3 buckets, Google Cloud storage buckets, and Azure bucket using the appropriate URI as the path."})
        private String path;
        @CommandLine.Option(names={"--from-stdin"}, description={"Read archive from standard input."})
        private boolean stdIn;

        private SourceOption() {
        }
    }

    protected record DumpInfo(String dbName, boolean stdIn, List<Path> archives) {
        public DumpInfo(Map.Entry<String, List<Path>> mapEntry) {
            this(mapEntry.getKey(), false, mapEntry.getValue());
        }
    }

    record FailedLoad(String dbName, Exception e) {
    }
}

