/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.bolt.protocol.common.connector.connection;

import io.netty.channel.Channel;
import java.time.Clock;
import java.util.ArrayList;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.neo4j.bolt.BoltServer;
import org.neo4j.bolt.protocol.common.BoltProtocol;
import org.neo4j.bolt.protocol.common.connection.Job;
import org.neo4j.bolt.protocol.common.connector.Connector;
import org.neo4j.bolt.protocol.common.connector.connection.AbstractConnection;
import org.neo4j.bolt.protocol.common.connector.connection.Connection;
import org.neo4j.bolt.protocol.common.connector.connection.listener.ConnectionListener;
import org.neo4j.bolt.protocol.common.fsm.StateMachine;
import org.neo4j.bolt.protocol.common.message.Error;
import org.neo4j.bolt.protocol.common.message.request.RequestMessage;
import org.neo4j.bolt.protocol.common.message.response.FailureMessage;
import org.neo4j.bolt.protocol.common.message.result.ResponseHandler;
import org.neo4j.bolt.protocol.common.signal.StateSignal;
import org.neo4j.bolt.runtime.BoltConnectionAuthFatality;
import org.neo4j.bolt.runtime.BoltConnectionFatality;
import org.neo4j.bolt.runtime.BoltProtocolBreachFatality;
import org.neo4j.exceptions.KernelException;
import org.neo4j.kernel.api.exceptions.Status;
import org.neo4j.logging.internal.LogService;
import org.neo4j.memory.HeapEstimator;
import org.neo4j.memory.LocalMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.util.FeatureToggles;

public class AtomicSchedulingConnection
extends AbstractConnection {
    private static final long SHALLOW_SIZE = HeapEstimator.shallowSizeOfInstance(AtomicSchedulingConnection.class);
    private static final int BATCH_SIZE = FeatureToggles.getInteger(BoltServer.class, (String)"max_batch_size", (int)100);
    private final ExecutorService executor;
    private final Clock clock;
    private final CompletableFuture<Void> closeFuture = new CompletableFuture();
    private final AtomicReference<State> state = new AtomicReference<State>(State.IDLE);
    private final LinkedBlockingDeque<Job> jobs = new LinkedBlockingDeque();
    private final AtomicInteger remainingInterrupts = new AtomicInteger();

    public AtomicSchedulingConnection(Connector connector, String id, Channel channel, long connectedAt, MemoryTracker memoryTracker, LogService logService, ExecutorService executor, Clock clock) {
        super(connector, id, channel, connectedAt, memoryTracker, logService);
        this.executor = executor;
        this.clock = clock;
    }

    @Override
    public boolean isIdling() {
        return this.state.get() == State.IDLE && !this.hasPendingJobs();
    }

    @Override
    public boolean hasPendingJobs() {
        return !this.jobs.isEmpty();
    }

    @Override
    public void submit(RequestMessage message, ResponseHandler responseHandler) {
        this.notifyListeners(listener -> listener.onRequestReceived(message));
        long queuedAt = this.clock.millis();
        this.submit(fsm -> {
            long processedForMillis;
            long processingStartedAt = this.clock.millis();
            long queuedForMillis = processingStartedAt - queuedAt;
            this.notifyListeners(listener -> listener.onRequestBeginProcessing(message, queuedForMillis));
            try {
                this.log.debug("[%s] Beginning execution of %s (queued for %d ms)", new Object[]{this.id, message, queuedForMillis});
                fsm.process(message, responseHandler);
                processedForMillis = this.clock.millis() - processingStartedAt;
            }
            catch (BoltConnectionFatality ex) {
                try {
                    this.notifyListeners(listener -> listener.onRequestFailedProcessing(message, ex));
                    throw ex;
                }
                catch (Throwable throwable) {
                    long processedForMillis2 = this.clock.millis() - processingStartedAt;
                    this.notifyListeners(listener -> listener.onRequestCompletedProcessing(message, processedForMillis2));
                    this.log.debug("[%s] Completed execution of %s (took %d ms)", new Object[]{this.id, message, processedForMillis2});
                    throw throwable;
                }
            }
            this.notifyListeners(listener -> listener.onRequestCompletedProcessing(message, processedForMillis2));
            this.log.debug("[%s] Completed execution of %s (took %d ms)", new Object[]{this.id, message, processedForMillis});
        });
    }

    @Override
    public void submit(Job job) {
        this.jobs.addLast(job);
        this.schedule(true);
    }

    private void schedule(boolean submissionHint) {
        if (!submissionHint && !this.hasPendingJobs()) {
            return;
        }
        if (this.state.compareAndSet(State.IDLE, State.SCHEDULED)) {
            this.log.debug("[%s] Scheduling connection for execution", new Object[]{this.id});
            this.notifyListeners(ConnectionListener::onScheduled);
            try {
                this.executor.submit(this::executeJobs);
            }
            catch (RejectedExecutionException ex) {
                Error error = Error.from((Status)Status.Request.NoThreadsAvailable, Status.Request.NoThreadsAvailable.code().description());
                String message = String.format("[%s] Unable to schedule for execution since there are no available threads to serve it at the moment. You can retry at a later time or consider increasing max thread pool size for bolt connector(s).", this.id);
                this.userLog.error(message);
                this.channel.writeAndFlush((Object)new FailureMessage(error.status(), error.message(), false));
                this.close();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void executeJobs() {
        Thread currentThread = Thread.currentThread();
        String originalThreadName = currentThread.getName();
        String customizedThreadName = String.format("%s [%s - %s]", originalThreadName, this.id, this.channel.remoteAddress());
        currentThread.setName(customizedThreadName);
        this.log.debug("[%s] Activating connection", new Object[]{this.id});
        this.notifyListeners(ConnectionListener::onActivated);
        try {
            this.doExecuteJobs();
        }
        catch (Throwable ex) {
            try {
                this.log.error("[" + this.id + "] Uncaught exception during job execution", ex);
                this.close();
            }
            catch (Throwable throwable) {
                this.notifyListeners(ConnectionListener::onIdle);
                this.log.debug("[%s] Returning to idle state", new Object[]{this.id});
                currentThread.setName(originalThreadName);
                State previousState = this.state.compareAndExchange(State.SCHEDULED, State.IDLE);
                switch (previousState) {
                    case SCHEDULED: {
                        this.schedule(false);
                        break;
                    }
                    case CLOSING: {
                        this.doClose();
                    }
                }
                throw throwable;
            }
            this.notifyListeners(ConnectionListener::onIdle);
            this.log.debug("[%s] Returning to idle state", new Object[]{this.id});
            currentThread.setName(originalThreadName);
            State previousState = this.state.compareAndExchange(State.SCHEDULED, State.IDLE);
            switch (previousState) {
                case SCHEDULED: {
                    this.schedule(false);
                    break;
                }
                case CLOSING: {
                    this.doClose();
                }
            }
        }
        this.notifyListeners(ConnectionListener::onIdle);
        this.log.debug("[%s] Returning to idle state", new Object[]{this.id});
        currentThread.setName(originalThreadName);
        State previousState = this.state.compareAndExchange(State.SCHEDULED, State.IDLE);
        switch (previousState) {
            case SCHEDULED: {
                this.schedule(false);
                break;
            }
            case CLOSING: {
                this.doClose();
            }
        }
    }

    private void doExecuteJobs() {
        StateMachine fsm = this.fsm();
        ArrayList<Job> batch = new ArrayList<Job>(BATCH_SIZE);
        while (!this.isClosing()) {
            this.jobs.drainTo(batch, BATCH_SIZE);
            if (!batch.isEmpty()) {
                this.log.debug("[%s] Executing %d scheduled jobs", new Object[]{this.id, batch.size()});
                batch.forEach(job -> this.executeJob(fsm, (Job)job));
            } else {
                if (!fsm.shouldStickOnThread() && !fsm.hasOpenStatement()) break;
                Job job2 = null;
                try {
                    this.log.debug("[%s] Waiting for additional jobs", new Object[]{this.id});
                    job2 = this.jobs.pollFirst(10L, TimeUnit.SECONDS);
                }
                catch (InterruptedException ex) {
                    this.log.debug("[" + this.id + "] Worker interrupted while awaiting new jobs", (Throwable)ex);
                }
                if (job2 != null) {
                    this.executeJob(fsm, job2);
                } else {
                    try {
                        fsm.validateTransaction();
                    }
                    catch (KernelException ex) {
                        this.log.error("[" + this.id + "] Failed to validate transaction", (Throwable)ex);
                        this.close();
                        break;
                    }
                }
            }
            batch.clear();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void executeJob(StateMachine fsm, Job job) {
        this.channel.write((Object)StateSignal.BEGIN_JOB_PROCESSING);
        try {
            job.perform(fsm);
        }
        catch (BoltConnectionAuthFatality ex) {
            this.close();
            if (ex.isLoggable()) {
                this.userLog.warn(ex.getMessage());
            }
        }
        catch (BoltProtocolBreachFatality ex) {
            this.close();
            this.log.warn("[" + this.id + "] Terminating connection due to protocol breach", (Throwable)ex);
        }
        catch (Throwable ex) {
            this.close();
            this.userLog.error("[" + this.id + "] Terminating connection due to unexpected error", ex);
        }
        finally {
            this.channel.write((Object)StateSignal.END_JOB_PROCESSING);
        }
    }

    @Override
    public boolean isInterrupted() {
        return this.remainingInterrupts.get() != 0;
    }

    @Override
    public void interrupt() {
        StateMachine fsm = this.fsm();
        this.remainingInterrupts.incrementAndGet();
        fsm.interrupt();
    }

    @Override
    public boolean reset() {
        int current;
        do {
            if ((current = this.remainingInterrupts.get()) != 0) continue;
            return true;
        } while (!this.remainingInterrupts.compareAndSet(current, current - 1));
        if (current == 1) {
            this.log.debug("[%s] Connection has been reset", new Object[]{this.id});
            return true;
        }
        this.log.debug("[%s] Interrupt has been cleared (%d interrupts remain active)", new Object[]{this.id, current - 1});
        return false;
    }

    @Override
    public boolean isClosing() {
        return this.state.get() == State.CLOSING;
    }

    @Override
    public boolean isClosed() {
        return this.state.get() == State.CLOSED;
    }

    @Override
    public void close() {
        State originalState;
        do {
            if ((originalState = this.state.get()) != State.CLOSING && originalState != State.CLOSED) continue;
            return;
        } while (!this.state.compareAndSet(originalState, State.CLOSING));
        this.log.debug("[%s] Marked connection for closure", new Object[]{this.id});
        this.notifyListenersSafely("markForClosure", ConnectionListener::onMarkedForClosure);
        if (originalState == State.IDLE) {
            this.log.debug("[%s] Connection is idling - Performing inline closure", new Object[]{this.id});
            this.doClose();
        }
    }

    private void doClose() {
        if (!this.state.compareAndSet(State.CLOSING, State.CLOSED)) {
            return;
        }
        this.log.debug("[%s] Closing connection", new Object[]{this.id});
        try {
            BoltProtocol protocol;
            while (!this.protocol.compareAndSet(protocol = (BoltProtocol)this.protocol.get(), null)) {
            }
            StateMachine fsm = this.fsm;
            if (fsm != null) {
                fsm.close();
            }
        }
        catch (Throwable ex) {
            this.log.warn("[" + this.id + "] Failed to terminate finite state machine", ex);
        }
        this.channel.close();
        this.memoryTracker.close();
        this.notifyListenersSafely("close", ConnectionListener::onClosed);
        this.closeFuture.complete(null);
    }

    @Override
    public Future<?> closeFuture() {
        return this.closeFuture;
    }

    private static enum State {
        IDLE,
        SCHEDULED,
        CLOSING,
        CLOSED;

    }

    public static class Factory
    implements Connection.Factory {
        private final ExecutorService executor;
        private final Clock clock;
        private final LogService logService;

        public Factory(ExecutorService executor, Clock clock, LogService logService) {
            this.executor = executor;
            this.clock = clock;
            this.logService = logService;
        }

        @Override
        public AtomicSchedulingConnection create(Connector connector, String id, Channel channel) {
            LocalMemoryTracker memoryTracker = new LocalMemoryTracker(connector.memoryPool(), 0L, 64L, null);
            memoryTracker.allocateHeap(SHALLOW_SIZE);
            return new AtomicSchedulingConnection(connector, id, channel, System.currentTimeMillis(), (MemoryTracker)memoryTracker, this.logService, this.executor, this.clock);
        }
    }
}

