/**
 * Mule ESB Tools
 *
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tools.artifact.archiver.api;

import org.mule.tools.artifact.archiver.internal.FileCollection;
import org.mule.tools.artifact.archiver.internal.MuleArchiver;

import java.io.File;
import java.io.IOException;
import java.util.Set;

/**
 * Builder for Mule Application archives.
 */
public class MuleApplicationArchiveBuilder
{

    private static final String MULE_DEPLOY_PROPERTIES = "mule-deploy.properties";
    private static final String MULE_APP_PROPERTIES = "mule-app.properties";
    private static final String MULE_CONFIG_XML = "mule-config.xml";
    private final FileCollection jarFolders = new FileCollection();
    private final FileCollection rootResourceFolder = new FileCollection();
    private final FileCollection resourceFolders = new FileCollection().returnDirectoriesOnly();
    private final FileCollection classesFolders = new FileCollection().returnDirectoriesOnly();
    private File destinationFile;

    /**
     * @param muleDeployPropertiesFile mule-deploy.properties application file
     * @return builder
     */
    public MuleApplicationArchiveBuilder setMuleDeployPropertiesFile(File muleDeployPropertiesFile)
    {
        if (!muleDeployPropertiesFile.getName().equals(MULE_DEPLOY_PROPERTIES))
        {
            throw new IllegalArgumentException("File must be named " + MULE_DEPLOY_PROPERTIES);
        }
        this.rootResourceFolder.addFile(muleDeployPropertiesFile);
        return this;
    }

    /**
     * @param muleAppPropertiesFile mule-app.properties application file
     * @return builder
     */
    public MuleApplicationArchiveBuilder setMuleAppPropertiesFile(File muleAppPropertiesFile)
    {
        if (!muleAppPropertiesFile.getName().equals(MULE_APP_PROPERTIES))
        {
            throw new IllegalArgumentException("File must be named " + MULE_APP_PROPERTIES);
        }
        this.rootResourceFolder.addFile(muleAppPropertiesFile);
        return this;
    }

    /**
     * @param muleConfigFile mule-config.xml configuration file
     * @return builder
     */
    public MuleApplicationArchiveBuilder setMuleConfigFile(File muleConfigFile)
    {
        if (!muleConfigFile.getName().equals(MULE_CONFIG_XML))
        {
            throw new IllegalArgumentException("File must be named " + MULE_CONFIG_XML);
        }
        this.rootResourceFolder.addFile(muleConfigFile);
        return this;
    }

    /**
     * @param appResourceFolder folder with all the configuration files of the application
     * @return builder
     */
    public MuleApplicationArchiveBuilder setAppResourceFolder(File appResourceFolder)
    {
        this.rootResourceFolder.addFolder(appResourceFolder);
        return this;
    }

    /**
     * @param destinationFile file to be created with the content of the app
     * @return
     */
    public MuleApplicationArchiveBuilder setDestinationFile(File destinationFile)
    {
        this.destinationFile = destinationFile;
        return this;
    }

    /**
     * Resource go under: app-folder/
     *
     * @param folder folder to include in the root folder of the app.
     * @return builder
     */
    public MuleApplicationArchiveBuilder addRootResourcesFolder(File folder)
    {
        this.rootResourceFolder.addFolder(folder);
        return this;
    }

    /**
     * Resource go under: app-folder/
     *
     * @param file file to be included in the root folder of the app
     * @return builder
     */
    public MuleApplicationArchiveBuilder addRootResourcesFile(File file)
    {
        this.rootResourceFolder.addFile(file);
        return this;
    }

    /**
     * Resource go under: app-folder/classes/
     *
     * @param folder folder content to be included in the classes folder of the app
     * @return
     */
    public MuleApplicationArchiveBuilder addResourcesFolder(File folder)
    {
        this.resourceFolders.addFolder(folder);
        return this;
    }

    /**
     * Resource go under: app-folder/classes/
     *
     * @param folders folders content to be included in the classes folder of the app
     * @return
     */
    public MuleApplicationArchiveBuilder addResourcesFolders(File... folders)
    {
        this.resourceFolders.addFolders(folders);
        return this;
    }

    public MuleApplicationArchiveBuilder addClassesFolder(File folder)
    {
        this.classesFolders.addFolder(folder);
        return this;
    }

    public MuleApplicationArchiveBuilder addClassesFolders(File... folders)
    {
        this.classesFolders.addFolders(folders);
        return this;
    }

    public MuleApplicationArchiveBuilder excludeFromClassesFolder(File... files)
    {
        this.resourceFolders.excludeFiles(files);
        return this;
    }

    public MuleApplicationArchiveBuilder addJarLibrary(File jarFile)
    {
        this.jarFolders.addFile(jarFile);
        return this;
    }

    public MuleApplicationArchiveBuilder addJarLibraries(File... jarFiles)
    {
        this.jarFolders.addFiles(jarFiles);
        return this;
    }

    public MuleApplicationArchiveBuilder addJarLibraryFolder(File folder)
    {
        this.jarFolders.addFolder(folder);
        return this;
    }

    /**
     * Creates the deployable archive for the application.
     * @throws IOException
     */
    public void createDeployableFile() throws IOException
    {
        final MuleArchiver muleArchiver = new MuleArchiver();
        for (File jarFolder : jarFolders.allFiles())
        {
            muleArchiver.addLib(jarFolder);
        }
        for (File classesFolder : classesFolders.allFiles())
        {
            muleArchiver.addClasses(classesFolder, null, transformToPatterns(resourceFolders.allExcludedFiles()));
        }
        for (File resourceFolder : resourceFolders.allFiles())
        {
            muleArchiver.addClasses(resourceFolder, null, transformToPatterns(resourceFolders.allExcludedFiles()));
        }

        for (File rootResourceFile : rootResourceFolder.allFiles())
        {
            if (rootResourceFile.isFile())
            {
                muleArchiver.addResourcesFile(rootResourceFile);
            }
            else
            {
                muleArchiver.addResources(rootResourceFile);
            }
        }
        muleArchiver.setDestFile(destinationFile);
        muleArchiver.createArchive();
    }

    private String[] transformToPatterns(Set<File> files)
    {
        final String[] patterns = new String[files.size()];
        int i = 0;
        for (File file : files)
        {
            patterns[i] = "**/" + file.getName();
            i++;
        }
        return patterns;
    }
}
