package org.mule.devkit.wsdl.parser;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.Queue;
import java.util.Stack;
import java.util.logging.Logger;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.beanutils.PropertyUtils;
import org.mule.devkit.wsdl.Address;
import org.mule.devkit.wsdl.Binding;
import org.mule.devkit.wsdl.Definition;
import org.mule.devkit.wsdl.Fault;
import org.mule.devkit.wsdl.Header;
import org.mule.devkit.wsdl.Input;
import org.mule.devkit.wsdl.Message;
import org.mule.devkit.wsdl.ObjectType;
import org.mule.devkit.wsdl.Operation;
import org.mule.devkit.wsdl.Output;
import org.mule.devkit.wsdl.Part;
import org.mule.devkit.wsdl.Port;
import org.mule.devkit.wsdl.PortType;
import org.mule.devkit.wsdl.Schema;
import org.mule.devkit.wsdl.Service;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

public class WSDLReader {

    private final static Logger logger = Logger
            .getLogger(WSDLReader.class.getName());

	private Element root;
	private Definition def;

	private static final String SERVICE = "service";
	private static final String PORT = "port";
	private static final String ADDRESS = "address";
	private static final String BINDING = "binding";
	private static final String OPERATION = "operation";
	private static final String INPUT = "input";
	private static final String HEADER = "header";
	private static final String OUTPUT = "output";
	private static final String FAULT = "fault";
	private static final String PORT_TYPE = "portType";
	private static final String DOCUMENTATION = "documentation";
	private static final String SCHEMA = "schema";
	private static final String IMPORT = "import";
	private static final String INCLUDE = "include";
	private static final String TYPES = "types";
	private static final String COMPLEX_TYPE = "complexType";
	private static final String SIMPLE_TYPE = "simpleType";
	private static final String ELEMENT = "element";
	private static final String MESSAGE = "message";
	private static final String PART = "part";
	private static final String SEQUENCE = "sequence";
	private static final String COMPLEX_CONTENT = "complexContent";
	private static final String EXTENSION = "extension";
	private static final String RESTRICTION = "restriction";

	private Stack<String> paths = new Stack<String>();
	public Definition getFromWsdl(String path)
			throws ParserConfigurationException, SAXException, IOException,
			DOMException, IllegalAccessException, InvocationTargetException,
			NoSuchMethodException {
		loadWsdlDoc(path);

		def = new Definition();

		def.setFileName(path);

		populteDefinition();

		return def;
	}

	private void populteDefinition() throws DOMException,
			IllegalAccessException, InvocationTargetException,
			NoSuchMethodException {
        logger.info("Populating definition");
		NodeList nodes = root.getChildNodes();
		for (int nodeIndex = 0; nodeIndex < nodes.getLength(); nodeIndex++) {
            try{
			Node node = nodes.item(nodeIndex);
			if (node.getNodeType() == Node.TEXT_NODE)
				continue;
			if (SERVICE.equals(node.getLocalName())) {
				processServiceNode(node);
			}
			if (BINDING.equals(node.getLocalName())) {
				processBindingNode(node);
			}
			if (PORT_TYPE.equals(node.getLocalName())) {
				processPortType(node);
			}
			if (TYPES.equals(node.getLocalName())) {
				processPortTypes(node);
			}
			if (MESSAGE.equals(node.getLocalName())) {
				processMessage(node);
			}}
            catch(Exception ex){
                logger.warning(ex.getMessage());
            }
		}
	}

	private void processMessage(Node node) throws DOMException,
			IllegalAccessException, InvocationTargetException,
			NoSuchMethodException {
		Message message = new Message();
		def.addMessage(message);
		populateAttributes(message, node);
		NodeList nodes = node.getChildNodes();
		for (int nodeIndex = 0; nodeIndex < nodes.getLength(); nodeIndex++) {
			Node childNode = nodes.item(nodeIndex);
			if (childNode.getNodeType() == Node.TEXT_NODE)
				continue;
			if (PART.equals(childNode.getLocalName())) {
				Part part = new Part();
				message.getPart().add(part);
				populateAttributes(part, childNode);
			}

		}
	}

	private void processPortTypes(Node node) throws DOMException,
			IllegalAccessException, InvocationTargetException,
			NoSuchMethodException {
		NodeList nodes = node.getChildNodes();
		for (int nodeIndex = 0; nodeIndex < nodes.getLength(); nodeIndex++) {
			Node childNode = nodes.item(nodeIndex);
			if (childNode.getNodeType() == Node.TEXT_NODE)
				continue;
			if (SCHEMA.equals(childNode.getLocalName())) {
				Schema schema = new Schema();
				def.getSchemas().add(schema);
				processSchema(schema, childNode);
			}

		}
	}

	private void processSchema(Schema schema, Node node) throws DOMException,
			IllegalAccessException, InvocationTargetException,
			NoSuchMethodException {
		NodeList nodes = node.getChildNodes();
		populateAttribute(schema, node, "targetNamespace");
		for (int nodeIndex = 0; nodeIndex < nodes.getLength(); nodeIndex++) {
			Node childNode = nodes.item(nodeIndex);
			if (childNode.getNodeType() == Node.TEXT_NODE)
				continue;
			if (ELEMENT.equals(childNode.getLocalName())
					|| SIMPLE_TYPE.equals(childNode.getLocalName())
					|| COMPLEX_TYPE.equals(childNode.getLocalName())) {
				ObjectType obj = new ObjectType();
				populateAttribute(obj, childNode, "name");
				populateAttribute(obj, childNode, "type");
				Object value = getAttribute(childNode, "abstract");
				if (value != null) {
					obj.setIsAbstract((String) value);
				}
				if (childNode.hasChildNodes()) {
					if (ELEMENT.equals(childNode.getLocalName())) {
						processElement(obj, childNode);
					} else if (COMPLEX_TYPE.equals(childNode.getLocalName())) {
						processComplexType(obj, childNode);
					} else if (SIMPLE_TYPE.equals(childNode.getLocalName())) {
						processSimpleType(obj, childNode);
					}
				}
				schema.getTypes().add(obj);
			}
			if (IMPORT.equals(childNode.getLocalName())) {
				processImport(childNode);
			}
			if (INCLUDE.equals(childNode.getLocalName())) {
				processInclude(childNode);
			}
		}
	}

	private void processImport(Node node) {
		try {
			for (int attributeIndex = 0; attributeIndex < node.getAttributes()
					.getLength(); attributeIndex++) {
				System.out.println(node.getAttributes().item(attributeIndex)
						.getNodeName()
						+ "="
						+ node.getAttributes().item(attributeIndex)
								.getNodeValue());
				if (node.getAttributes().item(attributeIndex).getNodeName()
						.equals("schemaLocation")) {
					{
						String path = node.getAttributes().item(attributeIndex)
								.getNodeValue();
						File fXmlFile = new File(path);
						if (path.startsWith("http://")) {
							URL url = new URL(path);
							fXmlFile = new File(
									tempDir,
									fXmlFile.getName());
							org.apache.commons.io.FileUtils.copyURLToFile(url,
									fXmlFile);
							System.out.println(path);
						}else{
							File temp = new File(paths.peek(),path);
							fXmlFile = new File(
									tempDir,
									fXmlFile.getName());
							org.apache.commons.io.FileUtils.copyURLToFile(temp.toURL(),
									fXmlFile);
							System.out.println(temp.getAbsolutePath());
						}
						DocumentBuilderFactory dbFactory = DocumentBuilderFactory
								.newInstance();
						dbFactory.setNamespaceAware(true);
						DocumentBuilder dBuilder = dbFactory
								.newDocumentBuilder();
						Document doc;

						doc = dBuilder.parse(fXmlFile);

						Element schemaRoot = doc.getDocumentElement();
						this.processSchema(new Schema(), schemaRoot);
					}
				}
			}
		} catch (SAXException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		} catch (DOMException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
		}
	}

	private void processInclude(Node node) {
		for (int attributeIndex = 0; attributeIndex < node.getAttributes()
				.getLength(); attributeIndex++) {
			System.out.println(node.getAttributes().item(attributeIndex).getNodeName()+"="+node.getAttributes().item(attributeIndex).getNodeValue());
		}
	}

	private void processSimpleType(ObjectType objectType, Node node)
			throws DOMException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException {
		NodeList nodes = node.getChildNodes();
		for (int nodeIndex = 0; nodeIndex < nodes.getLength(); nodeIndex++) {
			Node childNode = nodes.item(nodeIndex);
			if (childNode.getNodeType() == Node.TEXT_NODE)
				continue;
			if (COMPLEX_TYPE.equals(childNode.getLocalName())) {
				processComplexType(objectType, childNode);

			}
		}
	}

	private void processElement(ObjectType objectType, Node node)
			throws DOMException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException {
		NodeList nodes = node.getChildNodes();
		populateAttributes(objectType, node);
		for (int nodeIndex = 0; nodeIndex < nodes.getLength(); nodeIndex++) {
			Node childNode = nodes.item(nodeIndex);
			if (childNode.getNodeType() == Node.TEXT_NODE)
				continue;
			if (RESTRICTION.equals(childNode.getLocalName())) {
				processComplexType(objectType, childNode);

			}
		}
	}

	private void processComplexType(ObjectType objectType, Node node)
			throws DOMException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException {
		NodeList nodes = node.getChildNodes();
		for (int nodeIndex = 0; nodeIndex < nodes.getLength(); nodeIndex++) {
			Node childNode = nodes.item(nodeIndex);
			if (childNode.getNodeType() == Node.TEXT_NODE)
				continue;
			if (COMPLEX_CONTENT.equals(childNode.getLocalName())) {
				NodeList seqElements = childNode.getChildNodes();
				for (int element = 0; element < seqElements.getLength(); element++) {
					Node seqElement = seqElements.item(element);
					if (seqElement.getNodeType() == Node.TEXT_NODE)
						continue;
					if (EXTENSION.equals(seqElement.getLocalName())) {
						populateAttributes(objectType, seqElement);
						processComplexType(objectType, seqElement);
					} else if (RESTRICTION.equals(seqElement.getLocalName())) {
						populateAttributes(objectType, seqElement);
						processRestriction(objectType, seqElement);
					}
				}
			} else if (SEQUENCE.equals(childNode.getLocalName())) {
				NodeList seqElements = childNode.getChildNodes();
				for (int element = 0; element < seqElements.getLength(); element++) {
					Node seqElement = seqElements.item(element);
					if (seqElement.getNodeType() == Node.TEXT_NODE)
						continue;
					ObjectType innerObjectType = new ObjectType();

					if (ELEMENT.equals(seqElement.getLocalName())) {
						populateAttributes(innerObjectType, seqElement);
					}
					objectType.getInnerObjectType().add(innerObjectType);
				}
			}
		}
	}

	private void processRestriction(ObjectType objectType, Node seqElement) {
		// TODO Support for restrictions

	}

	private void processPortType(Node node) throws DOMException,
			IllegalAccessException, InvocationTargetException,
			NoSuchMethodException {
		PortType portType = new PortType();
		def.getPortTypes().add(portType);
		populateAttributes(portType, node);

		NodeList nodes = node.getChildNodes();
		for (int nodeIndex = 0; nodeIndex < nodes.getLength(); nodeIndex++) {
			Node childNode = nodes.item(nodeIndex);
			if (childNode.getNodeType() == Node.TEXT_NODE)
				continue;
			if (OPERATION.equals(childNode.getLocalName())) {
				Operation op = new Operation();
				portType.addOperation(op);
				populateAttributes(op, childNode);
				processOperation(op, childNode);
			}

		}
	}

	private void processBindingNode(Node node) throws DOMException,
			IllegalAccessException, InvocationTargetException,
			NoSuchMethodException {
		Binding binding = new Binding();
		def.getBindings().add(binding);
		populateAttributes(binding, node);

		NodeList nodes = node.getChildNodes();
		for (int nodeIndex = 0; nodeIndex < nodes.getLength(); nodeIndex++) {
			Node childNode = nodes.item(nodeIndex);
			if (childNode.getNodeType() == Node.TEXT_NODE)
				continue;
			if (OPERATION.equals(childNode.getLocalName())) {
				Operation op = new Operation();
				binding.addOperation(op);
				populateAttributes(op, childNode);
				processOperation(op, childNode);
			}

		}
	}

	private void processOperation(Operation operation, Node node)
			throws DOMException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException {
		NodeList nodes = node.getChildNodes();
		for (int nodeIndex = 0; nodeIndex < nodes.getLength(); nodeIndex++) {
			Node childNode = nodes.item(nodeIndex);
			if (childNode.getNodeType() == Node.TEXT_NODE)
				continue;
			if (INPUT.equals(childNode.getLocalName())) {
				processInput(childNode, operation);
			}
			if (OUTPUT.equals(childNode.getLocalName())) {
				processOutput(childNode, operation);
			}
			if (FAULT.equals(childNode.getLocalName())) {
				processFault(childNode, operation);
			}
			if (DOCUMENTATION.equals(childNode.getLocalName())) {
				processDocumentation(childNode, operation);
			}

		}
	}

	private void processDocumentation(Node node, Operation operation) {
		operation.setDocumentation(node.getTextContent());
	}

	private void processInput(Node node, Operation operation)
			throws DOMException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException {
		Input input = new Input();
		operation.setInput(input);
		populateAttributes(input, node);

		NodeList nodes = node.getChildNodes();
		for (int nodeIndex = 0; nodeIndex < nodes.getLength(); nodeIndex++) {
			Node childNode = nodes.item(nodeIndex);
			if (childNode.getNodeType() == Node.TEXT_NODE)
				continue;
			if (HEADER.equals(childNode.getLocalName())) {
				Header header = new Header();
				input.addHeader(header);
				populateAttributes(header, childNode);
			}
		}
	}

	private void processOutput(Node node, Operation operation)
			throws DOMException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException {
		Output output = new Output();
		operation.setOutput(output);
		populateAttributes(output, node);

	}

	private void processFault(Node node, Operation operation)
			throws DOMException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException {
		Fault fault = new Fault();
		operation.addFault(fault);
		populateAttributes(fault, node);
	}

	private void populateAttributes(Object entity, Node node)
			throws DOMException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException {
		for (int attributeIndex = 0; attributeIndex < node.getAttributes()
				.getLength(); attributeIndex++) {
			try{
			PropertyUtils.setProperty(entity,
					node.getAttributes().item(attributeIndex).getNodeName(),
					node.getAttributes().item(attributeIndex).getNodeValue());
			}catch(Exception ex){
				System.out.println("Could not process attribute:"+ node.getAttributes().item(attributeIndex).getNodeName()+". Error:"+ex.getMessage());
			}
		}
	}

	private void populateAttribute(Object entity, Node node, String name)
			throws DOMException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException {
		for (int attributeIndex = 0; attributeIndex < node.getAttributes()
				.getLength(); attributeIndex++) {
			if (name.equals(node.getAttributes().item(attributeIndex)
					.getNodeName())) {
				PropertyUtils
						.setProperty(entity,
								node.getAttributes().item(attributeIndex)
										.getNodeName(), node.getAttributes()
										.item(attributeIndex).getNodeValue());
				break;
			}
		}
	}

	private Object getAttribute(Node node, String name) throws DOMException,
			IllegalAccessException, InvocationTargetException,
			NoSuchMethodException {
		for (int attributeIndex = 0; attributeIndex < node.getAttributes()
				.getLength(); attributeIndex++) {
			if (name.equals(node.getAttributes().item(attributeIndex)
					.getNodeName())) {
				return node.getAttributes().item(attributeIndex).getNodeValue();
			}
		}
		return null;
	}

	private void processServiceNode(Node node) throws DOMException,
			IllegalAccessException, InvocationTargetException,
			NoSuchMethodException {
		Service service = new Service();
		def.getServices().add(service);
		populateAttributes(service, node);

		NodeList nodes = node.getChildNodes();
		for (int nodeIndex = 0; nodeIndex < nodes.getLength(); nodeIndex++) {
			Node childNode = nodes.item(nodeIndex);
			if (childNode.getNodeType() == Node.TEXT_NODE)
				continue;
			if (PORT.equals(childNode.getLocalName())) {
				Port port = new Port();
				service.addPort(port);
				populateAttributes(port, childNode);
				processPortNode(childNode, port);
			}
			if (DOCUMENTATION.equals(childNode.getLocalName())) {
				service.setDocumentation(childNode.getTextContent());
			}

		}
	}

	private void processPortNode(Node node, Port port) throws DOMException,
			IllegalAccessException, InvocationTargetException,
			NoSuchMethodException {
		NodeList nodes = node.getChildNodes();
		for (int nodeIndex = 0; nodeIndex < nodes.getLength(); nodeIndex++) {
			Node childNode = nodes.item(nodeIndex);
			if (childNode.getNodeType() == Node.TEXT_NODE)
				continue;
			if (ADDRESS.equals(childNode.getLocalName())) {
				processAddress(childNode, port);
			}

		}
	}

	private void processAddress(Node node, Port port) throws DOMException,
			IllegalAccessException, InvocationTargetException,
			NoSuchMethodException {
		Address address = new Address();
		port.setAddress(address);
		populateAttributes(address, node);
	}

    private File tempDir;

	private void loadWsdlDoc(String path) throws ParserConfigurationException,
			SAXException, IOException {
        logger.info("Loading WSDL");

		File fXmlFile = new File(path);
		if(path.startsWith("http://")){
			
			URL url = new URL(path);
			paths.push(url.getPath()+".wsdl");
			tempDir = new File(System.getProperty("java.io.tmpdir"));
			fXmlFile = new File(tempDir,url.getPath()+".wsdl");
			org.apache.commons.io.FileUtils.copyURLToFile(url,
					fXmlFile);
			System.out.println(paths.peek());
		}
		DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
		dbFactory.setNamespaceAware(true);
		DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();
		Document doc = dBuilder.parse(fXmlFile);
		root = doc.getDocumentElement();
	}

}
