/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.ws.model.cache;

import org.mule.api.DefaultMuleException;
import org.mule.api.MuleContext;
import org.mule.api.MuleException;
import org.mule.config.i18n.CoreMessages;
import org.mule.devkit.api.lifecycle.LifeCycleManager;
import org.mule.devkit.api.ws.authentication.WsdlSecurityStrategy;
import org.mule.devkit.api.ws.authentication.WsdlTimestamp;
import org.mule.devkit.api.ws.authentication.WsdlUsernameToken;
import org.mule.devkit.api.ws.transport.WsdlTransport;
import org.mule.devkit.internal.ws.common.EnhancedServiceDefinition;
import org.mule.devkit.internal.ws.transport.WsdlTransportException;
import org.mule.devkit.internal.ws.transport.WsdlTransportFactoryBuilder;
import org.mule.module.http.api.requester.HttpRequesterConfig;
import org.mule.module.ws.consumer.WSConsumerConfig;
import org.mule.module.ws.security.SecurityStrategy;
import org.mule.module.ws.security.WSSecurity;
import org.mule.module.ws.security.WssTimestampSecurityStrategy;
import org.mule.module.ws.security.WssUsernameTokenSecurityStrategy;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;

public class WSConsumerConfigCallable implements Callable<WSConsumerConfig> {

    private MuleContext muleContext;
    private EnhancedServiceDefinition enhancedServiceDefinition;

    public WSConsumerConfigCallable(MuleContext muleContext, EnhancedServiceDefinition enhancedServiceDefinition) {
        this.muleContext = muleContext;
        this.enhancedServiceDefinition = enhancedServiceDefinition;
    }

    @Override
    public WSConsumerConfig call() throws Exception {
        WSConsumerConfig wsConsumerConfig = new WSConsumerConfig();
        wsConsumerConfig.setMuleContext(muleContext);
        wsConsumerConfig.setWsdlLocation(enhancedServiceDefinition.getWsdlUrl().toString());
        wsConsumerConfig.setService(enhancedServiceDefinition.getService());
        wsConsumerConfig.setPort(enhancedServiceDefinition.getPort());
        wsConsumerConfig.setServiceAddress(enhancedServiceDefinition.getServiceAddress());

        initialiseSecurity(wsConsumerConfig);
        initialiseConnectorConfig(wsConsumerConfig);

        LifeCycleManager.executeInitialiseAndStart(wsConsumerConfig);
        return wsConsumerConfig;
    }

    private void initialiseConnectorConfig(WSConsumerConfig wsConsumerConfig) throws MuleException {
        if (!enhancedServiceDefinition.getTransport().isPresent()) {
            return;
        }
        WsdlTransport wsdlTransport = enhancedServiceDefinition.getTransport().get();
        try {
            HttpRequesterConfig httpRequesterConfig = WsdlTransportFactoryBuilder.build(wsdlTransport, muleContext);
            wsConsumerConfig.setConnectorConfig(httpRequesterConfig);
        } catch (WsdlTransportException e) {
            throw new DefaultMuleException(
                    CoreMessages.createStaticMessage("There was an error when trying to resolve the underlying transport, please refer to the logs to see more details."), e);
        }
    }

    private void initialiseSecurity(WSConsumerConfig wsConsumerConfig) {
        List<SecurityStrategy> securityStrategyList = getSecurityStrategies();

        if (!securityStrategyList.isEmpty()) {
            WSSecurity wsSecurity = new WSSecurity();
            wsSecurity.setStrategies(securityStrategyList);
            wsConsumerConfig.setSecurity(wsSecurity);
        }
    }

    private List<SecurityStrategy> getSecurityStrategies() {
        List<SecurityStrategy> securityStrategyList = new ArrayList<SecurityStrategy>();
        for (WsdlSecurityStrategy wsdlSecurityStrategy : enhancedServiceDefinition.getStrategies()) {
            if (wsdlSecurityStrategy instanceof WsdlUsernameToken) {
                WsdlUsernameToken wsdlUsernameToken = (WsdlUsernameToken) wsdlSecurityStrategy;

                WssUsernameTokenSecurityStrategy wssUsernameTokenSecurityStrategy = new WssUsernameTokenSecurityStrategy();
                wssUsernameTokenSecurityStrategy.setUsername(wsdlUsernameToken.getUsername());
                wssUsernameTokenSecurityStrategy.setPassword(wsdlUsernameToken.getPassword());
                wssUsernameTokenSecurityStrategy.setPasswordType(wsdlUsernameToken.getPasswordType());
                wssUsernameTokenSecurityStrategy.setAddCreated(wsdlUsernameToken.isAddCreated());
                wssUsernameTokenSecurityStrategy.setAddNonce(wsdlUsernameToken.isAddNonce());

                securityStrategyList.add(wssUsernameTokenSecurityStrategy);
            }

            if (wsdlSecurityStrategy instanceof WsdlTimestamp) {
                WsdlTimestamp wsdlTimestamp = (WsdlTimestamp) wsdlSecurityStrategy;

                WssTimestampSecurityStrategy wssTimestampSecurityStrategy = new WssTimestampSecurityStrategy();
                wssTimestampSecurityStrategy.setExpires(wsdlTimestamp.getExpires());

                securityStrategyList.add(wssTimestampSecurityStrategy);
            }
        }
        return securityStrategyList;
    }
}
