/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.lic.model;

import org.apache.commons.lang.StringUtils;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * This class is intended to avoid maven-artifact dependency,
 * since it is not usable in every runtime.
 * With this in mind, Version will represent an standar maven version,
 * with three or less numeric fields, and also its snapshot identifier,
 * if one is found.
 */
public class Version {

    private static final Pattern versionPattern = Pattern.compile("^([0-9])\\.([0-9]|x)\\.([0-9]|x)\\-?(.*SNAPSHOT)?$");

    private Integer major = -1;
    private Integer minor = -1;
    private Integer bug = -1;
    private boolean isSnapshot = false;

    public Version(String version) {
        if (StringUtils.isBlank(version)) {
            // Any version, as in the lower bound of [,3.x.x]
            return;
        }

        Matcher matcher = versionPattern.matcher(version);
        if (!matcher.matches()) {
            throw new IllegalArgumentException(version + " is not a valid version format");
        }

        isSnapshot = matcher.groupCount() == 4;

        major = Integer.valueOf(matcher.group(1));
        if (!StringUtils.equalsIgnoreCase(matcher.group(2), "x")) {
            minor = Integer.valueOf(matcher.group(2));
        }
        if (!StringUtils.equalsIgnoreCase(matcher.group(3), "x")) {
            bug = Integer.valueOf(matcher.group(3));
        }

    }

    public Integer major() {
        return major;
    }

    public Integer minor() {
        return minor;
    }

    public Integer bug() {
        return bug;
    }

    public boolean snapshot() {
        return isSnapshot;
    }

    public boolean isLowerThan(Version version, boolean includeLimit) {
        return includeLimit ? isLowerOrEqual(version) : isLower(version);
    }

    private boolean isLower(Version version) {
        return major == -1 || version.major() == -1 || (major < version.major() &&
                (minor == -1 || version.minor() == -1 || minor < version.minor()) &&
                (bug == -1 || version.bug() == -1 || bug < version.bug()));
    }

    private boolean isLowerOrEqual(Version version) {
        return major == -1 || version.major() == -1 || (major <= version.major() &&
                (minor == -1 || version.minor() == -1 || minor <= version.minor()) &&
                (bug == -1 || version.bug() == -1 || bug <= version.bug()));
    }

    public boolean isGraterThan(Version version, boolean includeLimit) {
        return includeLimit ? isGraterOrEqual(version) : isGrater(version);
    }

    private boolean isGrater(Version version) {
        return major == -1 || version.major() == -1 || (major > version.major() &&
                (minor == -1 || version.minor() == -1 || minor > version.minor()) &&
                (bug == -1 || version.bug() == -1 || bug > version.bug()));
    }

    private boolean isGraterOrEqual(Version version) {
        return major == -1 || version.major() == -1 || (major >= version.major() &&
                (minor == -1 || version.minor() == -1 || minor >= version.minor()) &&
                (bug == -1 || version.bug() == -1 || bug >= version.bug()));
    }
}
