/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.metadata;

import org.apache.commons.lang.StringUtils;
import org.mule.common.metadata.MetaDataKey;
import org.mule.common.metadata.MetaDataPropertyManager;
import org.mule.common.metadata.TypeMetaDataModel;
import org.mule.common.metadata.key.property.MetaDataKeyProperty;
import org.mule.common.metadata.key.property.dsql.DsqlFromMetaDataKeyProperty;
import org.mule.devkit.api.metadata.ComposedMetaDataKey;
import org.mule.devkit.api.metadata.exception.InvalidKeyException;
import org.mule.devkit.api.metadata.exception.InvalidSeparatorException;

import java.util.*;
import java.util.Map.Entry;

/**
 * <p>{@link MetaDataKey} default implementation. This should be used to describe the service entities/types names and display name.</p>
 *
 * @author Mulesoft, Inc
 */
public class DefaultComposedMetaDataKey implements ComposedMetaDataKey, TypeMetaDataModel {

    // WARNING: Modifying this field may break mule applications if the connector does not use a custom keySeparator
    public static final String DEFAULT_KEY_SEPARATOR = "||";

    private static final String DEFAULT_CATEGORY = "DEFAULT";

    private Map<String, String> orderedKeyDisplayNamebyId = new LinkedHashMap<String, String>();
    private MetaDataPropertyManager<MetaDataKeyProperty> metaDataKeyPropertiesManager;
    private String category = DEFAULT_CATEGORY;
    private String customSeparator = "";

    /**
     * Basic constructor, default separator will be used if none is provided when id's are fetched.
     */
    public DefaultComposedMetaDataKey() {
        initializePropertiesManager();
    }

    /**
     * Custom separator constructor. This separator will act as default for this key.
     *
     * @throws org.mule.devkit.api.metadata.exception.InvalidSeparatorException
     */
    public DefaultComposedMetaDataKey(String separator) {
        this();
        setSeparator(separator);
    }

    /**
     * Copy constructor. Both keys will preserve the same properties references
     */
    public DefaultComposedMetaDataKey(ComposedMetaDataKey origin) {
        this(origin.getSeparator());
        this.setCategory(origin.getCategory());

        Iterator<String> ids = origin.getSortedIds().iterator();
        Iterator<String> displayNames = origin.getSortedDisplayNames().iterator();

        while (ids.hasNext() && displayNames.hasNext()) {
            this.orderedKeyDisplayNamebyId.put(ids.next(), displayNames.next());
        }

        for (MetaDataKeyProperty property : origin.getProperties()) {
            this.addProperty(property);
        }
    }

    /**
     * Creates a ComposedMetaDataKey from a simple MetaDataKey.
     * Using the separator, the key will be split in different levels of the new ComposedMetaDataKey.
     *
     * @param separator KeySeparator for
     * @throws org.mule.devkit.api.metadata.exception.InvalidKeyException
     */
    public DefaultComposedMetaDataKey(MetaDataKey defaultKey, String separator) {
        this(separator);

        if (defaultKey == null) {
            throw new InvalidKeyException("Source MetaDataKey cannot be null");
        } else if (!StringUtils.contains(defaultKey.getId(), separator)) {
            throw new InvalidSeparatorException(String.format("Separator not present in key. MetaDataKey id: %s label: %s", defaultKey.getId(), defaultKey.getDisplayName()));
        }

        buildLevelsFromKey(defaultKey, separator);

        for (MetaDataKeyProperty metaDataKeyProperty : defaultKey.getProperties()) {
            this.addProperty(metaDataKeyProperty);
        }

        this.setCategory(defaultKey.getCategory());
    }

    /**
     * Creates a new level in the current key with this id.
     *
     * @throws org.mule.devkit.api.metadata.exception.InvalidKeyException
     */
    @Override
    public void addLevel(String levelId, String label) {
        if (!StringUtils.isBlank(levelId) && !StringUtils.isBlank(label) && !orderedKeyDisplayNamebyId.containsKey(levelId)) {
            this.orderedKeyDisplayNamebyId.put(levelId, label);
        } else if (orderedKeyDisplayNamebyId.containsKey(levelId)) {
            throw new InvalidKeyException("Duplicated id " + levelId);
        } else {
            throw new InvalidKeyException("Key id and it's label cannot be null");
        }
    }

    /**
     * This will return a single string representation of the key id's, appending each level id with
     * a the default separator @link{DefaultComposedMetaDataKey.DEFAULT_KEY_SEPARATOR}.
     */
    @Override
    public String getId() {
        return getId(customSeparator.isEmpty() ? DEFAULT_KEY_SEPARATOR : customSeparator);
    }

    @Override
    public Integer levels() {
        return orderedKeyDisplayNamebyId.size();
    }

    @Override
    public String getSeparator() {
        return this.customSeparator.isEmpty() ? DEFAULT_KEY_SEPARATOR : this.customSeparator;
    }

    private void setSeparator(String separator) {
        if (StringUtils.isBlank(separator)) {
            throw new InvalidSeparatorException("Separator cannot be empty nor blank");
        }

        this.customSeparator = separator;
    }

    /**
     * This will return a single string representation of the key labels, appending each level label with
     * a the default separator @link{DefaultComposedMetaDataKey.DEFAULT_KEY_SEPARATOR}.
     */
    @Override
    public String getDisplayName() {
        return getDisplayName(customSeparator.isEmpty() ? DEFAULT_KEY_SEPARATOR : customSeparator);
    }

    /**
     * This will return a single string representation of the key labels, appending each level label
     * with the parameterized separator.
     *
     * @param separator string used as division between ids
     */
    @Override
    public String getId(String separator) {
        StringBuffer composedId = new StringBuffer();
        Iterator<Entry<String, String>> it = orderedKeyDisplayNamebyId.entrySet().iterator();
        while (it.hasNext()) {
            composedId.append(it.next().getKey());
            if (it.hasNext()) {
                composedId.append(separator);
            }
        }

        return composedId.toString();
    }

    /**
     * This will return a single string representation of the key labels, appending each level label
     * with the parameterized separator.
     *
     * @param separator string used as division between labels
     */
    @Override
    public String getDisplayName(String separator) {
        StringBuffer composedLabel = new StringBuffer();
        Iterator<Entry<String, String>> it = orderedKeyDisplayNamebyId.entrySet().iterator();
        while (it.hasNext()) {
            composedLabel.append(it.next().getValue());
            if (it.hasNext())
                composedLabel.append(separator);
        }

        return composedLabel.toString();
    }

    @Override
    public List<String> getSortedIds() {
        return new ArrayList<String>(orderedKeyDisplayNamebyId.keySet());
    }

    @Override
    public List<String> getSortedDisplayNames() {
        return new ArrayList<String>(orderedKeyDisplayNamebyId.values());
    }

    @Override
    public String getCategory() {
        return category;
    }

    public void setCategory(String category) {
        this.category = category;
    }

    @Override
    public List<MetaDataKeyProperty> getProperties() {
        return this.metaDataKeyPropertiesManager.getProperties();
    }

    @Override
    public boolean addProperty(MetaDataKeyProperty metaDataKeyProperty) {
        return this.metaDataKeyPropertiesManager.addProperty(metaDataKeyProperty);
    }

    @Override
    public boolean removeProperty(MetaDataKeyProperty metaDataKeyProperty) {
        return this.metaDataKeyPropertiesManager.removeProperty(metaDataKeyProperty);
    }

    @Override
    public boolean hasProperty(Class<? extends MetaDataKeyProperty> metaDataKeyProperty) {
        return this.metaDataKeyPropertiesManager.hasProperty(metaDataKeyProperty);
    }

    @Override
    public <T extends MetaDataKeyProperty> T getProperty(Class<T> metaDataKeyProperty) {
        return this.metaDataKeyPropertiesManager.getProperty(metaDataKeyProperty);
    }

    @Override
    public String toString() {
        return "ComposedMetaDataKey:{ displayName:" + this.getDisplayName() + " id:" + this.getId() + " category:" + category + " }";
    }

    @Override
    public int hashCode() {
        int result = this.getId() != null ? this.getId().hashCode() : 0;
        result = 31 * result + (category != null ? category.hashCode() : 0);
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (!(obj instanceof DefaultComposedMetaDataKey))
            return false;

        DefaultComposedMetaDataKey that = (DefaultComposedMetaDataKey) obj;

        if (category != null ? !category.equals(that.category) : that.category != null)
            return false;
        if (getId() != null ? !getId().equals(that.getId()) : that.getId() != null)
            return false;

        return true;
    }

    /**
     * For keys comparison, the first criteria to match will be the {@link #category} of both keys, where, if it's not
     * possible to discriminate the order, the key id will take place.
     *
     * @param otherMetadataKey the key to be compared with
     * @return a negative integer, zero, or a positive integer as this object is less than, equal to, or greater than the specified object.
     */
    @Override
    public int compareTo(MetaDataKey otherMetadataKey) {
        int res = category.compareTo(otherMetadataKey.getCategory());
        if (res != 0) {
            return res;
        }
        return getId().compareTo(otherMetadataKey.getId());
    }

    /**
     * @deprecated use {@link #hasProperty(Class)}  instead
     */
    @Deprecated
    @Override
    public boolean isFromCapable() {
        return metaDataKeyPropertiesManager.hasProperty(DsqlFromMetaDataKeyProperty.class);
    }

    private void buildLevelsFromKey(MetaDataKey defaultKey, String separator) {
        List<String> ids = Arrays.asList(StringUtils.split(defaultKey.getId(), separator));
        List<String> labels = StringUtils.isBlank(defaultKey.getDisplayName()) ? ids : Arrays.asList(StringUtils.split(defaultKey.getDisplayName(), separator));

        if (labels.size() != ids.size()) {
            throw new InvalidKeyException("Invalid Key, a key must have a label for each keyId");
        }

        Iterator<String> idsIt = ids.iterator();
        Iterator<String> labelsIt = labels.iterator();
        while (idsIt.hasNext() && labelsIt.hasNext()) {
            this.addLevel(idsIt.next(), labelsIt.next());
        }
    }

    private void initializePropertiesManager() {
        this.metaDataKeyPropertiesManager = new MetaDataPropertyManager<MetaDataKeyProperty>();
        metaDataKeyPropertiesManager.addProperty(new DsqlFromMetaDataKeyProperty());
    }
}


