/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.devkit.internal.connection.management;

import static java.lang.Boolean.parseBoolean;

import org.apache.commons.pool.KeyedPoolableObjectFactory;
import org.mule.api.context.MuleContextAware;
import org.mule.api.lifecycle.Disposable;
import org.mule.api.lifecycle.Initialisable;
import org.mule.api.lifecycle.Startable;
import org.mule.api.lifecycle.Stoppable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionManagementConnectorFactory implements KeyedPoolableObjectFactory<ConnectionManagementConnectionKey, ConnectionManagementConnectorAdapter> {

    private static Logger logger = LoggerFactory.getLogger(ConnectionManagementConnectorFactory.class);
    private ConnectionManagementConnectionManager connManagementBasicConnectionManager;
    private boolean shouldSkipConnectionValidation;

    /**
     * This system property indicates whether devkit should skip validating a connection when it is borrowed from a pool.
     * Such behavior could be desired when validation is expensive or happens too frequently (like having a paged operation
     * with a very small page size).
     */
    public static final String SKIP_DEVKIT_CONNECTION_VALIDATION = "devkit.connection.validation.skip";

    public ConnectionManagementConnectorFactory(ConnectionManagementConnectionManager connManagementBasicConnectionManager) {
        this.connManagementBasicConnectionManager = connManagementBasicConnectionManager;
        this.shouldSkipConnectionValidation = parseBoolean(System.getProperty(SKIP_DEVKIT_CONNECTION_VALIDATION));
    }

    public ConnectionManagementConnectorAdapter makeObject(ConnectionManagementConnectionKey key) throws Exception {
        //TODO: REDO THIS VALIDATION
        if (key == null) {
            logger.warn("Connection key is null");
            throw new RuntimeException("Invalid key type.");
        }

        ConnectionManagementConnectionAdapter connection = connManagementBasicConnectionManager.newConnection();

        if (connection instanceof MuleContextAware) {
            ((MuleContextAware) connection).setMuleContext(connManagementBasicConnectionManager.getMuleContext());
        }
        if (connection instanceof Initialisable) {
            ((Initialisable) connection).initialise();
        }
        if (connection instanceof Startable) {
            ((Startable) connection).start();
        }
        if (connection != null && !connection.isConnected()) {
            connection.connect(key);
        }

        ConnectionManagementConnectorAdapter connector = connManagementBasicConnectionManager.newConnector(connection);

        if (connector instanceof MuleContextAware && connector != connection) {
            ((MuleContextAware) connector).setMuleContext(connManagementBasicConnectionManager.getMuleContext());
        }
        if (connector instanceof Initialisable && connector != connection) {
            ((Initialisable) connector).initialise();
        }
        if (connector instanceof Startable && connector != connection) {
            ((Startable) connector).start();
        }

        return connector;
    }

    public void destroyObject(ConnectionManagementConnectionKey key, ConnectionManagementConnectorAdapter adapter) throws Exception {
        //TODO: REDO THIS VALIDATION
        try {
            ConnectionManagementConnectionAdapter connection = connManagementBasicConnectionManager.getConnectionAdapter(adapter);
            connection.disconnect();
        } catch (Exception e) {
            throw e;
        } finally {
            if (adapter instanceof Stoppable) {
                ((Stoppable) adapter).stop();
            }
            if (adapter instanceof Disposable) {
                ((Disposable) adapter).dispose();
            }
        }
    }

    public boolean validateObject(ConnectionManagementConnectionKey key, ConnectionManagementConnectorAdapter adapter) {
        //TODO: REDO THIS VALIDATION
        try {
            ConnectionManagementConnectionAdapter connection = connManagementBasicConnectionManager.getConnectionAdapter(adapter);
            return connection.isConnected();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return false;
        }
    }

    public void activateObject(ConnectionManagementConnectionKey key, ConnectionManagementConnectorAdapter adapter) throws Exception {
        //TODO: REDO THIS VALIDATION
        if (!shouldSkipConnectionValidation) {
            try {
                ConnectionManagementConnectionAdapter connection = connManagementBasicConnectionManager.getConnectionAdapter(adapter);
                if (!connection.isConnected()) {
                    connection.connect(key);
                }
            } catch (Exception e) {
                throw e;
            }
        }
    }

    public void passivateObject(ConnectionManagementConnectionKey key, ConnectionManagementConnectorAdapter adapter) throws Exception {
    }

}
