/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.ws.metadata.utils;

import com.google.common.base.Optional;
import org.mule.devkit.internal.ws.common.WsdlUtils;
import org.mule.devkit.internal.ws.metadata.WsdlSchemaUtils;

import javax.wsdl.*;
import javax.wsdl.extensions.soap.SOAPHeader;
import javax.xml.transform.TransformerException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class InvokeWsdlResolver {

    List<SOAPHeader> operationHeaders;
    private OperationIOResolver operationIOResolver;
    private Definition definition;
    private List<String> schemas;
    private Service service;
    private Port port;
    private Optional<Part> messagePart;
    private Operation operation;

    public InvokeWsdlResolver(OperationMode operationMode, final String wsdlLocation, final String serviceName, final String portName, final String operationName)
            throws TransformerException {
        initialize(operationMode, wsdlLocation, serviceName, portName, operationName);
    }

    private void initialize(OperationMode operationMode, String wsdlLocation, String serviceName, String portName, String operationName) throws TransformerException {
        operationIOResolver = operationMode == OperationMode.INPUT ? new InputOperationResolver() : new OutputOperationResolver();

        this.definition = WsdlUtils.parseWSDL(wsdlLocation);
        this.schemas = WsdlSchemaUtils.getSchemas(definition);
        this.service = WsdlUtils.getService(definition, serviceName);
        this.port = WsdlUtils.getPort(service, portName);

        Binding binding = getPort().getBinding();
        PortType portType = binding.getPortType();
        this.operation = WsdlUtils.getOperation(portType, operationName);

        Optional<Message> optionalMessage = operationIOResolver.getMessage(operation);
        //WsdlUtils.validateNotNull(message, "There was an error while trying to resolve the message for the [" + operationName + "] operation.");

        BindingOperation bindingOperation = binding.getBindingOperation(operationName, null, null);
        operationHeaders = operationIOResolver.getHeaders(bindingOperation);

        Map<?, ?> parts = optionalMessage.isPresent() ? optionalMessage.get().getParts() : new HashMap<Object, Object>();
        messagePart = resolveMessagePart(bindingOperation, parts);
    }

    private Optional<Part> resolveMessagePart(BindingOperation bindingOperation, Map<?, ?> parts) {
        if (!parts.isEmpty()) {
            if (parts.size() == 1) {
                //hack to behave the same way as before when the message has just one part
                Object firstValueKey = parts.keySet().toArray()[0];
                return Optional.of((Part) parts.get(firstValueKey));
            } else {
                Optional<String> bodyPartNameOptional = operationIOResolver.getBodyPartName(bindingOperation);
                if (bodyPartNameOptional.isPresent()) {
                    return Optional.of((Part) parts.get(bodyPartNameOptional.get()));
                } else {
                    return Optional.absent();
                }
            }
        }
        return Optional.absent();
    }

    //getters
    public Definition getDefinition() {
        return definition;
    }

    public List<String> getSchemas() {
        return schemas;
    }

    public Service getService() {
        return service;
    }

    public Port getPort() {
        return port;
    }

    public Optional<Part> getMessagePart() {
        return messagePart;
    }

    public Operation getOperation() {
        return operation;
    }

    public List<SOAPHeader> getOperationHeaders() {
        return operationHeaders;
    }

    public enum OperationMode {
        INPUT, OUTPUT;
    }
}
