/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.internal.lic.model;

import com.google.common.base.Optional;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.mule.devkit.internal.lic.SecurityUtils;
import org.mule.devkit.internal.lic.exception.InvalidKeyException;
import org.mule.devkit.internal.lic.exception.InvalidLicenseException;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.security.PublicKey;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author MuleSoft, Inc
 */
public class CustomLicense {

    public static final String FEATURE_KEY = "feature.name";
    public static final String VERSIONS_KEY = "valid.versions";
    public static final String VENDOR_NAME_KEY = "vendor.name";
    public static final String CREATION_DATE_KEY = "creation.date";
    public static final String EXPIRATION_DATE_KEY = "expiration.date";
    private static final Logger logger = Logger.getLogger(CustomLicense.class);
    private static final Pattern versionRangePattern = Pattern.compile("(\\[|\\()([^,]*),(.*)(\\]|\\))");
    private final String featureId;
    private final String licFile;
    private final String infoFile;
    private final String signatureFile;
    private final Properties data;

    public CustomLicense(String featureId, String licName, LicenseProviderData licenseProviderData) throws InvalidKeyException {

        if (StringUtils.isBlank(featureId) || StringUtils.isBlank(licName)) {
            throw new IllegalArgumentException("Provider and ID and license name cannot be blank");
        }

        this.featureId = featureId;
        this.licFile = licName.concat(".lic");
        this.signatureFile = licName.concat(".sig");
        this.infoFile = licName.concat(".info");

        ZippedBundle licBundle = unzipBundle(licFile);
        if (!signatureIsValid(licBundle, licenseProviderData.getKey())) {
            throw new InvalidLicenseException(
                    "[" + signatureFile + "] signature is not valid for license " + infoFile + " or license was not provided by " + licenseProviderData.getName());
        }

        data = loadMetaData(licBundle);
    }

    public Optional<String> getProperty(String propertyName) {
        return Optional.fromNullable(data.getProperty(propertyName));
    }

    public boolean isValid(Entitlement entitlement) {
        return hasValidFeature() && hasValidDate() && hasValidVersion(entitlement.version());
    }

    public boolean hasValidVersion(String version) {

        Version current = new Version(version);
        String validVersions = data.getProperty(VERSIONS_KEY);
        if (!StringUtils.isBlank(validVersions)) {
            Matcher rangeMatcher = versionRangePattern.matcher(validVersions);
            if (rangeMatcher.matches()) {
                boolean minIncluded = StringUtils.equals(rangeMatcher.group(1), "[");
                String minVersion = rangeMatcher.group(2);
                String maxVersion = rangeMatcher.group(3);
                boolean maxIncluded = StringUtils.equals(rangeMatcher.group(4), "]");

                if ((!StringUtils.isBlank(minVersion) && !new Version(minVersion).isLowerThan(current, minIncluded)) || (!StringUtils.isBlank(maxVersion) && !new Version(
                        maxVersion).isGraterThan(current, maxIncluded))) {
                    return false;
                }
            }
        }
        return true;
    }

    public boolean hasValidFeature() {
        return StringUtils.equals(data.getProperty(FEATURE_KEY), featureId);
    }

    public boolean hasValidDate() {
        SimpleDateFormat dateFormat = new SimpleDateFormat("ddMMyyyy");
        Date today = new Date();
        String creation = data.getProperty(CREATION_DATE_KEY);
        String expiration = data.getProperty(EXPIRATION_DATE_KEY);

        if (StringUtils.isBlank(creation)) {
            throw new InvalidLicenseException("Invalid data found inside license info. Missing creation Date");
        }

        try {
            return !today.before(dateFormat.parse(creation)) && (StringUtils.isBlank(expiration) || !today.after(dateFormat.parse(expiration)));
        } catch (ParseException e) {
            throw new InvalidLicenseException("Invalid date found inside license info. Failed to parse creation or expiration date", e);
        }
    }

    private boolean signatureIsValid(ZippedBundle lic, PublicKey vendorKey) throws InvalidKeyException {

        Optional<byte[]> signature = lic.get(signatureFile);
        if (!signature.isPresent()) {
            logger.error("No signature named " + signatureFile + " found in bundle " + licFile);
            throw new InvalidKeyException("Failed to verify signature " + signatureFile + ". No signature named " + signatureFile + " found in bundle " + licFile);
        }

        Optional<byte[]> metadata = lic.get(infoFile);
        if (!metadata.isPresent()) {
            logger.error("No info file named " + infoFile + " found in bundle " + licFile);
            throw new InvalidKeyException("Failed to verify signature " + signatureFile + ". No signature named " + signatureFile + " found in bundle " + licFile);
        }

        try {
            return SecurityUtils.verify(signature.get(), metadata.get(), vendorKey);
        } catch (Exception e) {
            logger.error(e.getMessage());
            throw new IllegalArgumentException("An error occurred wile trying to verify signature [" + signatureFile + "] for file " + infoFile);
        }
    }

    private Properties loadMetaData(ZippedBundle licBundle) throws InvalidKeyException {

        Optional<byte[]> info = licBundle.get(infoFile);
        if (!info.isPresent()) {
            logger.error("No metadata file named " + infoFile + " found in bundle " + infoFile);
            throw new InvalidKeyException("Failed to load license data. No metadata file named  " + infoFile + "  found in " + infoFile);
        }

        try {
            Properties metadata = new Properties();
            metadata.load(new ByteArrayInputStream(info.get()));
            return metadata;
        } catch (Exception e) {
            logger.error(e.getMessage());
            throw new InvalidKeyException("Failed to load " + infoFile + " found in " + licBundle);
        }
    }

    private ZippedBundle unzipBundle(String file) {
        InputStream bundleIS = Thread.currentThread().getContextClassLoader().getResourceAsStream(file);
        if (bundleIS == null) {
            logger.error("License bundle for name [" + file + "] not found as resource");
            throw new IllegalArgumentException("License bundle for name [" + file + "] not found as resource");
        }

        return new ZippedBundle(bundleIS);
    }
}

