package org.mule.devkit.shade.connection.management;

import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.processor.MessageProcessor;



/**
 * Wrapper around {@link org.mule.api.annotations.Connector} annotated class that will infuse it with connection management capabilities. <p/> It can receive a {@link org.mule.config.PoolingProfile} which is a configuration object used to define the connection pooling parameters.
 * @param <Key> Connection key
 * @param <Adapter> Actual connector object that represents a connection
 * 
 */
public interface ConnectionManagementConnectionManager<Key extends ConnectionManagementConnectionKey, Adapter extends ConnectionManagementConnectorAdapter, Strategy> {

    /**
     * Borrow a connection from the pool
     *
     * @param connectorKey Key used to borrow the connector
     * @return An existing connector, or a newly created one
     * @throws Exception If the connection cannot be created
     */
    Adapter acquireConnection(Key connectorKey) throws Exception;

    /**
     * Return a connection to the pool
     *
     * @param connectorKey Key used to borrow the connector
     * @param connector    connector to be returned to the pool
     * @throws Exception If the connection cannot be returned
     */
    void releaseConnection(Key connectorKey, Adapter connector) throws Exception;

    /**
     * Destroy a connection
     *
     * @param connectorKey Key used to borrow the connector
     * @param connector    Connector to be destroyed
     * @throws Exception If the connection could not be destroyed.
     */
    void destroyConnection(Key connectorKey, Adapter connector) throws Exception;

    /**
     * Retrieve the default connection key
     *
     * @return The default connection key
     */
    Key getDefaultConnectionKey();


    /**
     * Retrieve the connection key evaluated with the received mule event
     *
     * @return The default connection key
     */

    Key getEvaluatedConnectionKey(org.mule.api.MuleEvent event) throws Exception;

    /**
     * Retrieve the reconnection strategy used by this connection
     * manager.
     *
     * @return The reconnection strategy {@see RetryPolicyTemplate}
     */
    org.mule.api.retry.RetryPolicyTemplate getRetryPolicyTemplate();

    Key getConnectionKey(MessageProcessor messageProcessor, MuleEvent event) throws Exception;

    ConnectionManagementConnectionAdapter newConnection();

    MuleContext getMuleContext();

    ConnectionManagementConnectorAdapter newConnector(ConnectionManagementConnectionAdapter<Strategy, Key> connection);

    ConnectionManagementConnectionAdapter getConnectionAdapter(ConnectionManagementConnectorAdapter adapter);
}
