/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.devkit.generation.studio;


import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.mule.devkit.generation.api.AbstractBaseProjectVerifier;
import org.mule.devkit.generation.api.AnnotationVerificationException;
import org.mule.devkit.generation.api.gatherer.Message;
import org.mule.devkit.generation.api.gatherer.NotificationGatherer;
import org.mule.devkit.generation.studio.editor.globalcloudconnector.GlobalCloudConnectorTypeBuilder;
import org.mule.devkit.generation.studio.utils.JaxbUnmarshaller;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.studio.NamespaceType;
import org.xml.sax.SAXException;

import javax.xml.XMLConstants;
import javax.xml.bind.JAXBElement;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Pattern;

public class ConfigRequiredLibrariesVerifier  extends AbstractBaseProjectVerifier {

    private static final String REQUIRED_LIBS_SCHEMA = "config-required-libs-schema.xsd";
    private static final String[] allowedExtensions = {"xml"};

    private Pattern requiredLibsNamePatter;
    private Module module;

    public ConfigRequiredLibrariesVerifier(){
        requiredLibsNamePatter = Pattern.compile(".+" + GlobalCloudConnectorTypeBuilder.REQUIRED_LIBS_SUFFIX);
    }

    @Override
    public boolean shouldVerify(List modules) {
        return true;
    }

    public List processableModules(List modules) {
        return modules;
    }

    @Override
    public void verify(List modules, NotificationGatherer gatherer)	throws AnnotationVerificationException {
        this.setGatherer(gatherer);
        this.module = (Module) modules.get(0);

        File requiredLibsDir = new File(ctx().getMavenInformation().getBuildDirectory(),
                                        GlobalCloudConnectorTypeBuilder.REQUIRED_LIBS_DIR);

        if (requiredLibsDir.exists()){
            Iterator<File> libsFilesIt = FileUtils.iterateFiles(requiredLibsDir, allowedExtensions, false);
            while (libsFilesIt.hasNext()){
                verifyXml(libsFilesIt.next(), gatherer);
            }
        }
    }

    private void verifyXml(File xml, NotificationGatherer gatherer){

        if (requiredLibsNamePatter.matcher(xml.getName()).matches()){
            InputStream schema = ConfigRequiredLibrariesVerifier.class.getClassLoader().getResourceAsStream(REQUIRED_LIBS_SCHEMA);
            String error;

            try {
                verifyXmlWithSchema(schema, xml);
                error = verifyElementHasLibrariesDeclaration(xml);

            } catch (SAXException e) {
                error = e.getMessage();
            } catch (IOException e) {
                error = e.getMessage();
            }

            if (!StringUtils.isBlank(error)) {
                gatherer.error(module, Message.INVALID_REQUIRED_LIBS_DEFINITION, xml.getName(), error);
            }
        }
    }

    /**
     * This method verifies that the 'require' element has at least one
     * valid library declaration
     * @param xml xml file previously validated with schema
     * @return error message if one found, empty if verification is passed
     */
    private String verifyElementHasLibrariesDeclaration(File xml) {
        String error = "";
        JAXBElement<NamespaceType> root = JaxbUnmarshaller.unmarshallXml(xml.getPath(), NamespaceType.class);
        if (!root.getValue().getRequiredLibs().getValue().hasElements()){
            error = "No required libraries found in element 'require'";
        }
        return error;
    }

    /**
     * this method verifies that the xml is valid according with the given schema
     * @param schemaSource schema for validation
     * @param xml xml to be verified
     * @throws SAXException
     * @throws IOException
     */
    private void verifyXmlWithSchema(InputStream schemaSource, File xml) throws IOException, SAXException {
        SchemaFactory factory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
        Schema schema;
        try {
            schema = factory.newSchema(new StreamSource(schemaSource));
        } catch (SAXException e) {
            throw new IllegalArgumentException("Invalid input schema for xml verification utility method", e);
        }

        schema.newValidator().validate(new StreamSource(xml));
    }

}