/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */


package org.mule.devkit.generation.studio;

import org.mule.devkit.generation.api.ConnectorCategory;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.studio.packaging.ModuleRelativePathBuilder;
import org.mule.devkit.verifiers.util.MulePathResolver;
import org.mule.devkit.model.module.Module;
import org.mule.util.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;

public class MuleStudioPluginXmlGenerator extends AbstractMuleStudioNamespaceGenerator {

    private static final String PLUGIN_XML_FILE_NAME = "plugin.xml";
    public static final String CORE_CONTRIBUTION = "org.mule.tooling.core.contribution";
    public static final String MAVEN_POPULATE = "org.mule.tooling.maven.populate";
    public static final String CONTRIBUTION_JAR_TAG = "contribution-jar";
    public static final String EXTERNAL_CONTRIBUTION_TAG = "externalContribution";

    private static List<Product> CONSUMES = Arrays.asList(Product.STUDIO_EDITOR_XML);
    private static List<Product> PRODUCES = Arrays.asList(Product.STUDIO_PLUGIN_XML);
    private static final String PROPERTIES_RELATIVE_PATH = "../../../META-INF/studio.properties";

    private Document pluginDoc;

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        try {

            createPluginDocument();

            Element pluginElement = pluginDoc.createElement("plugin");
            pluginDoc.appendChild(pluginElement);

            Element coreContributionExtension = createExtensionPoint(CORE_CONTRIBUTION);
            pluginElement.appendChild(coreContributionExtension);
            addExternalContribution(module, coreContributionExtension);

            if (!StringUtils.equalsIgnoreCase(ctx().getMavenInformation().getCategory(), ConnectorCategory.COMMUNITY.name())){
                addMavenPopulateExtensionPoint(pluginElement);
            }

            ModuleRelativePathBuilder relativePath = exportDoc(module);
            ctx().registerProduct(Product.STUDIO_PLUGIN_XML, module, relativePath);

        } catch (Exception e) {
            throw new GenerationException("Error generating Mule Studio plugin.xml", e);
        }
    }

    private void addMavenPopulateExtensionPoint(Element pluginElement) {
        Element mavenPopulateExtension = createExtensionPoint(MAVEN_POPULATE);
        pluginElement.appendChild(mavenPopulateExtension);
        addJarContribution(mavenPopulateExtension);
    }

    private void addJarContribution(Element extensionElement){
        Element contributionElement = pluginDoc.createElement(CONTRIBUTION_JAR_TAG);
        contributionElement.setAttribute("jar", ctx().getMavenInformation().getArtifactId() + "-" + ctx().getMavenInformation().getVersion() + ".jar");
        extensionElement.appendChild(contributionElement);
    }

    private ModuleRelativePathBuilder exportDoc(Module module) throws IOException, TransformerException {

        TransformerFactory transformerFactory = TransformerFactory.newInstance();
        Transformer transformer = transformerFactory.newTransformer();

        DOMSource source = new DOMSource(pluginDoc);
        ModuleRelativePathBuilder relativePath = new ModuleRelativePathBuilder(PLUGIN_XML_FILE_NAME);
        StreamResult result = new StreamResult(ctx().getCodeModel().getCodeWriter().openBinary(null, relativePath.build(module).getFullPath()));
        transformer.transform(source, result);
        return relativePath;
    }

    private void addExternalContribution(Module module, Element extensionElement) throws IOException {

        Element externalContributionElement = pluginDoc.createElement(EXTERNAL_CONTRIBUTION_TAG);
        externalContributionElement.setAttribute("contributionJar", ctx().getMavenInformation().getArtifactId() + "-" + ctx().getMavenInformation().getVersion() + ".jar");
        externalContributionElement.setAttribute("contributionLibPathInMule", "/plugins");
        externalContributionElement.setAttribute("contributionLibs", ctx().getMavenInformation().getArtifactId() + "-" + ctx().getMavenInformation().getVersion() + ".zip");
        externalContributionElement.setAttribute("contributionSources", ctx().getMavenInformation().getArtifactId() + "-" + ctx().getMavenInformation().getVersion() + "-sources.jar");
        externalContributionElement.setAttribute("contributionJavaDocs", ctx().getMavenInformation().getArtifactId() + "-" + ctx().getMavenInformation().getVersion() + "-javadoc.jar");
        externalContributionElement.setAttribute("contributionNamespace", module.getXmlNamespace());
        externalContributionElement.setAttribute("contributionNamespaceFile", module.getCurrentSchemaLocation());
        externalContributionElement.setAttribute("contributionNamespacePrefix", module.getModuleName());
        externalContributionElement.setAttribute("contributionType", "cloud-connector");
        externalContributionElement.setAttribute("path", (ctx().<ModuleRelativePathBuilder>getProduct(Product.STUDIO_EDITOR_XML, module)).build(module).getRelativeToModulePath());
        externalContributionElement.setAttribute("version", ctx().getMavenInformation().getVersion());
        externalContributionElement.setAttribute("name", module.getFriendlyName());
        externalContributionElement.setAttribute("minimumVersion", module.getMinMuleVersion().toString());
        externalContributionElement.setAttribute("contributionVersion", ctx().getMavenInformation().getVersion());

        lookUpPropertiesFile(module, extensionElement, externalContributionElement);

        extensionElement.appendChild(externalContributionElement);
    }

    private void createPluginDocument() throws ParserConfigurationException {
        DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
        DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
        pluginDoc = documentBuilder.newDocument();
    }

    /**
     * Method to override(or add) attributes of plugin.xml if META-INF/studio.properties file exists in the connector's folder.
     *
     * @param module
     * @param extensionElement
     * @param externalContributionElement
     * @throws IOException
     */
    private void lookUpPropertiesFile(Module module, Element extensionElement, Element externalContributionElement) throws IOException {
        File propertiesFile = MulePathResolver.getFileResolvingPackages(module, PROPERTIES_RELATIVE_PATH);

        if (propertiesFile.exists()){
            ctx().note("Properties file found in ["+ propertiesFile.getCanonicalPath() +"]. Keep in mind that, if any property is overlapped against the default one, the customized one will overwrite the default one.");
            FileInputStream fileInputStream = new FileInputStream(propertiesFile);
            Properties connectorProperties = new Properties();
            connectorProperties.load(fileInputStream);
            for (String propertyName : connectorProperties.stringPropertyNames()) {
                String propertyValue = connectorProperties.getProperty(propertyName);
                if (externalContributionElement.hasAttribute(propertyName)){
                    ctx().warn("The property [" + propertyName + "] was overwritten, new value is [" + propertyValue + "] (instead of default one [" + extensionElement.getAttribute(propertyName)+ "])");
                }
                externalContributionElement.setAttribute(propertyName, propertyValue);
            }
        }
    }

    private Element createExtensionPoint(String point){

        Element extensionElement = pluginDoc.createElement("extension");
        extensionElement.setAttribute("point", point);

        return extensionElement;
    }

}