/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.devkit.generation.studio;

import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.studio.packaging.ModuleRelativePathBuilder;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ProcessorMethod;
import org.mule.devkit.verifiers.util.MulePathResolver;
import org.mule.util.IOUtils;

import java.io.*;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static org.apache.commons.io.filefilter.DirectoryFileFilter.DIRECTORY;
import static org.mule.devkit.generation.studio.editor.MuleStudioUtils.*;

public class MuleStudioIconsGenerator extends AbstractMuleStudioNamespaceGenerator {

    private ArrayList<ModuleRelativePathBuilder> iconsFileName;
    private final AcceptedFilesAsIconFilter acceptedFilesAsIconFilter = new AcceptedFilesAsIconFilter();
    private static List<Product> PRODUCES = Collections.singletonList(Product.STUDIO_ICONS);

    private static final String ICONS_NEW_FOLDER = "icons/";
    private static final String LIGHT_THEME = "theme.light";
    private static final String CLASSIC_THEME = "theme.classic";
    private static final String LIGHT_THEME_FOLDER_DEST = ICONS_NEW_FOLDER + LIGHT_THEME;
    private static final String CLASSIC_THEME_FOLDER_DEST = ICONS_NEW_FOLDER + CLASSIC_THEME;
    private static final String PNG_EXTENSION = ".png";
    private static final List<String> reservedFolderNames = Arrays.asList(LIGHT_THEME, CLASSIC_THEME);

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        iconsFileName = new ArrayList<>();

        createCloudConnectorIcon(module);

        if (module.hasTransformers()) {
            createTransformerIcon(module);
        }

        if (module.hasSources()) {
            createSourceIcon(module);
        }

        if (hasContainer(module)) {
            createContainerIcon(module);
        }

        if (hasContainerList(module)) {
            createFlowIcon(module);
        }

        copyThemesFolders(module);
        ctx().registerProduct(Product.STUDIO_ICONS, module, iconsFileName);
    }

    private boolean hasContainer(Module module) {
        for (ProcessorMethod processorMethod : module.getProcessorMethods()) {
            if (processorMethod.isContainer()) {
                return true;
            }
        }
        return false;
    }

    private boolean hasContainerList(Module module) {
        for (ProcessorMethod processorMethod : module.getProcessorMethods()) {
            if (processorMethod.isContainerList()) {
                return true;
            }
        }
        return false;
    }

    private void createSourceIcon(Module module) throws GenerationException {
        copyFile(String.format(ICONS_FOLDER + CLASSIC_THEME_FOLDER + SUFFIX_ENDPOINT_LARGE_PNG, module.getModuleName()), CLASSIC_THEME_FOLDER_DEST, module, CLASSIC_THEME_FOLDER + SUFFIX_ENDPOINT_LARGE_PNG);
        copyFile(String.format(ICONS_FOLDER + CLASSIC_THEME_FOLDER + SUFFIX_ENDPOINT_SMALL_PNG, module.getModuleName()), CLASSIC_THEME_FOLDER_DEST, module, CLASSIC_THEME_FOLDER + SUFFIX_ENDPOINT_SMALL_PNG);

        copyFile(String.format(ICONS_FOLDER + LIGHT_THEME_FOLDER + SUFFIX_ENDPOINT_LARGE_PNG, module.getModuleName()), LIGHT_THEME_FOLDER_DEST, module, LIGHT_THEME_FOLDER + SUFFIX_ENDPOINT_LARGE_PNG);
        copyFile(String.format(ICONS_FOLDER + LIGHT_THEME_FOLDER + SUFFIX_ENDPOINT_SMALL_PNG, module.getModuleName()), LIGHT_THEME_FOLDER_DEST, module, LIGHT_THEME_FOLDER + SUFFIX_ENDPOINT_SMALL_PNG);
    }

    private void createTransformerIcon(Module module) throws GenerationException {
        copyFile(String.format(ICONS_FOLDER + CLASSIC_THEME_FOLDER + SUFFIX_TRANSFORMER_LARGE_PNG, module.getModuleName()), CLASSIC_THEME_FOLDER_DEST, module, CLASSIC_THEME_FOLDER + SUFFIX_TRANSFORMER_LARGE_PNG);
        copyFile(String.format(ICONS_FOLDER + CLASSIC_THEME_FOLDER + SUFFIX_TRANSFORMER_SMALL_PNG, module.getModuleName()), CLASSIC_THEME_FOLDER_DEST, module, CLASSIC_THEME_FOLDER + SUFFIX_TRANSFORMER_SMALL_PNG);

        copyFile(String.format(ICONS_FOLDER + LIGHT_THEME_FOLDER + SUFFIX_TRANSFORMER_LARGE_PNG, module.getModuleName()), LIGHT_THEME_FOLDER_DEST, module, LIGHT_THEME_FOLDER + SUFFIX_TRANSFORMER_LARGE_PNG);
        copyFile(String.format(ICONS_FOLDER + LIGHT_THEME_FOLDER + SUFFIX_TRANSFORMER_SMALL_PNG, module.getModuleName()), LIGHT_THEME_FOLDER_DEST, module, LIGHT_THEME_FOLDER + SUFFIX_TRANSFORMER_SMALL_PNG);
    }

    private void createCloudConnectorIcon(Module module) throws GenerationException {
        copyFile(String.format(ICONS_FOLDER + CLASSIC_THEME_FOLDER + SUFFIX_CONNECTOR_SMALL_PNG, module.getModuleName()), CLASSIC_THEME_FOLDER_DEST, module, CLASSIC_THEME_FOLDER + SUFFIX_CONNECTOR_SMALL_PNG);
        copyFile(String.format(ICONS_FOLDER + CLASSIC_THEME_FOLDER + SUFFIX_CONNECTOR_LARGE_PNG, module.getModuleName()), CLASSIC_THEME_FOLDER_DEST, module, CLASSIC_THEME_FOLDER + SUFFIX_CONNECTOR_LARGE_PNG);

        copyFile(String.format(ICONS_FOLDER + LIGHT_THEME_FOLDER + SUFFIX_CONNECTOR_SMALL_PNG, module.getModuleName()), LIGHT_THEME_FOLDER_DEST, module, LIGHT_THEME_FOLDER + SUFFIX_CONNECTOR_SMALL_PNG);
        copyFile(String.format(ICONS_FOLDER + LIGHT_THEME_FOLDER + SUFFIX_CONNECTOR_LARGE_PNG, module.getModuleName()), LIGHT_THEME_FOLDER_DEST, module, LIGHT_THEME_FOLDER + SUFFIX_CONNECTOR_LARGE_PNG);
    }

    private void createContainerIcon(Module module) throws GenerationException {
        copyFile(String.format(ICONS_FOLDER + CLASSIC_THEME_FOLDER + SUFFIX_CONTAINER_LARGE_PNG, module.getModuleName()), CLASSIC_THEME_FOLDER_DEST, module, CLASSIC_THEME_FOLDER + SUFFIX_CONTAINER_LARGE_PNG);
        copyFile(String.format(ICONS_FOLDER + CLASSIC_THEME_FOLDER + SUFFIX_CONTAINER_SMALL_PNG, module.getModuleName()), CLASSIC_THEME_FOLDER_DEST, module, CLASSIC_THEME_FOLDER + SUFFIX_CONTAINER_SMALL_PNG);

        copyFile(String.format(ICONS_FOLDER + LIGHT_THEME_FOLDER + SUFFIX_CONTAINER_LARGE_PNG, module.getModuleName()), LIGHT_THEME_FOLDER_DEST, module, LIGHT_THEME_FOLDER + SUFFIX_CONTAINER_LARGE_PNG);
        copyFile(String.format(ICONS_FOLDER + LIGHT_THEME_FOLDER + SUFFIX_CONTAINER_SMALL_PNG, module.getModuleName()), LIGHT_THEME_FOLDER_DEST, module, LIGHT_THEME_FOLDER + SUFFIX_CONTAINER_SMALL_PNG);
    }

    private void createFlowIcon(Module module) throws GenerationException {
        copyFile(String.format(ICONS_FOLDER + CLASSIC_THEME_FOLDER + SUFFIX_FLOW_LARGE_PNG, module.getModuleName()), CLASSIC_THEME_FOLDER_DEST, module, CLASSIC_THEME_FOLDER + SUFFIX_FLOW_LARGE_PNG);
        copyFile(String.format(ICONS_FOLDER + CLASSIC_THEME_FOLDER + SUFFIX_FLOW_SMALL_PNG, module.getModuleName()), CLASSIC_THEME_FOLDER_DEST, module, CLASSIC_THEME_FOLDER + SUFFIX_FLOW_SMALL_PNG);

        copyFile(String.format(ICONS_FOLDER + LIGHT_THEME_FOLDER + SUFFIX_FLOW_LARGE_PNG, module.getModuleName()), LIGHT_THEME_FOLDER_DEST, module, LIGHT_THEME_FOLDER + SUFFIX_FLOW_LARGE_PNG);
        copyFile(String.format(ICONS_FOLDER + LIGHT_THEME_FOLDER + SUFFIX_FLOW_SMALL_PNG, module.getModuleName()), LIGHT_THEME_FOLDER_DEST, module, LIGHT_THEME_FOLDER + SUFFIX_FLOW_SMALL_PNG);
    }

    private void copyFile(String fileName, String folder, Module module, String sufix) throws GenerationException {
        InputStream iconStream = null;
        OutputStream outputStream = null;
        try {
            outputStream = createFile(module, folder, fileName);
            File fileToCopy = MulePathResolver.getFileResolvingPackages(module, fileName);
            if (fileToCopy.exists()) {
                //taking the icons from the connectors/icons folder
                iconStream = new FileInputStream(fileToCopy);
            } else {
                //using the default ones
                String fileIconPath = String.format(sufix, "generic");
                iconStream = MuleStudioIconsGenerator.class.getClassLoader().getResourceAsStream(fileIconPath);
            }
            IOUtils.copy(iconStream, outputStream);
        } catch (IOException e) {
            throw new GenerationException("Error copying icons to output folder: " + e.getMessage(), e);
        } finally {
            IOUtils.closeQuietly(outputStream);
            IOUtils.closeQuietly(iconStream);
        }
    }

    private OutputStream createFile(Module module, String folder, String fileName) throws GenerationException {
        if (fileName.contains("/")) {
            fileName = fileName.substring(fileName.lastIndexOf("/") + 1);
        }
        try {
            ModuleRelativePathBuilder pathBuilder = new ModuleRelativePathBuilder(folder + "/" + fileName);
            String fullIconPath = pathBuilder.build(module).getFullPath();
            iconsFileName.add(pathBuilder);
            return ctx().getCodeModel().getCodeWriter().openBinary(null, fullIconPath);
        } catch (IOException e) {
            throw new GenerationException("Could not create file or folder " + fileName + ": " + e.getMessage(), e);
        }
    }

    private void copyThemesFolders(Module module) throws GenerationException {

        final File iconsFolder = MulePathResolver.getFileResolvingPackages(module, ICONS_FOLDER);
        if(iconsFolder.exists()){
            final File[] themeFolders = iconsFolder.listFiles((FileFilter) DIRECTORY);

            for (final File themeFolder : themeFolders) {
                if (!isReservedFolderName(themeFolder)) {
                    for (File file : themeFolder.listFiles(acceptedFilesAsIconFilter)) {
                        copyFile(ICONS_FOLDER + themeFolder.getName() + "/" + file.getName(), ICONS_NEW_FOLDER + themeFolder.getName(), module, "");
                    }
                }
            }
        }
    }

    private class AcceptedFilesAsIconFilter implements FileFilter {

        @Override
        public boolean accept(File file) {
            return file.isFile() && file.canRead() &&
                    file.getName().endsWith(PNG_EXTENSION);
        }
    }

    private boolean isReservedFolderName(File themeFolder) {
        return reservedFolderNames.contains(themeFolder.getName());
    }
}