/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.generation.studio;

import com.google.common.base.Optional;
import org.apache.commons.lang.StringUtils;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.ModuleGenerator;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.apidoc.VelocityUtils;
import org.mule.devkit.generation.studio.packaging.ModuleRelativePathBuilder;
import org.mule.devkit.model.apidoc.Element;
import org.mule.devkit.model.apidoc.ModuleModel;
import org.mule.devkit.model.code.writer.FilerCodeWriter;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.devkit.utils.NameUtils;

import java.io.IOException;
import java.io.OutputStream;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Paths;
import java.util.*;

public class MuleStudioContextualHelpGenerator extends AbstractMuleStudioGenerator implements ModuleGenerator {

    private final List<Product> CONSUMES = Arrays.asList(Product.APIDOC_MODEL);

    private org.mule.devkit.model.module.Module module;
    private List<ModuleRelativePathBuilder> pathBuilders = new ArrayList<ModuleRelativePathBuilder>();
    private final String DOC = "doc";

    @Override
    public boolean shouldGenerate(org.mule.devkit.model.module.Module module) {
        return module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC;
    }

    @Override
    public void generate(org.mule.devkit.model.module.Module module) throws GenerationException {
        this.module = module;
        VelocityEngine engine = VelocityUtils.getVelocityEngine();
        try {
            generateIndex(engine);
            generateElements(engine, getApidocModel().configs(), DOC + "/config");
            generateElements(engine, getApidocModel().processors(), DOC + "/processor");
            generateElements(engine, getApidocModel().transformers(), DOC + "/transformer");
            generateElements(engine, getApidocModel().filters(), DOC + "/filter");
            generateElements(engine, getApidocModel().sources(), DOC + "/source");
            ctx().registerProduct(Product.STUDIO_CONTEXTUAL_HELP, module, pathBuilders);
        } catch (IOException e) {
            throw new GenerationException("Error while generating Contextual Help files: " + e.getMessage(), e);
        }
    }

    private void generateIndex(VelocityEngine engine) throws IOException {
        Template indexTemplate = engine.getTemplate("studio-ch-index.vm");
        VelocityContext indexContext = VelocityUtils.getContextBuilder()
                                            .put(defaultContextProperties())
                                            .put("element", getApidocModel()).build();
        generateElement(indexTemplate, indexContext, generateFullPath(DOC, "index"));
    }

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    private void generateElements(VelocityEngine engine, List<? extends Element> elements, String folderName)
            throws IOException
    {
        Template elementIndexTemplate = engine.getTemplate("studio-ch-element-index.vm");
        Template elementTemplate = engine.getTemplate("studio-ch-element.vm");
        for (Element element : elements) {
            VelocityContext context = VelocityUtils.getContextBuilder()
                                        .put(defaultContextProperties())
                                        .put("element", element)
                                        .build();
            generateElement(elementTemplate, context, generateFullPath(folderName, element.xsdName()));
        }
        if(!elements.isEmpty()) {
            VelocityContext context = VelocityUtils.getContextBuilder()
                                        .put(defaultContextProperties())
                                        .put("elements", elements)
                                        .put("kind", Paths.get(folderName).getFileName().toString())
                                        .build();
            generateElement(elementIndexTemplate, context, generateFullPath(folderName, "index"));
        }
    }

    private void generateElement(Template template, VelocityContext context, String fullPath)
            throws IOException
    {
        StringWriter writer = new StringWriter();
        template.merge(context, writer);
        OutputStream os = null;
        try {
            os = new FilerCodeWriter(ctx().getFiler()).openBinary(null, fullPath);
            os.write(writer.toString().getBytes());
        } finally {
            if (os != null) {
                os.close();
            }
        }
    }

    private String generateFullPath(String folderName, String fileName)  {
        fileName = (!StringUtils.isBlank(fileName))? fileName + ".html" : fileName;
        ModuleRelativePathBuilder pathBuilder = new ModuleRelativePathBuilder(Paths.get(folderName, fileName).toString());
        pathBuilders.add(pathBuilder);
        return pathBuilder.build(module).getFullPath();
    }

    private ModuleModel getApidocModel(){
        return ctx().getApidocModel();
    }

    private Map<String, Object> defaultContextProperties(){
        Map<String, Object> props = new HashMap<>();
        props.put("docUrl", repoDocUrl());
        props.put("connectorVersion", ctx().getMavenInformation().getVersion());
        props.put("connectorName", ctx().getMavenInformation().getName());
        props.put("NameUtils", NameUtils.class);
        props.put("StringUtils", StringUtils.class);
        return props;
    }

    private String repoDocUrl(){
        String path;
        try {
            path = new URL(ctx().getMavenInformation().getUrl()).getPath();
        } catch (MalformedURLException e) {
            return "";
        }
        Optional<String> owner = Optional.absent();
        Optional<String> name = Optional.absent();
        for (String segment : path.split("/"))
            if (segment.length() > 0)
                if (!owner.isPresent())
                    owner = Optional.fromNullable(segment);
                else if (!name.isPresent())
                    name = Optional.fromNullable(segment);
                else
                    break;

        return name.get().equals("mule-devkit-parent")? "" : String.format("http://%s.github.io/%s", owner.get(), name.get());
    }
}
