/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.devkit.generation.studio.packaging;

import java.io.File;
import java.io.IOException;
import java.util.List;

import org.codehaus.plexus.archiver.ArchiverException;
import org.codehaus.plexus.archiver.jar.JarArchiver;

public class PackagePlanExecutor {

    private RelativePathMap relativePathMap;
    private SignComponent signComponent;

    public static class RelativePathMap implements Placeholder.PlaceholderMap<File> {

        private String rootPath;

        public RelativePathMap(String rootPath) {
            this.rootPath = rootPath;
        }

        public RelativePathMap(File rootPath) {
            this.rootPath = rootPath.getAbsolutePath();
        }

        @Override
        public File visitPlugin(Placeholder placeholder) {
            return new File(Placeholder.UPDATE_SITE_DIRECTORY.visit(this), "plugins");
        }

        @Override
        public File visitFeature(Placeholder placeholder) {
            return new File(Placeholder.UPDATE_SITE_DIRECTORY.visit(this), "features");
        }

        @Override
        public File vistClassesDirectory(Placeholder placeholder) {
            return new File(this.rootPath, "classes");
        }

        @Override
        public File visitUpdateSite(Placeholder placeholder) {
            return new File(this.rootPath, "update-site");
        }

        @Override
        public File visitOutputDirectory(Placeholder placeholder) {
            return new File(Placeholder.GENERATED_SOURCES_DIRECTORY.visit(this), "mule");
        }

        @Override
        public File visitGeneratedSources(Placeholder placeholder) {
            return new File(this.rootPath, "generated-sources");
        }

        @Override
        public File visitRootDirectory(Placeholder placeholder) {
            return new File(this.rootPath);
        }

    }

    public PackagePlanExecutor(RelativePathMap relativePathMap, SignComponent signComponent) {
        this.relativePathMap = relativePathMap;
        this.signComponent = signComponent;
    }

    public File fileOrFail(FutureFile maybeFile, RelativePathMap relativePathMap) {
        File f = file(maybeFile, relativePathMap);
        if (!f.exists()) {
            throw new IllegalArgumentException("Error: Path does not exist: " + f);
        }
        return f;
    }

    public File file(FutureFile maybeFile, RelativePathMap relativePathMap) {
        return new File(maybeFile.getFirst().visit(relativePathMap), maybeFile.getSecond());
    }


    public void execute(ArchiverPlan archiverPlan) {
        JarArchiver archiver = new JarArchiver();
        FutureFile manifest = archiverPlan.getManifest();
        FutureFile destFutureFile = archiverPlan.getDestFile();
        List<Tuple<FutureFile, String>> files = archiverPlan.getFiles();
        List<ArchiverPlan> children = archiverPlan.getChildren();
        List<FutureFile> directoriesToAdd = archiverPlan.getDirectoriesToAdd();

        if (destFutureFile == null || files.size() == 0) {
            throw new RuntimeException("Invalid Promise Jar Archiver");
        }

        if (manifest != null) {
            try {
                File file = file(manifest,relativePathMap);
                if (file.exists()) archiver.setManifest(file);
            } catch (ArchiverException e) {
                throw new RuntimeException(e);
            }
        }

        for (ArchiverPlan archiverPlanChild : children) {
            this.execute(archiverPlanChild);
        }

        try {
            for (Tuple<FutureFile, String> tuple : files) {
                File file = file(tuple.getFirst(), relativePathMap);
                if (file.exists()) archiver.addFile(file, tuple.getSecond());
            }

            for (FutureFile futureFile : directoriesToAdd) {
                File dir = file(futureFile,relativePathMap);
                if (dir.exists()) archiver.addDirectory(dir);
            }

            File destFile = file(destFutureFile, relativePathMap);
            archiver.setDestFile(destFile);

            archiver.createArchive();

            if (signComponent != null) {
                signComponent.sign(destFile);
            }

        } catch (ArchiverException e) {
            throw new RuntimeException(e);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

    }


}
