/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */


package org.mule.devkit.generation.studio.editor.globalcloudconnector;

import org.mule.api.annotations.ConnectStrategy;
import org.mule.api.annotations.TestConnectivity;
import org.mule.api.annotations.display.UserDefinedMetaData;
import org.mule.api.callback.HttpCallback;
import org.mule.devkit.api.transformer.TransformingValue;
import org.mule.devkit.generation.api.Context;
import org.mule.devkit.generation.studio.editor.CacheConfigNestedElementBuilder;
import org.mule.devkit.generation.studio.editor.GlobalTypeBuilder;
import org.mule.devkit.generation.studio.editor.MuleStudioEditorXmlGenerator;
import org.mule.devkit.generation.studio.editor.PoolingProfileNestedElementBuilder;
import org.mule.devkit.generation.studio.editor.ReconnectionNestedElementBuilder;
import org.mule.devkit.generation.studio.editor.callback.HttpCallbackNestedElementBuilder;
import org.mule.devkit.generation.studio.editor.callback.OAuthConfigNestedElementsBuilder;
import org.mule.devkit.generation.studio.utils.JaxbUnmarshaller;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.Variable;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ProcessorMethod;
import org.mule.devkit.model.module.connectivity.ConnectMethod;
import org.mule.devkit.model.studio.AttributeCategory;
import org.mule.devkit.model.studio.AttributeType;
import org.mule.devkit.model.studio.EnumElement;
import org.mule.devkit.model.studio.EnumType;
import org.mule.devkit.model.studio.GlobalType;
import org.mule.devkit.model.studio.Group;
import org.mule.devkit.model.studio.MetaDataType;
import org.mule.devkit.model.studio.NamespaceType;
import org.mule.devkit.model.studio.NestedElementReference;
import org.mule.devkit.model.studio.libraries.RequiredLibraries;
import org.mule.util.StringUtils;

import com.google.common.base.Optional;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.xml.bind.JAXBElement;

public abstract class GlobalCloudConnectorTypeBuilder extends GlobalTypeBuilder {

    public static final String REQUIRED_LIBS_DIR = "src/main/resources/META-INF";
    public static final String REQUIRED_LIBS_SUFFIX = "-required-libs.xml";
    public static final String STUDIO_FULL_QUALIFIED_PACKAGE_VALUE_PERSISTANT = "org.mule.tooling.ui.modules.core.widgets.meta.";
    public static final String STUDIO_POOLING_PROFILE_PERSISTANT_VALUE = STUDIO_FULL_QUALIFIED_PACKAGE_VALUE_PERSISTANT + "PoolingProfileValuePersistence";
    public static final String STUDIO_CACHE_CONFIGURATION_PERSISTANT_VALUE = STUDIO_FULL_QUALIFIED_PACKAGE_VALUE_PERSISTANT + "CacheConfigurationValuePersistence";
    public static final String STUDIO_HTTP_TRANSACTION_PERSISTANT_VALUE = STUDIO_FULL_QUALIFIED_PACKAGE_VALUE_PERSISTANT + "HTTPTransactionValuePersistence";

    private boolean abstrac7;
    private String globalRefId;

    public GlobalCloudConnectorTypeBuilder(Context context, Module module, boolean abstrac7, String globalRefId) {
        super(context, module);
        this.abstrac7 = abstrac7;
        this.globalRefId = globalRefId == null ? helper.getGlobalRefId(module.getModuleName()) : globalRefId;
    }

    /**
     * Overriding the build behaviour to specify each global-cloud-connector element.
     * @return
     */
    @Override
    public GlobalType build() {
        GlobalType globalCloudConnector = super.build();
        buildConnectivityTestingAttr(globalCloudConnector);
        buildMetaDataAttr(globalCloudConnector, module.hasDynamicMetadata());
        buildRequiredLibsAttr(globalCloudConnector);
        buildSupportUserDefinedMetaData(globalCloudConnector, module);
        globalCloudConnector.setVersions(buildVersionsString());

        return globalCloudConnector;
    }

    /**
     * Checks if it's needed to put the flag supportUserDefinedMetaData to true, in case any @Processor returns a
     * {@link TransformingValue}
     * @param globalCloudConnector to set supportUserDefinedMetaData to true if applies
     * @param module to iterate over its processors checking for the return type
     */
    protected void buildSupportUserDefinedMetaData(GlobalType globalCloudConnector, Module module){
        globalCloudConnector.setSupportsUserDefinedMetaData(hasProcessorsWithTransformingValue(module));
    }

    /**
     * @param module
     * @return true if at least a @Processor returns {@link TransformingValue} and one of them has been annotated with
     * UserDefinedMetaData. False otherwise
     */
    private boolean hasProcessorsWithTransformingValue(Module module) {
        for (ProcessorMethod processorMethod : module.getProcessorMethods()) {
            if (processorMethod.hasAnnotation(UserDefinedMetaData.class)){
                return true;
            }
        }
        return false;
    }

    /**
     * Populate the GlobalType with the required libraries described in the xml
     * with the same configElementName under path <root>/META-INF
     *
     * @param globalCloudConnector to set required libraries to
     */
    private void buildRequiredLibsAttr(GlobalType globalCloudConnector){
        final String fileName = moduleName + "-" + globalCloudConnector.getLocalId() + REQUIRED_LIBS_SUFFIX;

        File rootLibrariesDir = new File(context.getMavenInformation().getBuildDirectory(), REQUIRED_LIBS_DIR);
        File requiredLibrariesXml = new File(rootLibrariesDir, fileName);
        if (requiredLibrariesXml.exists()){
            loadRequiredLibraries(globalCloudConnector, requiredLibrariesXml);
        }
    }

    private void loadRequiredLibraries(GlobalType globalCloudConnector, File libXml) {
        JAXBElement<NamespaceType> root = JaxbUnmarshaller.unmarshallXml(libXml.getPath(), NamespaceType.class);
        if (root.getValue().getRequiredLibs() != null) {
            RequiredLibraries libs = root.getValue().getRequiredLibs().getValue();
            if (libs != null && libs.hasElements()) {
                globalCloudConnector.setRequiredLibs(libs);
            }
        }
    }

    /**
     * Determines weather the current global element supports connectivity testing on Studio, or not
     *
     * @param globalCloudConnector to set the connectivity testing to
     */
    protected abstract void buildConnectivityTestingAttr(GlobalType globalCloudConnector);

    /**
     * Determines weather the current global element supports static or dynamic DataSense in Studio
     *
     * @param globalCloudConnector
     * @param dynamicMetadata indicates if the current module supports dynamic metadata ( check {@link org.mule.devkit.model.module.Module#hasDynamicMetadata()})
     */
    protected void buildMetaDataAttr(GlobalType globalCloudConnector, boolean dynamicMetadata){
        String metaDataSwitch = dynamicMetadata ? "dynamic" : "static";
        globalCloudConnector.setMetaData(metaDataSwitch);
        globalCloudConnector.setMetaDataKeyDisplay("DISPLAY_NAME");
    }

    protected void setConnectivityTestingAttribute(GlobalType globalCloudConnector, Optional<Method> methodOptional) {
        boolean active = false;
        String label = null;
        if (methodOptional.isPresent()){
            TestConnectivity testConnectivity = methodOptional.get().getAnnotation(TestConnectivity.class);
            active = testConnectivity.active();
            label = testConnectivity.label();
        }
        setConnectivityTestingAttribute(globalCloudConnector, active, label);
    }

    private void setConnectivityTestingAttribute(GlobalType globalCloudConnector, boolean hasConnectivityTesting, String label) {
        // DEVKIT-558: Connectivity Testing only works starting in Mule 3.4.0
        if (!module.getMinMuleVersion().atLeastBase("3.4")) {
            hasConnectivityTesting = false;
        }
        if (hasConnectivityTesting){
            globalCloudConnector.setConnectivityTesting("on");
            globalCloudConnector.setConnectivityTestingLabel(StringUtils.isBlank(label)? TestConnectivity.TEST_CONNECTION : label);
        }else{
            globalCloudConnector.setConnectivityTesting("off");
        }
    }

    @Override
    protected List<AttributeCategory> getAttributeCategories() {
        Group group = new Group();
        group.setId(moduleName + "GenericProperties");
        group.getRegexpOrEncodingOrModeSwitch().add(objectFactory.createGroupName(createNameAttributeType()));
        group.setCaption(helper.formatCaption(MuleStudioEditorXmlGenerator.GROUP_DEFAULT_CAPTION));
        return processConfigurableFields(group);
    }

    @Override
    protected void createMetaDataAttributes(Map<String,Group> groupsByName, Map<String,AttributeCategory> attributeCategoriesByName) {
        if (module.hasDynamicMetadata()) {
            Group defaultGroup = groupsByName.get(GENERAL_GROUP_NAME);
            if (defaultGroup == null) {
                defaultGroup = new Group();
                defaultGroup.setCaption(helper.formatCaption(GENERAL_GROUP_NAME));
                defaultGroup.setId(StringUtils.uncapitalize(GENERAL_GROUP_NAME));
                groupsByName.put(GENERAL_GROUP_NAME, defaultGroup);

                if (attributeCategoriesByName.get(GENERAL_GROUP_NAME) == null) {
                    attributeCategoriesByName.put(GENERAL_GROUP_NAME, new AttributeCategory());
                    attributeCategoriesByName.get(GENERAL_GROUP_NAME).setCaption(GENERAL_GROUP_NAME);
                    attributeCategoriesByName.get(GENERAL_GROUP_NAME).getGroup().add(defaultGroup);
                } else {
                    attributeCategoriesByName.get(GENERAL_GROUP_NAME).getGroup().add(defaultGroup);
                }
            }

            /**
             * <useMetaData>name="useMetaData" caption="useMetaData"</useMetaData>
             */

            MetaDataType metaDataType = new MetaDataType();
            metaDataType.setName("useMetaData");
            metaDataType.setCaption("useMetaData");

            defaultGroup.getRegexpOrEncodingOrModeSwitch().add(helper.createJAXBElement(metaDataType));

        }
    }

    @Override
    protected void processConnectionAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        List<AttributeType> connAttrbs= getConnectionAttributes();
        if (!connAttrbs.isEmpty()){
            Group connectionAttributesGroup = new Group();
            connectionAttributesGroup.setCaption(helper.formatCaption(CONNECTION_GROUP_NAME));
            connectionAttributesGroup.setId(StringUtils.uncapitalize(CONNECTION_GROUP_NAME));

            groupsByName.put(CONNECTION_GROUP_NAME, connectionAttributesGroup);
            connectionAttributesGroup.getRegexpOrEncodingOrModeSwitch().addAll(helper.createJAXBElements(connAttrbs));

            AttributeCategory defaultAttributeCategory = attributeCategoriesByName.get(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION);
            defaultAttributeCategory.getGroup().add(connectionAttributesGroup);
        }
    }

    /**
     * Returns the list of elements that should be within the CONNECTION_GROUP_NAME
     * @return
     */
    protected abstract List<AttributeType> getConnectionAttributes();

    @Override
    protected  void createPoolingProfileAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        if (hasConnectionMethod() && connectMethod().getStrategy().equals(ConnectStrategy.MULTIPLE_INSTANCES)) {
            AttributeCategory poolProfileTab = new AttributeCategory();
            poolProfileTab.setCaption("Pooling Profile");
            poolProfileTab.setDescription("Pooling profile options.");

            Group poolingProfileGroup = new Group();
            poolingProfileGroup.setCaption("Pooling Profile");
            poolingProfileGroup.setId("poolingProfile");

            NestedElementReference childElement = new NestedElementReference();
            childElement.setName(MuleStudioEditorXmlGenerator.URI_PREFIX + moduleName + "/" + PoolingProfileNestedElementBuilder.POOLING_PROFILE_ELEMENT);
            childElement.setCaption("");
            childElement.setDescription("");
            childElement.setInplace(true);
            childElement.setValuePersistence(STUDIO_POOLING_PROFILE_PERSISTANT_VALUE);
            poolingProfileGroup.getRegexpOrEncodingOrModeSwitch().add(objectFactory.createGroupChildElement(childElement));

            poolProfileTab.getGroup().add(poolingProfileGroup);
            attributeCategoriesByName.put("Pooling Profile", poolProfileTab);
        }
    }

    @Override
    protected  void createCacheConfigAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        if (hasConnectionMethod()
                && connectMethod().getStrategy().equals(ConnectStrategy.SINGLE_INSTANCE)) {
            AttributeCategory cacheConfigTab = new AttributeCategory();
            cacheConfigTab.setCaption("Cache Configuration");
            cacheConfigTab.setDescription("Cache configuration options.");

            Group cacheConfigGroup = new Group();
            cacheConfigGroup.setCaption("Cache Configuration");
            cacheConfigGroup.setId("cacheConfig");

            NestedElementReference childElement = new NestedElementReference();
            childElement.setName(MuleStudioEditorXmlGenerator.URI_PREFIX + moduleName + "/" + CacheConfigNestedElementBuilder.CACHE_CONFIG_ELEMENT);
            childElement.setCaption("");
            childElement.setDescription("");
            childElement.setInplace(true);
            childElement.setValuePersistence(STUDIO_CACHE_CONFIGURATION_PERSISTANT_VALUE);
            cacheConfigGroup.getRegexpOrEncodingOrModeSwitch().add(objectFactory.createGroupChildElement(childElement));

            cacheConfigTab.getGroup().add(cacheConfigGroup);
            attributeCategoriesByName.put("Cache Config", cacheConfigTab);
        }
    }

    @Override
    protected void createReconnectionAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        if (hasConnectionMethod()) {
            AttributeCategory reconnectionTab = new AttributeCategory();
            reconnectionTab.setCaption("Reconnection");
            reconnectionTab.setDescription("Reconnection options.");

            Group strategiesGroup = new Group();
            strategiesGroup.setCaption("Strategies");
            strategiesGroup.setId("reconnectionStrategies");

            NestedElementReference childElement = new NestedElementReference();
            childElement.setName(MuleStudioEditorXmlGenerator.URI_PREFIX + "core/" + ReconnectionNestedElementBuilder.RECONNECTION_STRATEGY_ELEMENT);
            childElement.setValuePersistence(STUDIO_HTTP_TRANSACTION_PERSISTANT_VALUE);
            childElement.setCaption("");
            childElement.setDescription("");
            childElement.setInplace(true);

            strategiesGroup.getRegexpOrEncodingOrModeSwitch().add(objectFactory.createGroupChildElement(childElement));

            childElement = new NestedElementReference();
            childElement.setName(MuleStudioEditorXmlGenerator.URI_PREFIX + "core/" + ReconnectionNestedElementBuilder.ABSTRACT_RECONNECTION_STRATEGY);
            childElement.setCaption("");
            childElement.setDescription("");
            childElement.setAllowSubTypes(true);
            childElement.setVisibleInDialog(false);
            childElement.setAllowedSubTypes("http://www.mulesoft.org/schema/mule/core/reconnect,http://www.mulesoft.org/schema/mule/core/reconnect-custom-strategy,http://www.mulesoft.org/schema/mule/core/reconnect-forever");

            strategiesGroup.getRegexpOrEncodingOrModeSwitch().add(objectFactory.createGroupChildElement(childElement));

            reconnectionTab.getGroup().add(strategiesGroup);
            attributeCategoriesByName.put("Reconnection", reconnectionTab);
        }
    }

    /**
     * For the given element, this method will create the OAuth tab in the studio configuration properties.
     * It should be called iff the current global element is working on a {@link org.mule.api.annotations.oauth.OAuth2}
     *
     * @param groupsByName
     * @param attributeCategoriesByName
     */
    protected void buildOAuthConfig(Map<String,Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        AttributeCategory oauthAttributeCategory = new AttributeCategory();
        oauthAttributeCategory.setCaption(helper.formatCaption("OAuth"));
        oauthAttributeCategory.setDescription(helper.formatDescription("OAuth."));

        Group oauthCallbackConfigGroup = helper.createGroup("oauthCallbackConfigGroup", "OAuth callback config");
        oauthAttributeCategory.getGroup().add(oauthCallbackConfigGroup);

        NestedElementReference oauthCallbackConfigChildElement = new NestedElementReference();
        oauthCallbackConfigChildElement.setName(MuleStudioEditorXmlGenerator.URI_PREFIX + moduleName + "/" + OAuthConfigNestedElementsBuilder.OAUTH_CALLBACK_CONFIG_ELEMENT);
        oauthCallbackConfigChildElement.setCaption("");
        oauthCallbackConfigChildElement.setDescription("");
        oauthCallbackConfigChildElement.setInplace(true);
        oauthCallbackConfigGroup.getRegexpOrEncodingOrModeSwitch().add(objectFactory.createGroupChildElement(oauthCallbackConfigChildElement));

        Group oauthStoreConfigGroup = helper.createGroup("oauthStoreConfigGroup", "OAuth object store config");
        oauthAttributeCategory.getGroup().add(oauthStoreConfigGroup);

        NestedElementReference oauthStoreConfigChildElement = new NestedElementReference();
        oauthStoreConfigChildElement.setName(MuleStudioEditorXmlGenerator.URI_PREFIX + moduleName + "/" + OAuthConfigNestedElementsBuilder.OAUTH_STORE_CONFIG_ELEMENT);
        oauthStoreConfigChildElement.setCaption("");
        oauthStoreConfigChildElement.setDescription("");
        oauthStoreConfigChildElement.setInplace(true);
        oauthStoreConfigGroup.getRegexpOrEncodingOrModeSwitch().add(objectFactory.createGroupChildElement(oauthStoreConfigChildElement));

        attributeCategoriesByName.put("OAuth", oauthAttributeCategory);
        if(module.getMinMuleVersion().atLeast("3.5")) {
            Group defaultGroup = groupsByName.get(CONNECTION_GROUP_NAME);
            if (defaultGroup == null) {
                defaultGroup = new Group();
                defaultGroup.setCaption(helper.formatCaption(CONNECTION_GROUP_NAME));
                defaultGroup.setId(StringUtils.uncapitalize(CONNECTION_GROUP_NAME));
                groupsByName.put(CONNECTION_GROUP_NAME, defaultGroup);

                if (attributeCategoriesByName.get(CONNECTION_GROUP_NAME) == null) {
                    attributeCategoriesByName.put(CONNECTION_GROUP_NAME, new AttributeCategory());
                    attributeCategoriesByName.get(CONNECTION_GROUP_NAME).setCaption(CONNECTION_GROUP_NAME);
                    attributeCategoriesByName.get(CONNECTION_GROUP_NAME).getGroup().add(defaultGroup);
                } else {
                    attributeCategoriesByName.get(CONNECTION_GROUP_NAME).getGroup().add(defaultGroup);
                }
            }

            EnumType onNoTokenEnum = new EnumType();
            onNoTokenEnum.setCaption(helper.formatCaption("On No Token"));
            onNoTokenEnum.setDescription(helper.formatDescription("Action to take when no token is found"));
            onNoTokenEnum.setName("onNoToken");
            onNoTokenEnum.setXsdType("string");
            onNoTokenEnum.setAllowsCustom(false);
            onNoTokenEnum.setDefaultValue("EXCEPTION");
            EnumElement el1 = new EnumElement();
            el1.setValue("EXCEPTION");
            onNoTokenEnum.getOption().add(el1);
            EnumElement el2 = new EnumElement();
            el2.setValue("STOP_FLOW");
            onNoTokenEnum.getOption().add(el2);

            defaultGroup.getRegexpOrEncodingOrModeSwitch().add(helper.createJAXBElement(onNoTokenEnum));
        }
    }

    @Override
    protected String getExtendsBasedOnType() {
        return MuleStudioEditorXmlGenerator.URI_PREFIX + module.getModuleName() + '/' + globalRefId;
    }

    @Override
    protected String getNameDescriptionBasedOnType() {
        return helper.formatDescription("Give a name to this configuration so it can be later referenced by config-ref.");
    }

    @Override
    protected String getImage() {
        return helper.getConnectorImage(module);
    }

    @Override
    protected String getIcon() {
        return helper.getConnectorIcon(module);
    }

    @Override
    protected Boolean isAbstract() {
        return abstrac7;
    }

    @Override
    protected void setMetadataAttributes(Variable<?> variable, AttributeType attributeType) {
        if (module.hasDynamicMetadata()) {
            if(!variable.isOptional() && !variable.hasDefaultValue()) {
                attributeType.setRequiredForDataSense(Boolean.TRUE);
            }
        }
    }

    protected List<AttributeType> getConnectionAttributes( List<Field> configurableFields) {
        List<AttributeType> parameters = new ArrayList<AttributeType>();
        for (Field configurableField : configurableFields) {
            parameters.add(getAttributeType(configurableField));
        }
        return parameters;
    }

    /**
     * We override this at global element, as each type of connection strategy might have to specify if the current
     * element is required. The only change here is that each global element might have the required to true.
     * @param variable
     * @return
     */
    @Override
    protected AttributeType getAttributeType(Variable variable) {
        AttributeType parameter = super.getAttributeType(variable);
        setRequiredConnectionParameter(parameter, variable);
        return parameter;
    }

    /**
     * @param defaultGroup
     * @return
     */
    protected List<AttributeCategory> processConfigurableFields(Group defaultGroup) {
        Map<String, AttributeCategory> attributeCategoriesByName = processVariableElements(getConfigurableFieldsSorted());
        List<AttributeCategory> attributeCategories = new ArrayList<AttributeCategory>(attributeCategoriesByName.values());

        AttributeCategory attributeCategory = attributeCategoriesByName.get(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION);
        if (attributeCategory != null) {
            attributeCategory.setDescription(helper.formatDescription(module.getModuleName() + " configuration properties"));
            List<Group> groups = attributeCategory.getGroup();
            if (groups.isEmpty()) {
                groups.add(defaultGroup);
            } else {
                groups.add(0, defaultGroup);
            }
        } else {
            attributeCategory = new AttributeCategory();
            attributeCategory.setCaption(helper.getFormattedCaption(module));
            attributeCategory.setDescription(helper.formatDescription(module.getModuleName() + " configuration properties"));
            attributeCategory.getGroup().add(defaultGroup);
            attributeCategories.add(attributeCategory);
        }
        return attributeCategories;
    }

    @Override
    protected void createHttpCallbackConfig(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName){
        if (module.hasProcessorMethodWithParameter(HttpCallback.class)){
            AttributeCategory httpCallbackAttributeCategory = new AttributeCategory();
            httpCallbackAttributeCategory.setCaption(helper.formatCaption("Http Callback"));
            httpCallbackAttributeCategory.setDescription(helper.formatDescription("Http Callback."));

            Group httpCallbackConfigGroup = helper.createGroup("httpCallbackConfigGroup", "Http callback config");
            httpCallbackAttributeCategory.getGroup().add(httpCallbackConfigGroup);

            NestedElementReference httpCallbackConfigChildElement = new NestedElementReference();
            httpCallbackConfigChildElement.setName(MuleStudioEditorXmlGenerator.URI_PREFIX + moduleName + "/" + HttpCallbackNestedElementBuilder.CALLBACK_CONFIG_ELEMENT);
            httpCallbackConfigChildElement.setCaption("");
            httpCallbackConfigChildElement.setDescription("");
            httpCallbackConfigChildElement.setInplace(true);
            httpCallbackConfigGroup.getRegexpOrEncodingOrModeSwitch().add(objectFactory.createGroupChildElement(httpCallbackConfigChildElement));

            attributeCategoriesByName.put("Http Callback", httpCallbackAttributeCategory);
        }
    }

    /**
     * Useful to mark in Studio that the current parameter should be written in the global element
     * @param attributeType
     * @param variable
     */
    protected void setRequiredConnectionParameter(AttributeType attributeType, Variable variable) {
        attributeType.setRequired(!variable.isOptional());
    }

    protected abstract List<Field> getConfigurableFieldsSorted() ;

    protected abstract boolean hasConnectionMethod();

    protected abstract ConnectMethod connectMethod();

}