/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.generation.studio;

import org.mule.devkit.generation.api.AnnotationVerificationException;
import org.mule.devkit.generation.api.MultiModuleAnnotationVerifier;
import org.mule.devkit.generation.api.gatherer.Message;
import org.mule.devkit.generation.api.gatherer.NotificationGatherer;
import org.mule.devkit.generation.studio.editor.MuleStudioUtils;
import org.mule.devkit.verifiers.util.MulePathResolver;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.devkit.model.module.ProcessorMethod;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;


public class MultiModuleStudioPackageVerifier implements MultiModuleAnnotationVerifier {

    @Override
    public boolean shouldVerify(List<Module> modules) {
        return true;
    }

    @Override
    public List<Module> processableModules(List<Module> modules) {
        return modules;
    }

    @Override
    public void verify(List<Module> modules, NotificationGatherer gatherer) throws AnnotationVerificationException {
        /**
         * this condition ensures that if an {@link org.mule.api.annotations.ExpressionLanguage} is used, it won't
         * generate the update site for Studio if there is not a @Connector or @Module in the same project
         * */
        if (!hasConnectorOrModule(modules)){
            gatherer.error(modules.get(0), Message.DEVKIT_PACKAGE_PROPERTY_REQUIRES_AT_LEAST_ONE_MODULE);
        }else {
            //Module validation
            verifyModules(modules, gatherer);
        }
    }

    private void verifyModules(List<Module> modules, NotificationGatherer gatherer) {
        List<Module> modulesToBeValidated=  getModuleByKind(modules, Arrays.asList(ModuleKind.CONNECTOR, ModuleKind.GENERIC));
        for (Module module: modulesToBeValidated){
            verifyIcon(module, gatherer);
        }
    }

    private void verifyIcon(Module module, NotificationGatherer gatherer) {

        if (module.hasProcessors() || hasWsdlComponent(module)) {
            checkIcon(String.format(MuleStudioUtils.GENERIC_CLOUD_CONNECTOR_LARGE, module.getModuleName()), module, gatherer);
            checkIcon(String.format(MuleStudioUtils.GENERIC_CLOUD_CONNECTOR_LARGE, module.getModuleName()), module, gatherer);
        }

        if (module.hasSources()) {
            checkIcon(String.format(MuleStudioUtils.GENERIC_ENDPOINT_LARGE, module.getModuleName()), module, gatherer);
            checkIcon(String.format(MuleStudioUtils.GENERIC_ENDPOINT_SMALL, module.getModuleName()), module, gatherer);
        }

        if (module.hasTransformers()) {
            checkIcon(String.format(MuleStudioUtils.GENERIC_TRANSFORMER_LARGE, module.getModuleName()), module, gatherer);
            checkIcon(String.format(MuleStudioUtils.GENERIC_TRANSFORMER_SMALL, module.getModuleName()), module, gatherer);
        }

        if (hasContainer(module)){
            checkIcon(String.format(MuleStudioUtils.GENERIC_CONTAINER_LARGE, module.getModuleName()), module, gatherer);
            checkIcon(String.format(MuleStudioUtils.GENERIC_CONTAINER_SMALL, module.getModuleName()), module, gatherer);
        }

        if (hasContainerList(module)){
            checkIcon(String.format(MuleStudioUtils.GENERIC_FLOW_LARGE, module.getModuleName()), module, gatherer);
            checkIcon(String.format(MuleStudioUtils.GENERIC_FLOW_SMALL, module.getModuleName()), module, gatherer);
        }

    }

    private void checkIcon(String fileName, Module module, NotificationGatherer gatherer)  {
        File fileToCopy = MulePathResolver.getFileResolvingPackages(module, fileName);
        if (!fileToCopy.exists()) {
            gatherer.warn(module, Message.MISSING_ICON,module.getModuleName(),fileToCopy.getAbsolutePath());
        }
    }

    private List<Module> getModuleByKind(List<Module> modules, List<ModuleKind> kinds){
        List<Module> specificModules= new ArrayList<Module>();
        for(Module module: modules){
            if(kinds.contains(module.getKind())){
                specificModules.add(module);
            }
        }
        return specificModules;
    }


    private boolean hasConnectorOrModule(List<Module> modules) {
        boolean hasConnectorOrModule = false;
        for (Module module : modules) {
            if (module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC){
                hasConnectorOrModule = true;
            }
        }
        return hasConnectorOrModule;
    }

    private boolean hasWsdlComponent(Module module){
        return module.manager().wsdlProviderComponent().size() > 0;
    }

    private boolean hasContainer(Module module)
    {
        for (ProcessorMethod processorMethod : module.getProcessorMethods())
        {
            if (processorMethod.isContainer()){
                return true;
            }
        }
        return false;
    }

    private boolean hasContainerList(Module module)
    {
        for (ProcessorMethod processorMethod : module.getProcessorMethods())
        {
            if (processorMethod.isContainerList()){
                return true;
            }
        }
        return false;
    }
}

