/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.generation.studio.editor;

import org.mule.api.annotations.ConnectStrategy;
import org.mule.api.annotations.Paged;
import org.mule.api.annotations.Query;
import org.mule.api.annotations.QueryOperator;
import org.mule.api.annotations.display.Placement;
import org.mule.api.annotations.param.MetaDataKeyParam;
import org.mule.api.annotations.param.MetaDataStaticKey;
import org.mule.devkit.generation.api.Context;
import org.mule.devkit.model.Identifiable;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.Parameter;
import org.mule.devkit.model.Variable;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ProcessorMethod;
import org.mule.devkit.model.module.SourceMethod;
import org.mule.devkit.model.module.components.connection.ConnectionManagementComponent;
import org.mule.devkit.model.module.connectivity.ConnectionManagementCapability;
import org.mule.devkit.model.module.connectivity.ManagedConnectionModule;
import org.mule.devkit.model.module.oauth.OAuthVersion;
import org.mule.devkit.model.schema.SchemaConstants;
import org.mule.devkit.model.studio.*;
import org.mule.devkit.utils.NameUtils;
import org.mule.util.StringUtils;

import javax.lang.model.type.TypeKind;
import javax.xml.bind.JAXBElement;
import java.util.*;

public abstract class BaseStudioXmlBuilder {

    public static final String GENERAL_GROUP_NAME = "General";
    public static final String CONNECTION_GROUP_NAME = "Connection";
    public static final String ADVANCED_CONNECTION_GROUP_NAME = "Connector Configuration Overrides";
    private static final String QUERY_GROUP_NAME = "Query";
    private static final String PAGING_GROUP_NAME = "Paging";
    private static final String MESSAGE_SOURCE_POLLING_GROUP_NAME = "Polling";

    protected ObjectFactory objectFactory;
    protected MuleStudioUtils helper;
    protected Module module;
    protected Method method;
    protected String moduleName;
    protected Context context;


    protected BaseStudioXmlBuilder(Context context) {
        this.context = context;
        helper = new MuleStudioUtils();
        objectFactory = new ObjectFactory();
    }

    protected BaseStudioXmlBuilder(Context context, Module module) {
        this(context);
        this.module = module;
        moduleName = module.getModuleName();
    }

    protected Group createGroupWithModeSwitch(List<? extends Method> methods) {
        ModeType modeSwitch = getModeType(methods);
        return getModuleConnectorGenericGroup(modeSwitch);
    }

    protected Group createGroupWithNoOperationModeSwitchForProcessors(List<? extends Method> methods) {
        return createGroupWithNoOperationModeSwitch(methods, "processor");
    }

    protected Group createGroupWithNoOperationModeSwitchForSources(List<? extends Method> methods) {
        return createGroupWithNoOperationModeSwitch(methods, "source");
    }

    protected Group createGroupWithNoOperationModeSwitch(List<? extends Method> methods, String typeOfConnectorName) {
        ModeType modeSwitch = getModeType(methods);
        //adding the no-operation
        NoOperationType noOperation = new NoOperationType();
        noOperation.setAbstractElement(MuleStudioEditorXmlGenerator.URI_PREFIX + module.getModuleName() + '/' + helper.getGlobalRefId(module.getModuleName()));
        noOperation.setVersions(buildVersionsString());
        noOperation.setConnectorName(typeOfConnectorName);
        modeSwitch.setNoOperation(noOperation);
        return getModuleConnectorGenericGroup(modeSwitch);
    }

    private ModeType getModeType(List<? extends Method> methods) {
        ModeType modeSwitch = new ModeType();
        modeSwitch.getMode().addAll(this.getModes(methods));
        modeSwitch.setCaption(helper.formatCaption("Operation"));
        modeSwitch.setName(StringUtils.capitalize(moduleName) + " operations to execute");
        modeSwitch.setDescription(helper.formatDescription("Operation"));
        modeSwitch.setAlwaysCombo(true);
        return modeSwitch;
    }

    private Group getModuleConnectorGenericGroup(ModeType modeSwitch) {
        Group group = new Group();
        group.setId(module.getModuleName() + "ConnectorGeneric");
        group.getRegexpOrEncodingOrModeSwitch().add(objectFactory.createGroupModeSwitch(modeSwitch));
        group.setCaption(helper.formatCaption(MuleStudioEditorXmlGenerator.GROUP_DEFAULT_CAPTION));
        return group;
    }

    protected List<ModeElementType> getModes(List<? extends Method> methods) {
        List<ModeElementType> modes = new ArrayList<ModeElementType>();
        for (Method method : methods) {
            ModeElementType mode = new ModeElementType();
            mode.setModeId(MuleStudioEditorXmlGenerator.URI_PREFIX + module.getModuleName() + '/' + helper.getLocalId(method));
            mode.setModeLabel(StringUtils.capitalize(helper.getFriendlyName(method)));
            modes.add(mode);
        }

        return modes;
    }

    protected BaseStudioXmlBuilder(Context context, Method method, Module module) {
        this(context, module);
        this.method = method;
    }

    protected List<AttributeCategory> processMethodParameters() {
        Map<String, AttributeCategory> attributeCategoriesByName = processVariableElements(getParametersSorted());

        if(method instanceof ProcessorMethod) {
            ProcessorMethod processor = (ProcessorMethod) method;

            if (shouldAddAccessTokenId(processor)){
                processOAuthProtectedMethod(attributeCategoriesByName);
            }

            if (processor.isPaged()) {
                processPagedMethod(attributeCategoriesByName, processor);
            }

        }

        if(method instanceof SourceMethod) {
            if (((SourceMethod) method).isPolling()) {
                processPoolingSourceMethod(attributeCategoriesByName, (SourceMethod) method);
            }

        }

        return new ArrayList<AttributeCategory>(attributeCategoriesByName.values());
    }

    private void processPagedMethod(Map<String, AttributeCategory> attributeCategoriesByName, ProcessorMethod processor)
    {
        Paged cfg = processor.getPagingAnnotation();

        StringAttributeType fetchSize = new StringAttributeType();
        fetchSize.setName("fetchSize");
        fetchSize.setRequired(false);
        fetchSize.setDefaultValue(String.valueOf(cfg.defaultFetchSize()));
        fetchSize.setCaption(helper.formatCaption("Fetch Size"));
        fetchSize.setDescription(helper.formatDescription("The amount of items to fetch on each invocation to the data source"));
        fetchSize.setJavaType("java.lang.String");

        AttributeCategory attributeCategory = this.getOrCreateDefaultAttributeCategory(attributeCategoriesByName);

        Group group = new Group();
        group.setCaption(helper.formatCaption(PAGING_GROUP_NAME));
        group.setId(StringUtils.uncapitalize((PAGING_GROUP_NAME)));

        group.getRegexpOrEncodingOrModeSwitch().add(helper.createJAXBElement(fetchSize));

        attributeCategory.getGroup().add(group);

    }

    private void processPoolingSourceMethod(Map<String, AttributeCategory> attributeCategoriesByName, SourceMethod sourceMethod)
    {
        StringAttributeType pollingPeriod = new StringAttributeType();
        pollingPeriod.setName("pollingPeriod");
        pollingPeriod.setRequired(false);
        pollingPeriod.setDefaultValue(String.valueOf(sourceMethod.getPollingPeriod()));
        pollingPeriod.setCaption(helper.formatCaption("Polling Period"));
        pollingPeriod.setDescription(helper.formatDescription("Polling Period for Message Sources with Polling Strategy in ms"));
        pollingPeriod.setJavaType("java.lang.String");

        AttributeCategory attributeCategory = this.getOrCreateDefaultAttributeCategory(attributeCategoriesByName);

        Group group = new Group();
        group.setCaption(helper.formatCaption(MESSAGE_SOURCE_POLLING_GROUP_NAME));
        group.setId(StringUtils.uncapitalize((MESSAGE_SOURCE_POLLING_GROUP_NAME)));

        group.getRegexpOrEncodingOrModeSwitch().add(helper.createJAXBElement(pollingPeriod));

        attributeCategory.getGroup().add(group);

    }

    private Group getOrCreateGroup(Map<String, AttributeCategory> attributeCategoriesByName, String name)
    {
        AttributeCategory attributeCategory = null;
        if (attributeCategoriesByName.get(name) != null) {
            attributeCategory = attributeCategoriesByName.get(name);
        } else {
            attributeCategory = new AttributeCategory();
            attributeCategory.setCaption(helper.formatCaption(name));
            attributeCategory.setDescription(name);
            attributeCategoriesByName.put(name,attributeCategory);
        }

        Group group = getGroup(attributeCategory, name);

        if (group == null) {
            group = new Group();
            group.setCaption(helper.formatCaption(name));
            group.setId(StringUtils.uncapitalize(name));
            attributeCategory.getGroup().add(group);
        }
        return group;
    }

    /**
     * @param processor the processor to be tested to determine if supports OAuth "accessTokenId" attribute or not
     * @return true if the {@code processor} has been annotated with {@link org.mule.api.annotations.oauth.OAuthProtected} and
     * either the module has a OAuth2Component with OAuth .V2 or the module is implementing the {@link org.mule.api.annotations.oauth.OAuth2}
     * annotation itself (old scenario). False otherwise.
     */
    private boolean shouldAddAccessTokenId(ProcessorMethod processor) {
        return processor.isOAuthProtected()
                && (
                module.manager().oauth2Component().isPresent() && module.manager().oauth2Component().get().getOAuthVersion().equals(OAuthVersion.V2)
                        || processor.canBeUsedInOAuthManagement() && processor.getOAuthModule().getOAuthVersion().equals(OAuthVersion.V2)
        );
    }

    private void processOAuthProtectedMethod(Map<String, AttributeCategory> attributeCategoriesByName)
    {
        StringAttributeType accessTokenId = new StringAttributeType();
        accessTokenId.setName("accessTokenId");
        accessTokenId.setRequired(false);
        accessTokenId.setCaption(helper.formatCaption("Access Token Id"));
        accessTokenId.setDescription(helper.formatDescription("The id of the access token that will be used to authenticate the call"));
        accessTokenId.setJavaType("java.lang.String");

        Group group = getOrCreateGroup(attributeCategoriesByName, MuleStudioEditorXmlGenerator.ADVANCED_ATTRIBUTE_CATEGORY_CAPTION);

        group.getRegexpOrEncodingOrModeSwitch().add(helper.createJAXBElement(accessTokenId));
    }

    private Group getGroup(AttributeCategory attributeCategory, String groupName) {
        if(attributeCategory != null) {
            for(Group group : attributeCategory.getGroup()) {
                if(StringUtils.uncapitalize(groupName).equalsIgnoreCase(group.getId())) {
                    return group;
                }
            }
        }
        return null;
    }

    /**
     * TODO refactor this method to remove all the common behaviour from {@link org.mule.devkit.generation.studio.editor.globalcloudconnector.GlobalCloudConnectorTypeBuilder} to each concrete subclass of {@link org.mule.devkit.generation.studio.editor.globalcloudconnector.GlobalCloudConnectorTypeBuilder}
     */
    protected Map<String, AttributeCategory> processVariableElements(List<? extends Variable> variableElements) {

        Map<String, Group> groupsByName = new LinkedHashMap<String, Group>();
        Map<String, AttributeCategory> attributeCategoriesByName = new LinkedHashMap<String, AttributeCategory>();
        getOrCreateDefaultAttributeCategory(attributeCategoriesByName); // Create default category
        processConnectionAttributes(groupsByName, attributeCategoriesByName);

        if (hasStrategy(module, ConnectStrategy.SINGLE_INSTANCE)){
            createCacheConfigAttributes(groupsByName, attributeCategoriesByName);
        }
        if (hasStrategy(module, ConnectStrategy.MULTIPLE_INSTANCES)){
            createPoolingProfileAttributes(groupsByName, attributeCategoriesByName);
        }

        createReconnectionAttributes(groupsByName, attributeCategoriesByName);
        createOAuthConfig(groupsByName, attributeCategoriesByName);
        createHttpCallbackConfig(groupsByName, attributeCategoriesByName);
        createMetaDataAttributes(groupsByName, attributeCategoriesByName);

        for (Variable parameter : variableElements) {
            JAXBElement<? extends AttributeType> jaxbElement = createJaxbElement(parameter);
            AttributeCategory attributeCategory = getOrCreateAttributeCategory(attributeCategoriesByName, parameter.getAnnotation(Placement.class));
            Group group = getOrCreateGroup(groupsByName, parameter);
            group.getRegexpOrEncodingOrModeSwitch().add(jaxbElement);

            if (!attributeCategory.getGroup().contains(group)) {
                attributeCategory.getGroup().add(group);
            }
        }

        return attributeCategoriesByName;
    }

    /**
     * TODO:refactor needed, this method is copied and pasted in {@link org.mule.devkit.generation.connectivity.PoolGenerator} and {@link MuleStudioEditorXmlGenerator}
     * @param module module to test
     * @param strategy specific strategy to test against
     * @return true if the connector, or if any of its strategies with @ConnectionManagement, has @Connect method using the same
     * strategy as the parametrized. False otherwise
     */
    private boolean hasStrategy(Module module, ConnectStrategy strategy) {
        if (module instanceof ManagedConnectionModule){
            if (((ManagedConnectionModule) module).getConnectMethod().getStrategy().equals(strategy)){
                return true;
            }
        }else{
            for (ConnectionManagementComponent connectionManagementComponent : module.manager().connectionManagementComponents()) {
                if (connectionManagementComponent.getConnectMethod().getStrategy().equals(strategy)){
                    return true;
                }
            }
        }
        return false;
    }

    protected void createMetaDataAttributes(Map<String,Group> groupsByName, Map<String,AttributeCategory> attributeCategoriesByName) {
        // override if necessary
    }

    protected void processConnectionAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        // override if necessary
    }

    protected void createReconnectionAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        // override if necessary
    }

    protected void createPoolingProfileAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        // override if necessary
    }

    protected void createCacheConfigAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        // override if necessary
    }

    protected void createOAuthConfig(Map<String,Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        // override if necessary
    }

    protected void createHttpCallbackConfig(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName){
        // override if necessary
    }

    private AttributeCategory getOrCreateDefaultAttributeCategory(Map<String, AttributeCategory> attributeCategoriesByName) {
        return getOrCreateAttributeCategory(attributeCategoriesByName, null);

    }

    private AttributeCategory getOrCreateAttributeCategory(Map<String, AttributeCategory> attributeCategoriesByName, Placement placement) {
        if (placement == null || StringUtils.isBlank(placement.tab())) {
            if (!attributeCategoriesByName.containsKey(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION)) {
                AttributeCategory attributeCategoryGeneral = new AttributeCategory();
                attributeCategoryGeneral.setCaption(helper.formatCaption(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION));
                attributeCategoryGeneral.setDescription(helper.formatDescription(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_DESCRIPTION));
                attributeCategoriesByName.put(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION, attributeCategoryGeneral);



            }
            return attributeCategoriesByName.get(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION);
        } else {
            String attributeCategoryName;
            if (StringUtils.isNotBlank(placement.tab())) {
                attributeCategoryName = placement.tab();
            } else {
                attributeCategoryName = MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION;
            }
            if (!attributeCategoriesByName.containsKey(attributeCategoryName)) {
                AttributeCategory attributeCategory = new AttributeCategory();
                attributeCategory.setCaption(helper.formatCaption(attributeCategoryName));
                attributeCategory.setDescription(helper.formatDescription(attributeCategoryName));
                attributeCategoriesByName.put(attributeCategoryName, attributeCategory);
            }
            return attributeCategoriesByName.get(attributeCategoryName);
        }
    }

    private Group getOrCreateGroup(Map<String, Group> groupsByName, Variable parameter) {
        Placement placement = parameter != null ? parameter.getAnnotation(Placement.class) : null;
        if (parameter != null && parameter.isQuery()) {
            Group queryGroup = new Group();
            queryGroup.setCaption(helper.formatCaption(QUERY_GROUP_NAME));
            queryGroup.setId(StringUtils.uncapitalize(QUERY_GROUP_NAME));
            groupsByName.put(QUERY_GROUP_NAME,queryGroup);
            return groupsByName.get(QUERY_GROUP_NAME);
        }
        if (placement == null || StringUtils.isBlank(placement.group())) {
            if (!groupsByName.containsKey(GENERAL_GROUP_NAME)) {
                Group groupGeneral = new Group();
                groupGeneral.setCaption(helper.formatCaption(GENERAL_GROUP_NAME));
                groupGeneral.setId(StringUtils.uncapitalize(GENERAL_GROUP_NAME));
                groupsByName.put(GENERAL_GROUP_NAME, groupGeneral);
            }
            return groupsByName.get(GENERAL_GROUP_NAME);
        } else {
            String groupName = placement.group();
            if (!groupsByName.containsKey(groupName)) {
                Group group = new Group();
                group.setCaption(groupName);
                group.setId(StringUtils.uncapitalize(groupName));
                groupsByName.put(groupName, group);
            }
            return groupsByName.get(groupName);
        }
    }

    protected boolean isConfigurableVariable(Variable<?> variable) {
        if(module != null && module.getConfigurableFields() != null) {
            return module.getConfigurableFields().contains(variable);
        } else {
            return false;
        }
    }

    protected JAXBElement<? extends AttributeType> createJaxbElement(Variable<?> variable) {
        AttributeType attributeType;
        if (variable.isQuery()) {
            attributeType = createQueryType(variable);
        } else if(variable.isMetaDataKey()) {
            attributeType = createTypeChooserType(variable);
        } else if (variable.asType().isEnum()) {
            attributeType = createEnumType(variable);
        } else if (variable.isText()) {
            attributeType = createTextType(variable);
        } else if (isComplexList(variable)) {
            attributeType = createComplexListType(variable);
        } else if (isListOfMaps(variable) || isSimpleList(variable) || isSimpleMap(variable) || isSimpleSet(variable)) {
            attributeType = handleCollectionVariable(variable);
        } else if (variable.asType().isComplexType() && !variable.isRefOnly()) {
            NestedElementReference complexTypeNestedElementReference = createNestedElementReference(variable);
            return objectFactory.createGroupChildElement(complexTypeNestedElementReference);
        } else {
            attributeType = createAttributeType(variable);
        }

        if (isConfigurableVariable(variable)) {
            setMetadataAttributes(variable, attributeType);
        }

        return helper.createJAXBElement(attributeType);
    }

    protected void setMetadataAttributes(Variable<?> variable, AttributeType attributeType) {
    }

    protected AttributeType createTextType(Variable<?> variable) {
        TextType text = new TextType();

        text.setName(variable.getName());
        text.setCaption(NameUtils.friendlyNameFromCamelCase(variable.getName()));
        text.setRequired(!variable.isOptional());
        text.setIsToElement(true);
        text.setWrapWithCDATA(true);
        text.setVisibleInDialog(true);
        text.setNestedName(variable.getName());

        return text;
    }

    protected AttributeType createQueryType(Variable<?> variable) {
        QueryType type = new QueryType();
        type.setSupportsExpressions(true);
        type.setAssociatedConfig("config-ref");
        Query qVariable = variable.getAnnotation(org.mule.api.annotations.Query.class);
        List<QueryOperator> disabledOperators = Arrays.asList(qVariable.disabledOperators());
        type.setAndOperator(disabledOperators.contains(QueryOperator.AND) ? "disabled" : "enabled");
        type.setOrOperator(disabledOperators.contains(QueryOperator.OR) ? "disabled" : "enabled");
        type.setNativeQuery(module.hasQueryTranslator() ? "enabled" : "disabled");
        type.setLimit(qVariable.limit()? "enabled" : "disabled");
        type.setOffset(qVariable.offset()? "enabled" : "disabled");
        type.setOrderBy(qVariable.orderBy() ? "enabled" : "disabled");
        helper.setAttributeTypeInfo(variable, type);
        return type;
    }

    private AttributeType createComplexListType(Variable parameter) {
        ObjectListAttributeType objectListAttributeTyper = new ObjectListAttributeType();
        objectListAttributeTyper.setListName(NameUtils.uncamel(parameter.getName()));
        setCommonCollectionAttributes(parameter, objectListAttributeTyper);
        return objectListAttributeTyper;
    }

    private boolean isComplexList(Variable parameter) {
        return parameter.asType().isArrayOrList() && !parameter.getTypeArguments().isEmpty()
                && parameter.getTypeArguments().get(0).asTypeMirror().getKind().equals(TypeKind.DECLARED)
                && parameter.getTypeArguments().get(0).isComplexTypeWithGetterAndSetter(false);
    }

    private AttributeType createTypeChooserType(Variable<?> variable) {
        TypeChooserType type = new TypeChooserType();
        type.setSupportsExpressions(true);
        type.setAssociatedConfig("config-ref");
        type.setAffects(variable.getAnnotation(MetaDataKeyParam.class).affects().toString());
        helper.setAttributeTypeInfo(variable, type);
        return type;
    }

    private NestedElementReference createNestedElementReference(Variable parameter) {
        NestedElementReference childElement = new NestedElementReference();
        String defaultValue = parameter.getDefaultValue();
        if (defaultValue != null) {
            childElement.setDefaultValue(defaultValue);
        }
        childElement.setName(MuleStudioEditorXmlGenerator.URI_PREFIX + moduleName + '/' + NameUtils.uncamel(parameter.getName()));
        childElement.setCaption(helper.getFormattedCaption(parameter));
        childElement.setAllowMultiple(false);
        childElement.setInplace(true);
        childElement.setJavaType(parameter.getJavaType());
        childElement.setRequired(!parameter.isOptional());
        return childElement;
    }

    private AttributeType handleCollectionVariable(Variable parameter) {
        AttributeType attributeType;
        if (isListOfMaps(parameter)) {
            ListOfMapAttributeType objectListAttributeType = new ListOfMapAttributeType();
            objectListAttributeType.setRequired(!parameter.isOptional());
            objectListAttributeType.setListName(NameUtils.uncamel(parameter.getName()));
            objectListAttributeType.setInnerName(NameUtils.uncamel(SchemaConstants.INNER_PREFIX + NameUtils.singularize(parameter.getName())));
            if (parameter.isMetaDataStaticKey()){
                objectListAttributeType.setMetaDataStaticKey(parameter.getAnnotation(MetaDataStaticKey.class).type());
            }
            setCommonCollectionAttributes(parameter, objectListAttributeType);
            attributeType = objectListAttributeType;
        } else if (isSimpleList(parameter) || isSimpleSet(parameter)) {
            StringListAttributeType stringListAttributeTyper = new StringListAttributeType();

            stringListAttributeTyper.setListName(NameUtils.uncamel(parameter.getName()));
            setCommonCollectionAttributes(parameter, stringListAttributeTyper);
            attributeType = stringListAttributeTyper;
        } else {
            StringMapAttributeType stringMapAttributeType = new StringMapAttributeType();
            stringMapAttributeType.setMapName(NameUtils.uncamel(parameter.getName()));
            if (parameter.isMetaDataStaticKey()){
                stringMapAttributeType.setMetaDataStaticKey(parameter.getAnnotation(MetaDataStaticKey.class).type());
            }
            setCommonCollectionAttributes(parameter, stringMapAttributeType);
            attributeType = stringMapAttributeType;
        }
        // DEVKIT-275 Need required information in Studio
        attributeType.setRequired(!parameter.isOptional());
        return attributeType;
    }

    private void setCommonCollectionAttributes(Variable parameter, CollectionAttributeType collectionAttributeType) {
        collectionAttributeType.setItemName(NameUtils.uncamel(NameUtils.singularize(parameter.getName())));
        collectionAttributeType.setLocalName(helper.getLocalId(method, parameter));
        collectionAttributeType.setCaption(helper.getFormattedCaption(parameter));
        collectionAttributeType.setJavaType(parameter.getJavaType());
        String defaultValue = parameter.getDefaultValue();
        if (defaultValue != null) {
            collectionAttributeType.setDefaultValue(defaultValue);
        }
        if (method != null) {
            collectionAttributeType.setDescription(helper.formatDescription(method.getJavaDocParameterSummary(parameter.getName())));
        } else {
            collectionAttributeType.setDescription(helper.formatDescription(parameter.getJavaDocSummary()));
        }
    }

    private boolean isListOfMaps(Variable parameter) {
        return parameter.asType().isArrayOrList() && !parameter.getTypeArguments().isEmpty() && parameter.getTypeArguments().get(0).isMap();
    }

    private boolean isSimpleMap(Variable parameter) {
        return parameter.asType().isMap() && (parameter.getTypeArguments().isEmpty() || !parameter.getTypeArguments().get(1).isCollection());
    }

    private boolean isSimpleList(Variable parameter) {
        return parameter.asType().isArrayOrList() && (parameter.getTypeArguments().isEmpty() || !parameter.getTypeArguments().get(0).isCollection());
    }

    private boolean isSimpleSet(Variable parameter) {
        return parameter.asType().isSet() && (parameter.getTypeArguments().isEmpty() || !parameter.getTypeArguments().get(0).isCollection());
    }

    private List<Parameter> getParametersSorted() {
        List<Parameter> parameters = new ArrayList<Parameter>(method.getParameters());
        Iterator<Parameter> iterator = parameters.iterator();
        while (iterator.hasNext()) {
            if (iterator.next().shouldBeIgnored()) {
                iterator.remove();
            }
        }

        Collections.sort(parameters, new VariableComparator());
        return parameters;
    }

    private AttributeType createAttributeType(Variable parameter) {
        AttributeType attributeType = helper.createAttributeTypeIgnoreEnumsAndCollections(parameter);
        if (attributeType != null) {
            helper.setAttributeTypeInfo(parameter, attributeType);
        }
        return attributeType;
    }

    protected List<AttributeType> getConnectionAttributes(ConnectionManagementCapability connectionManagementCapability) {
        List<AttributeType> parameters = new ArrayList<AttributeType>();
        for (Parameter connectAttributeType : connectionManagementCapability.getConnectMethod().getParameters()) {
            parameters.add(getAttributeType(connectAttributeType));
        }
        return parameters;
    }

    /**
     * For a given variable, it generates the element that will represent it in the editors.xml file. If the current module supports connection management,
     * then it should be written in each processor
     * @param variable
     * @return
     */
    protected AttributeType getAttributeType(Variable variable) {
        AttributeType parameter = helper.createAttributeTypeIgnoreEnumsAndCollections(variable);
        helper.setAttributeTypeInfo(variable, parameter);
        // As these can be overriden at operation level, they are always optional
        parameter.setRequired(false);
        setMetadataAttributes(variable, parameter);
        return parameter;
    }

    private EnumType createEnumType(Variable<?> variable) {
        EnumType enumType = new EnumType();
        enumType.setSupportsExpressions(true);
        enumType.setAllowsCustom(true);
        helper.setAttributeTypeInfo(variable, enumType);
        for (Identifiable<?> enumMember : ((org.mule.devkit.model.EnumType) variable.asType()).getEnumConstants()) {
            String enumConstant = enumMember.getName();
            EnumElement enumElement = new EnumElement();
            enumElement.setValue(enumConstant);
            enumType.getOption().add(enumElement);
        }
        Collections.sort(enumType.getOption(), new EnumElementComparator());
        return enumType;
    }

    protected String buildVersionsString() {
        /*
         * DEVKIT-368: Studio needs the supported ESB versions in order to support
         * multiple cloud connectors.
         * TODO: Fix once Studio provides a better way to indicate that this is unbounded (a.k.a. remove 8.0.0!!!)
         */
        return "[" + module.getMinMuleVersion().toCompleteNumericVersion()  + ",8.0.0]";

    }
}