/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.generation.studio;

import org.mule.api.annotations.display.Icons;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.studio.packaging.ModuleRelativePathBuilder;
import org.mule.devkit.generation.studio.utils.MuleStudioPathResolver;
import org.mule.devkit.model.module.Module;
import org.mule.util.IOUtils;

import java.io.*;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class MuleStudioIconsGenerator extends AbstractMuleStudioNamespaceGenerator {

    private static List<Product> PRODUCES = Arrays.asList(Product.STUDIO_ICONS);
    private ArrayList<ModuleRelativePathBuilder> iconsFileName;

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        iconsFileName = new ArrayList<ModuleRelativePathBuilder>();

        Icons icons = module.getAnnotation(Icons.class);

        String connectorSmall = String.format(Icons.GENERIC_CLOUD_CONNECTOR_SMALL, module.getModuleName());
        String connectorLarge = String.format(Icons.GENERIC_CLOUD_CONNECTOR_LARGE, module.getModuleName());

        String transformerSmall = String.format(Icons.GENERIC_TRANSFORMER_SMALL, module.getModuleName());
        String transformerLarge = String.format(Icons.GENERIC_TRANSFORMER_LARGE, module.getModuleName());

        String endpointSmall = String.format(Icons.GENERIC_ENDPOINT_SMALL, module.getModuleName());
        String endpointLarge = String.format(Icons.GENERIC_ENDPOINT_LARGE, module.getModuleName());

        if (icons != null) {
            connectorSmall = icons.connectorSmall();
            connectorLarge = icons.connectorLarge();

            transformerSmall = icons.transformerSmall();
            transformerLarge = icons.transformerLarge();

            endpointSmall = icons.endpointSmall();
            endpointLarge = icons.endpointLarge();
        }

        copyFile(connectorSmall, "icons/small", module, Icons.SUFIX_CONNECTOR_24X16_PNG);
        copyFile(connectorLarge, "icons/large", module, Icons.SUFIX_CONNECTOR_48X32_PNG);
        if (module.hasTransformers()) {
            copyFile(transformerSmall, "icons/small", module, Icons.SUFIX_TRANSFORMER_24X16_PNG);
            copyFile(transformerLarge, "icons/large", module, Icons.SUFIX_TRANSFORMER_48X32_PNG);
        }
        if (module.hasSources()) {
            copyFile(endpointSmall, "icons/small", module, Icons.SUFIX_ENDPOINT_24X16_PNG);
            copyFile(endpointLarge, "icons/large", module, Icons.SUFIX_ENDPOINT_48X32_PNG);
        }

        ctx().registerProduct(Product.STUDIO_ICONS, module, iconsFileName);
    }

    private void copyFile(String fileName, String folder, Module module, String sufix) throws GenerationException {
        InputStream iconStream = null;
        OutputStream outputStream = null;
        try {
            outputStream = createFile(module, folder, fileName);
            File fileToCopy = MuleStudioPathResolver.getFileResolvingPackages(module, fileName);
            if (fileToCopy.exists()) {
                //taking the icons from the connectors/icons folder
                iconStream = new FileInputStream(fileToCopy);
            }else{
                //using the default ones
                String fileIconPath = String.format(sufix, "generic");
                iconStream = MuleStudioIconsGenerator.class.getClassLoader().getResourceAsStream(fileIconPath);
            }
            IOUtils.copy(iconStream, outputStream);
        } catch (IOException e) {
            throw new GenerationException("Error copying icons to output folder: " + e.getMessage(), e);
        } finally {
            IOUtils.closeQuietly(outputStream);
            IOUtils.closeQuietly(iconStream);
        }
    }

    private OutputStream createFile(Module module, String folder, String fileName) throws GenerationException {
        if (fileName.contains("/")) {
            fileName = fileName.substring(fileName.lastIndexOf("/") + 1);
        }
        try {
            ModuleRelativePathBuilder pathBuilder = new ModuleRelativePathBuilder(folder + "/" + fileName);
            String fullIconPath = pathBuilder.build(module).getFullPath();
            iconsFileName.add(pathBuilder);
            return ctx().getCodeModel().getCodeWriter().openBinary(null, fullIconPath);
        } catch (IOException e) {
            throw new GenerationException("Could not create file or folder " + fileName + ": " + e.getMessage(), e);
        }
    }
}