/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.studio.editor;

import org.apache.commons.lang.StringUtils;
import org.mule.api.annotations.Connector;
import org.mule.api.annotations.MetaDataScope;
import org.mule.api.annotations.MetaDataSwitch;
import org.mule.api.annotations.NoMetaData;
import org.mule.devkit.generation.api.Context;
import org.mule.devkit.generation.api.MultiModuleGenerator;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.studio.AbstractMuleStudioGenerator;
import org.mule.devkit.generation.studio.packaging.ModuleRelativePathBuilder;
import org.mule.devkit.generation.studio.utils.ModuleComparator;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.devkit.model.module.ProcessorMethod;
import org.mule.devkit.model.module.SourceMethod;
import org.mule.devkit.model.module.connectivity.ManagedConnectionModule;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.model.studio.AbstractElementType;
import org.mule.devkit.model.studio.EndpointType;
import org.mule.devkit.model.studio.GlobalType;
import org.mule.devkit.model.studio.NamespaceType;
import org.mule.devkit.model.studio.NestedElementType;
import org.mule.devkit.model.studio.ObjectFactory;
import org.mule.devkit.model.studio.PatternType;
import org.mule.devkit.model.studio.StudioModel;

import java.util.*;

import javax.lang.model.type.DeclaredType;
import javax.xml.bind.JAXBElement;

public class MuleStudioEditorXmlGenerator extends AbstractMuleStudioGenerator implements MultiModuleGenerator {

    public static final String URI_PREFIX = "http://www.mulesoft.org/schema/mule/";
    public static final String GLOBAL_CLOUD_CONNECTOR_LOCAL_ID = "config";
    public static final String ATTRIBUTE_CATEGORY_DEFAULT_CAPTION = "General";
    public static final String ATTRIBUTE_CATEGORY_DEFAULT_DESCRIPTION = "General";
    public static final String ADVANCED_ATTRIBUTE_CATEGORY_CAPTION = "Advanced";
    public static final String GROUP_DEFAULT_CAPTION = "Basic Settings";
    private ObjectFactory objectFactory = new ObjectFactory();
    private final static List<Product> PRODUCES = Arrays.asList(Product.STUDIO_EDITOR_XML);
    private static final String EDITOR_XML_FILE_NAME = "editors.xml";

    @Override
    public List<Product> consumes() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(List<Module> modules) {
        for(Module module: modules){
            if(module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC){
                return true;
            }
        }
        return false;
    }

    @Override
    public List<Module> processableModules(List<Module> modules) {
        List<Module> specificModules= new ArrayList<Module>();
        for(Module module: modules){
            if(module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC || module.getKind() == ModuleKind.METADATA_CATEGORY){
                specificModules.add(module);
            }
        }
        return specificModules;
    }

    @Override
    public void generate(List<Module> modules) {
        List<Module> connectorOrModule = getSortedConnectorAndOauthModules(modules);
        List<Module> metaDataCategoryModules = getModuleByKind(modules, Arrays.asList(ModuleKind.METADATA_CATEGORY));

        for(Module module: connectorOrModule){
            List<Module> usedMetaDataCategoryModules = getSpecificUsedMetaDataCategoryModules(metaDataCategoryModules, module);
            generateModule(module, usedMetaDataCategoryModules);
        }
    }

    /**
     * Before processing the modules we need to make sure that (for multi-module scenarios, connection management + oauth),
     * the first modules to work with are the connection management ones.
     * <p>If the Oauth are the first ones to be parsed, the generation of the editors.xml will be messy, and have lots
     * of inconsistent information regarding the connector</p>
     */
    private List<Module> getSortedConnectorAndOauthModules(List<Module> modules) {
        List<Module> connectorOrModule = getModuleByKind(modules, Arrays.asList(ModuleKind.CONNECTOR, ModuleKind.GENERIC));
        Collections.sort(connectorOrModule, new ModuleComparator());
        return connectorOrModule;
    }

    public static class PatternTypeOperationsChooser implements StudioModel.BuilderWithArgs<Boolean, JAXBElement<PatternType>> {

        private Context ctx;
        private Module module;

        public PatternTypeOperationsChooser(Context ctx, Module module) {
            this.ctx = ctx;
            this.module = module;
        }

        @Override
        public JAXBElement<PatternType> build(Boolean isOAuth) {
            if (isOAuth) {
                return new OAuthPatternTypeOperationsBuilder(ctx, module, PatternTypes.CLOUD_CONNECTOR).build();
            } else {
                return new PatternTypeOperationsBuilder(ctx, module, PatternTypes.CLOUD_CONNECTOR).build();
            }
        }

    }

    public static class ProcessorMethodsChooser implements StudioModel.BuilderWithArgs<Boolean, List<JAXBElement<? extends AbstractElementType>>> {

        private Module module;
        private ObjectFactory objectFactory;
        private Context ctx;

        public ProcessorMethodsChooser(Context ctx, Module module, ObjectFactory objectFactory) {
            this.module = module;
            this.ctx = ctx;
            this.objectFactory = objectFactory;
        }

        @Override
        public List<JAXBElement<? extends AbstractElementType>> build(Boolean isOAuth) {
            List<JAXBElement<? extends AbstractElementType>> list = new ArrayList<JAXBElement<? extends AbstractElementType>>();

            for (ProcessorMethod processorMethod : module.getProcessorMethods()) {
                PatternType cloudConnector = new PatternTypeBuilder(ctx, processorMethod, module).build();
                if (processorMethod.hasDynamicMetaData() || processorMethod.hasQuery() || processorMethod.hasStaticKeyMetaData()
                        || processorMethod.hasMetaDataScope()) {
                    cloudConnector.setMetaData("dynamic");
                } else if (isMetaDataEnabled(processorMethod, module)) {
                    cloudConnector.setMetaData("static");
                }
                if (processorMethod.hasStaticKeyOutputMetaData()){
                    cloudConnector.setMetaDataStaticKey(processorMethod.getStaticKeyOutputMetaData().type());
                }
                List<String> categories= new ArrayList<String>();
                List<DeclaredType> annotationValuesCategories = new ArrayList<DeclaredType>();
                if (processorMethod.hasMetaDataScope()){
                    annotationValuesCategories.add(processorMethod.metaDataScope());
                }else {
                    if (module.getAnnotation(MetaDataScope.class)!= null){
                        annotationValuesCategories.add(module.metaDataScope());
                    }
                }
                for (DeclaredType declaredType: annotationValuesCategories){
                    String fullQualifiedName = declaredType.toString();
                    categories.add(fullQualifiedName.substring(fullQualifiedName.lastIndexOf(".") + 1));
                }
                if (!categories.isEmpty()){
                    cloudConnector.setCategories(StringUtils.join(categories, ","));
                }

                list.add(objectFactory.createNamespaceTypeCloudConnector(cloudConnector));
            }

            if (module instanceof OAuthModule) {
                PatternType authorize = new OAuthPatternTypeBuilder(ctx, "authorize", module).build();
                list.add(objectFactory.createNamespaceTypeCloudConnector(authorize));

                PatternType unAuthorize = new OAuthPatternTypeBuilder(ctx, "unauthorize", module).build();
                list.add(objectFactory.createNamespaceTypeCloudConnector(unAuthorize));

            }

            return list;
        }

        //TODO: Duplicated code, check MessageProcessorGenerator
        private boolean isMetaDataEnabled(Method processorMethod, Module module) {
            return module.getMinMuleVersion().atLeastBase("3.4") && processorMethod.getAnnotation(NoMetaData.class) == null &&
                    ((module.isConnector() && MetaDataSwitch.ON.equals(module.getAnnotation(Connector.class).metaData()))||
                            (ModuleKind.GENERIC.equals(module.getKind())
                                    && MetaDataSwitch.ON.equals(module.getAnnotation(org.mule.api.annotations.Module.class).metaData())));
        }
    }



    private void executeOncePerNamespace(NamespaceType namespace, Module module, List<Module> usedMetaDataCategoryModules) {
        String moduleName = module.getModuleName();

        namespace.setPrefix(moduleName);
        namespace.setUrl(URI_PREFIX + moduleName);

        ctx().getStudioModel().addPatternTypeOperation(moduleName, new PatternTypeOperationsChooser(ctx(), module));

        GlobalType globalCloudConnector = new ParentCloudConnectorTypeBuilder(ctx(), module).build();
        namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeGlobalCloudConnector(globalCloudConnector));

        if (module instanceof ManagedConnectionModule) {

            if (((ManagedConnectionModule) module).getConnectMethod().isSingleInstance()) {
                NestedElementType cacheConfigNestedElementType = new CacheConfigNestedElementBuilder(ctx(), module).build();
                namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNested(cacheConfigNestedElementType));
            } else {
                NestedElementType poolingProfileNestedElementType = new PoolingProfileNestedElementBuilder(ctx(), module).build();
                namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNested(poolingProfileNestedElementType));
            }


            NestedElementType reconnectionNestedElement = new ReconnectionNestedElementBuilder(ctx(), module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNested(reconnectionNestedElement));
        }

        processTransformerMethods(module, namespace);
        processSourceMethods(module, namespace);
    }

    private void generateModule(Module module, List<Module> usedMetaDataCategoryModules) {
        String moduleName = module.getModuleName();
        boolean isOAuth = module instanceof OAuthModule;

        NamespaceType namespace = ctx().getStudioModel().getOrCreateNamespace(module.getModuleName());

        if (isOAuth) {
            ctx().getStudioModel().addIsOAuth(moduleName, isOAuth);
            ctx().getStudioModel().addNestedElements(moduleName, new OAuthConfigNestedElementsBuilder(ctx(), module));
        }

        ctx().getStudioModel().addProcessorMethods(moduleName, new ProcessorMethodsChooser(ctx(), module, objectFactory));
        ctx().getStudioModel().addNestedElements(moduleName, new NestedsBuilder(ctx(), module));

        if (!moduleName.equals(namespace.getPrefix())) {
            executeOncePerNamespace(namespace, module, usedMetaDataCategoryModules);
        }

        GlobalType globalCloudConnector = new GlobalCloudConnectorTypeBuilder(ctx(), module, usedMetaDataCategoryModules, false, ParentCloudConnectorTypeBuilder.PARENT_CONFIG).build();
        namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeGlobalCloudConnector(globalCloudConnector));
        StudioModel.ConfigRefBuilder<JAXBElement<? extends AbstractElementType>> simpleConfigRefBuilder = ctx().getStudioModel().getConfigBuilderRef(moduleName);
        if (ctx().getStudioModel().getConfigBuilderRef(moduleName) == null) {
            simpleConfigRefBuilder = new SimpleConfigRefBuilder(ctx(), module);
            ctx().getStudioModel().addConfigBuilderRef(module.getModuleName(), simpleConfigRefBuilder);
        }
        simpleConfigRefBuilder.addRequiredType(module.getConfigElementName());

        String editorFileName = EDITOR_XML_FILE_NAME;

        ModuleRelativePathBuilder editorXMLPath = new ModuleRelativePathBuilder(editorFileName);
        ctx().getStudioModel().addNamespaceType(moduleName, editorXMLPath.build(module).getFullPath());
        ctx().registerProduct(Product.STUDIO_EDITOR_XML, module, editorXMLPath);
    }

    private void processTransformerMethods(Module module, NamespaceType namespace) {
        if (module.hasTransformers()) {
            namespace.getConnectorOrEndpointOrGlobal().add(new PatternTypeOperationsBuilder(ctx(), module, PatternTypes.TRANSFORMER).build());
            namespace.getConnectorOrEndpointOrGlobal().add(new AbstractTransformerBuilder(ctx(), module).build());
            GlobalType globalTransformer = new GlobalTransformerTypeOperationsBuilder(ctx(), module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeGlobalTransformer(globalTransformer));
        }
        for (Method transformerMethod : module.getTransformerMethods()) {
            PatternType transformer = new PatternTypeBuilder(ctx(), transformerMethod, module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeTransformer(transformer));
            GlobalType globalTransformer = new GlobalTransformerTypeBuilder(ctx(), transformerMethod, module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createNamespaceTypeGlobalTransformer(globalTransformer));
        }
    }

    private void processSourceMethods(Module module, NamespaceType namespace) {
        List<SourceMethod> sourceMethods = module.getSourceMethods();
        if (!sourceMethods.isEmpty()) {
            EndpointType endpointTypeListingOps = new EndpointTypeOperationsBuilder(ctx(), module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createCloudConnectorEndpoint(endpointTypeListingOps));
        }
        for (Method sourceMethod : sourceMethods) {
            EndpointType endpoint = new EndpointTypeBuilder(ctx(), sourceMethod, module).build();
            namespace.getConnectorOrEndpointOrGlobal().add(objectFactory.createCloudConnectorEndpoint(endpoint));
        }
    }

    //TODO repeated code, see MultiModuleMetaDataVerifier, MuleStudioEditorXmlGenerator, ConnectionManagerGenerator
    private List<Module> getSpecificUsedMetaDataCategoryModules(List<Module> metaDataCategoryModules, Module module) {
        List<Module> usedMetaDataCategoryModules= new ArrayList<Module>();
        List<DeclaredType> usedCategories = new ArrayList<DeclaredType>();
        //Obtaining all the used categories by reading the annotation field value
        if (module.getAnnotation(MetaDataScope.class) != null){
            usedCategories.add(module.metaDataScope());
        }
        for (ProcessorMethod processorMethod : module.getProcessorMethods()){
            if (processorMethod.hasMetaDataScope()){
                usedCategories.add(processorMethod.metaDataScope());
            }
        }

        //Looking up for the actual modules that match with the previous merged list (usedCategories)
        for (Module metaDataCategoryModule: metaDataCategoryModules){
            String fullQualifiedName = metaDataCategoryModule.getQualifiedName().toString();
            for (DeclaredType declaredType: usedCategories){
                if (fullQualifiedName.equals(declaredType.toString())){
                    usedMetaDataCategoryModules.add(metaDataCategoryModule);
                    break;
                }
            }
        }
        return usedMetaDataCategoryModules;
    }

    private List<Module> getModuleByKind(List<Module> modules, List<ModuleKind> kinds){
        List<Module> specificModules= new ArrayList<Module>();
        for(Module module: modules){
            if(kinds.contains(module.getKind())){
                specificModules.add(module);
            }
        }
        return specificModules;
    }
}