/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */

package org.mule.devkit.generation.studio;

import org.mule.common.MuleVersion;
import org.mule.devkit.generation.api.Context;
import org.mule.devkit.generation.api.Generator;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.model.module.Module;
import org.mule.util.StringUtils;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

public abstract class AbstractMuleStudioGenerator implements Generator {

    private final static List<Product> CONSUMES = Arrays.asList(new Product[]{});
    private final static List<Product> PRODUCES = Arrays.asList(new Product[]{});

    private Context context;

    @Override
    public Context ctx() {
        return context;
    }

    @Override
    public void setCtx(Context generationContext) {
        this.context = generationContext;
    }

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    /**
     * Returns the ID used to identify the eclipse feature.
     * Features with the same ID are considered the same component inside Studio
     * Feature IDs have the following format:
     *     <i>org.mule.tooling.ui.contribution.{connectorName}.{connectorMinMuleVersion}</i>
     * @param modules List of feature modules
     * @return Returns the ID used to identify the eclipse feature.
     */
    protected String getFeatureId(Collection<Module> modules) {
        return ctx().getProduct(Product.STUDIO_FEATURE_NAME);
   }


    /**
     * Returns the ID used to identify the eclipse plugin.
     * Plugins with the same ID are considered the same component inside Studio
     * Plugins IDs have the following format:
     *     <i>org.mule.tooling.ui.contribution.{connectorName}.{connectorMinMuleVersion}</i>
     * @param  symbolicName Suffix containing org.mule.tooling.ui.contribution.{connectorName}
     * @param module The module
     * @return Returns the ID used to identify the eclipse plugin.
     */
    protected String getPluginId(String symbolicName, Module module) {
        if(symbolicName != null) {
            return symbolicName + "." + module.getMinMuleVersion().toCompleteNumericVersion();
        } else {
            return null;
        }
    }

    protected MuleVersion getMaxVersion(Collection<Module> modules) {
        MuleVersion maxVersion = null;
        for(Module module : modules) {
            if(maxVersion == null || maxVersion.priorTo(module.getMinMuleVersion())) {
                maxVersion = module.getMinMuleVersion();
            }
        }
        return maxVersion;
    }

    protected String getLabel(Collection<Module> modules, String defaultExtensionName) {
        MuleVersion maxVersion = getMaxVersion(modules);

        String extensionName = getExtensionNameForFeatureLabel(modules, defaultExtensionName);
        if(maxVersion != null) {
            String muleVersion = maxVersion.toString();
            return String.format("%s (Mule %s+)", extensionName, muleVersion);
        } else {
            return extensionName;
        }
    }

    /*
     * The Studio feature label is build with the following logic:
     * 1) [Friendly Name]  [Connector/Module] (Mule [Min Version]+)
     * 2) [Project Name in pom.xml] (Mule [Min Version]+)
     * The second will happen only in Multi Module extensions that at
     * least one of the modules has a different Friendly Name
     */
    private String getExtensionNameForFeatureLabel(Collection<Module> modules, String defaultExtensionName) {
        String name = null;
        String type = null;
        boolean usingFriendlyName = true;

        for(Module module : modules) {
            if(name == null) {
                name = module.getFriendlyName();
                if(name != null && !(name.trim().toLowerCase().endsWith("connector") || name.trim().toLowerCase().endsWith("module"))) {
                    if(module.isConnector()) {
                        type = "Connector";
                    } else {
                        type = "Module";
                    }
                }
            } else {
                if(!name.equals(module.getFriendlyName())) {
                    name = defaultExtensionName;
                    usingFriendlyName = false;
                    break;
                } else {
                    // If multi module has both Connectors and Modules, default to Module type
                    if(module.isModule() && "Connector".equals(type)) {
                        type = "Module";
                    }
                }
            }
        }

        // Just in case, all the friendlyNames were null
        if(name == null) {
            name = defaultExtensionName;
            usingFriendlyName = false;
        }

        if(usingFriendlyName) {
            return StringUtils.isBlank(type) ? name : name + " " + type;
        } else {
            return name;
        }
    }
}