package org.mule.devkit.generation.studio.utils;

import org.apache.commons.lang.StringUtils;
import org.mule.devkit.model.module.Module;

import java.io.File;

/**
 * Utility class to help resolving paths within a DevKit's project
 */
public class MuleStudioPathResolver {

    /**
     * Given a {@code module}, it returns the full path to the source file removing all the packages it has.
     * <p> Sample: The connector is under firstPackage.secondPackage.HelloConnector.java, which means a full path like the
     * following, /Users/<your-user>/Hello-Connector/src/main/java/firstPackage/secondPackage/HelloConnector.java.
     * The returned string will be "/Users/<your-user>/Hello-Connector/src/main/java"
     * </p>
     *
     * @param module The module to reference with
     * @return the path to the java class where your concrete class is
     */
    public static String getPathWithoutPackages(Module module) {
        String sourcePath = module.getPathToSourceFile();
        int packageCount = StringUtils.countMatches(module.getQualifiedName().toString(), ".") + 1;
        while (packageCount > 0) {
            sourcePath = sourcePath.substring(0, sourcePath.lastIndexOf("/"));
            packageCount--;
        }
        return sourcePath;
    }

    /**
     * Given a {@code module} and a {@code relativeFilePath}, it creates a {@link File} and returns it. Keep in mind
     * the {@code relativeFilePath} should be relative to the actual module, which means that if you want to access
     * a root level, "../" must be added.
     * <p> Sample:</p>
     * <ol>
     *     <li>The connector is under firstPackage.secondPackage.HelloConnector.java, which means a full path like the
     * following, /Users/<your-user>/Hello-Connector/src/main/java/firstPackage/secondPackage/HelloConnector.java. </li>
     *     <li>If you want to access /Users/<your-user>/Hello-Connector/some/folder/file.txt, {@code relativeFilePath}
     *     must have the value of "../../../some/folder/file.txt"</li>
     * </ol>
     * </p>
     *
     * @see {@link #getPathWithoutPackages(org.mule.devkit.model.module.Module)}
     *
     * @param module The module to reference with
     * @param relativeFilePath the file to access (must be relative to the module/connector folder file)
     * @return a File with the concrete file (recommended use after calling this method is {@link java.io.File#exists()})
     */
    public static File getFileResolvingPackages(Module module, String relativeFilePath){
        String sourcePath = getPathWithoutPackages(module);
        return new File(sourcePath, relativeFilePath);
    }
}
