/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.generation.studio.editor;

import org.mule.api.annotations.Paged;
import org.mule.api.annotations.Query;
import org.mule.api.annotations.QueryOperator;
import org.mule.api.annotations.display.Placement;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.MetaDataKeyParam;
import org.mule.api.annotations.param.MetaDataStaticKey;
import org.mule.devkit.generation.api.Context;
import org.mule.devkit.generation.utils.NameUtils;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.Identifiable;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.Parameter;
import org.mule.devkit.model.Variable;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ProcessorMethod;
import org.mule.devkit.model.module.connectivity.ManagedConnectionModule;
import org.mule.devkit.model.module.oauth.OAuthVersion;
import org.mule.devkit.model.schema.SchemaConstants;
import org.mule.devkit.model.studio.AttributeCategory;
import org.mule.devkit.model.studio.AttributeType;
import org.mule.devkit.model.studio.CollectionAttributeType;
import org.mule.devkit.model.studio.EnumElement;
import org.mule.devkit.model.studio.EnumType;
import org.mule.devkit.model.studio.Group;
import org.mule.devkit.model.studio.ListOfMapAttributeType;
import org.mule.devkit.model.studio.ModeElementType;
import org.mule.devkit.model.studio.ModeType;
import org.mule.devkit.model.studio.NestedElementReference;
import org.mule.devkit.model.studio.ObjectFactory;
import org.mule.devkit.model.studio.ObjectListAttributeType;
import org.mule.devkit.model.studio.QueryType;
import org.mule.devkit.model.studio.StringAttributeType;
import org.mule.devkit.model.studio.StringListAttributeType;
import org.mule.devkit.model.studio.StringMapAttributeType;
import org.mule.devkit.model.studio.TypeChooserType;
import org.mule.util.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.lang.model.type.TypeKind;
import javax.xml.bind.JAXBElement;

public abstract class BaseStudioXmlBuilder {

    public static final String GENERAL_GROUP_NAME = "General";
    public static final String CONNECTION_GROUP_NAME = "Connection";
    public static final String ADVANCED_CONNECTION_GROUP_NAME = "Connector Configuration Overrides";
    private static final String QUERY_GROUP_NAME = "Query";
    private static final String PAGING_GROUP_NAME = "Paging";

    protected ObjectFactory objectFactory;
    protected MuleStudioUtils helper;
    protected Module module;
    protected Method method;
    protected String moduleName;
    protected Context context;


    protected BaseStudioXmlBuilder(Context context) {
        this.context = context;
        helper = new MuleStudioUtils();
        objectFactory = new ObjectFactory();
    }

    protected BaseStudioXmlBuilder(Context context, Module module) {
        this(context);
        this.module = module;
        moduleName = module.getModuleName();
    }

    protected Group createGroupWithModeSwitch(List<? extends Method> methods) {
        ModeType modeSwitch = new ModeType();
        modeSwitch.getMode().addAll(this.getModes(methods));
        modeSwitch.setCaption(helper.formatCaption("Operation"));
        modeSwitch.setName(StringUtils.capitalize(moduleName) + " operations to execute");
        modeSwitch.setDescription(helper.formatDescription("Operation"));
        modeSwitch.setAlwaysCombo(true);

        Group group = new Group();
        group.setId(module.getModuleName() + "ConnectorGeneric");
        group.getRegexpOrEncodingOrModeSwitch().add(objectFactory.createGroupModeSwitch(modeSwitch));
        group.setCaption(helper.formatCaption(MuleStudioEditorXmlGenerator.GROUP_DEFAULT_CAPTION));
        return group;
    }

    protected List<ModeElementType> getModes(List<? extends Method> methods) {
        List<ModeElementType> modes = new ArrayList<ModeElementType>();
        for (Method method : methods) {
            ModeElementType mode = new ModeElementType();
            mode.setModeId(MuleStudioEditorXmlGenerator.URI_PREFIX + module.getModuleName() + '/' + helper.getLocalId(method));
            mode.setModeLabel(StringUtils.capitalize(helper.getFriendlyName(method)));
            modes.add(mode);
        }

        return modes;
    }

    protected BaseStudioXmlBuilder(Context context, Method method, Module module) {
        this(context, module);
        this.method = method;
    }

    protected List<AttributeCategory> processMethodParameters() {
        Map<String, AttributeCategory> attributeCategoriesByName = processVariableElements(getParametersSorted());

        if(method instanceof ProcessorMethod) {
            ProcessorMethod processor = (ProcessorMethod) method;

            if (processor.canBeUsedInOAuthManagement() &&
                processor.getOAuthModule().getOAuthVersion().equals(OAuthVersion.V2) &&
                processor.isOAuthProtected()) {

                processOAuthProtectedMethod(attributeCategoriesByName);
            }

            if (processor.isPaged()) {
                processPagedMethod(attributeCategoriesByName, processor);
            }

        }


        return new ArrayList<AttributeCategory>(attributeCategoriesByName.values());
    }

    private void processPagedMethod(Map<String, AttributeCategory> attributeCategoriesByName, ProcessorMethod processor)
    {
        Paged cfg = processor.getPagingAnnotation();

        StringAttributeType fetchSize = new StringAttributeType();
        fetchSize.setName("fetchSize");
        fetchSize.setRequired(false);
        fetchSize.setDefaultValue(String.valueOf(cfg.defaultFetchSize()));
        fetchSize.setCaption(helper.formatCaption("Fetch Size"));
        fetchSize.setDescription(helper.formatDescription("The amount of items to fetch on each invocation to the data source"));
        fetchSize.setJavaType("java.lang.String");

        AttributeCategory attributeCategory = this.getOrCreateDefaultAttributeCategory(attributeCategoriesByName);

        Group group = new Group();
        group.setCaption(helper.formatCaption(PAGING_GROUP_NAME));
        group.setId(StringUtils.uncapitalize((PAGING_GROUP_NAME)));

        group.getRegexpOrEncodingOrModeSwitch().add(helper.createJAXBElement(fetchSize));

        attributeCategory.getGroup().add(group);

    }

    private Group getOrCreateGroup(Map<String, AttributeCategory> attributeCategoriesByName, String name)
    {
        AttributeCategory attributeCategory = null;
        if (attributeCategoriesByName.get(name) != null) {
            attributeCategory = attributeCategoriesByName.get(name);
        } else {
            attributeCategory = new AttributeCategory();
            attributeCategory.setCaption(helper.formatCaption(name));
            attributeCategory.setDescription(name);
            attributeCategoriesByName.put(name,attributeCategory);
        }

        Group group = getGroup(attributeCategory, name);

        if (group == null) {
            group = new Group();
            group.setCaption(helper.formatCaption(name));
            group.setId(StringUtils.uncapitalize(name));
            attributeCategory.getGroup().add(group);
        }
        return group;
    }

    private void processOAuthProtectedMethod(Map<String, AttributeCategory> attributeCategoriesByName)
    {
        StringAttributeType accessTokenId = new StringAttributeType();
        accessTokenId.setName("accessTokenId");
        accessTokenId.setRequired(false);
        accessTokenId.setCaption(helper.formatCaption("Access Token Id"));
        accessTokenId.setDescription(helper.formatDescription("The id of the access token that will be used to authenticate the call"));
        accessTokenId.setJavaType("java.lang.String");

        Group group = getOrCreateGroup(attributeCategoriesByName, MuleStudioEditorXmlGenerator.ADVANCED_ATTRIBUTE_CATEGORY_CAPTION);

        group.getRegexpOrEncodingOrModeSwitch().add(helper.createJAXBElement(accessTokenId));
    }

    private Group getDefaultGroup(AttributeCategory attributeCategory) {
        return getGroup(attributeCategory, GENERAL_GROUP_NAME);
    }

    private Group getGroup(AttributeCategory attributeCategory, String groupName) {
        if(attributeCategory != null) {
            for(Group group : attributeCategory.getGroup()) {
                if(StringUtils.uncapitalize(groupName).equalsIgnoreCase(group.getId())) {
                    return group;
                }
            }
        }
        return null;
    }

    protected List<AttributeCategory> processConfigurableFields(Group defaultGroup) {
        Map<String, AttributeCategory> attributeCategoriesByName = processVariableElements(getConfigurableFieldsSorted());
        List<AttributeCategory> attributeCategories = new ArrayList<AttributeCategory>(attributeCategoriesByName.values());

        AttributeCategory attributeCategory = attributeCategoriesByName.get(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION);
        if (attributeCategory != null) {
            attributeCategory.setDescription(helper.formatDescription(module.getModuleName() + " configuration properties"));
            List<Group> groups = attributeCategory.getGroup();
            if (groups.isEmpty()) {
                groups.add(defaultGroup);
            } else {
                groups.add(0, defaultGroup);
            }
        } else {
            attributeCategory = new AttributeCategory();
            attributeCategory.setCaption(helper.getFormattedCaption(module));
            attributeCategory.setDescription(helper.formatDescription(module.getModuleName() + " configuration properties"));
            attributeCategory.getGroup().add(defaultGroup);
            attributeCategories.add(attributeCategory);
        }
        return attributeCategories;
    }

    private Map<String, AttributeCategory> processVariableElements(List<? extends Variable> variableElements) {

        Map<String, Group> groupsByName = new LinkedHashMap<String, Group>();
        Map<String, AttributeCategory> attributeCategoriesByName = new LinkedHashMap<String, AttributeCategory>();
        getOrCreateDefaultAttributeCategory(attributeCategoriesByName); // Create default category
        processConnectionAttributes(groupsByName, attributeCategoriesByName);
        createPoolingProfileAttributes(groupsByName, attributeCategoriesByName);
        createReconnectionAttributes(groupsByName, attributeCategoriesByName);
        createOAuthConfig(groupsByName, attributeCategoriesByName);
        createMetaDataAttributes(groupsByName, attributeCategoriesByName);

        for (Variable parameter : variableElements) {
            JAXBElement<? extends AttributeType> jaxbElement = createJaxbElement(parameter);
            AttributeCategory attributeCategory = getOrCreateAttributeCategory(attributeCategoriesByName, parameter.getAnnotation(Placement.class));
            Group group = getOrCreateGroup(groupsByName, parameter);
            group.getRegexpOrEncodingOrModeSwitch().add(jaxbElement);

            if (!attributeCategory.getGroup().contains(group)) {
                attributeCategory.getGroup().add(group);
            }
        }

        return attributeCategoriesByName;
    }

    protected void createMetaDataAttributes(Map<String,Group> groupsByName, Map<String,AttributeCategory> attributeCategoriesByName) {
       // override if necessary
    }

    protected void processConnectionAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        // override if necessary
    }

    protected void createReconnectionAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        // override if necessary
    }

    protected void createPoolingProfileAttributes(Map<String, Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        // override if necessary
    }

    protected void createOAuthConfig(Map<String,Group> groupsByName, Map<String, AttributeCategory> attributeCategoriesByName) {
        // override if necessary
    }

    private AttributeCategory getOrCreateDefaultAttributeCategory(Map<String, AttributeCategory> attributeCategoriesByName) {
        return getOrCreateAttributeCategory(attributeCategoriesByName, null);

    }

    private AttributeCategory getOrCreateAttributeCategory(Map<String, AttributeCategory> attributeCategoriesByName, Placement placement) {
        if (placement == null || StringUtils.isBlank(placement.tab())) {
            if (!attributeCategoriesByName.containsKey(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION)) {
                AttributeCategory attributeCategoryGeneral = new AttributeCategory();
                attributeCategoryGeneral.setCaption(helper.formatCaption(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION));
                attributeCategoryGeneral.setDescription(helper.formatDescription(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_DESCRIPTION));
                attributeCategoriesByName.put(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION, attributeCategoryGeneral);



            }
            return attributeCategoriesByName.get(MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION);
        } else {
            String attributeCategoryName;
            if (StringUtils.isNotBlank(placement.tab())) {
                attributeCategoryName = placement.tab();
            } else {
                attributeCategoryName = MuleStudioEditorXmlGenerator.ATTRIBUTE_CATEGORY_DEFAULT_CAPTION;
            }
            if (!attributeCategoriesByName.containsKey(attributeCategoryName)) {
                AttributeCategory attributeCategory = new AttributeCategory();
                attributeCategory.setCaption(helper.formatCaption(attributeCategoryName));
                attributeCategory.setDescription(helper.formatDescription(attributeCategoryName));
                attributeCategoriesByName.put(attributeCategoryName, attributeCategory);
            }
            return attributeCategoriesByName.get(attributeCategoryName);
        }
    }

    private Group getOrCreateGroup(Map<String, Group> groupsByName, Variable parameter) {
        Placement placement = parameter != null ? parameter.getAnnotation(Placement.class) : null;
        if (parameter != null && parameter.isQuery()) {
            Group queryGroup = new Group();
            queryGroup.setCaption(helper.formatCaption(QUERY_GROUP_NAME));
            queryGroup.setId(StringUtils.uncapitalize(QUERY_GROUP_NAME));
            groupsByName.put(QUERY_GROUP_NAME,queryGroup);
            return groupsByName.get(QUERY_GROUP_NAME);
        }
        if (placement == null || StringUtils.isBlank(placement.group())) {
            if (!groupsByName.containsKey(GENERAL_GROUP_NAME)) {
                Group groupGeneral = new Group();
                groupGeneral.setCaption(helper.formatCaption(GENERAL_GROUP_NAME));
                groupGeneral.setId(StringUtils.uncapitalize(GENERAL_GROUP_NAME));
                groupsByName.put(GENERAL_GROUP_NAME, groupGeneral);
            }
            return groupsByName.get(GENERAL_GROUP_NAME);
        } else {
            String groupName = placement.group();
            if (!groupsByName.containsKey(groupName)) {
                Group group = new Group();
                group.setCaption(groupName);
                group.setId(StringUtils.uncapitalize(groupName));
                groupsByName.put(groupName, group);
            }
            return groupsByName.get(groupName);
        }
    }

    protected boolean isConfigurableVariable(Variable<?> variable) {
        if(module != null && module.getConfigurableFields() != null) {
            return module.getConfigurableFields().contains(variable);
        } else {
            return false;
        }
    }

    protected JAXBElement<? extends AttributeType> createJaxbElement(Variable<?> variable) {
        AttributeType attributeType;
       if (variable.isQuery()) {
            attributeType = createQueryType(variable);
        } else if(variable.isMetaDataKey()) {
            attributeType = createTypeChooserType(variable);
        } else if (variable.asType().isEnum()) {
            attributeType = createEnumType(variable);
        } else if (isComplexList(variable)) {
            attributeType = createComplexListType(variable);
        } else if (isListOfMaps(variable) || isSimpleList(variable) || isSimpleMap(variable) || isSimpleSet(variable)) {
            attributeType = handleCollectionVariable(variable);
        } else if (variable.asType().isComplexType() && !variable.isRefOnly()) {
            NestedElementReference complexTypeNestedElementReference = createNestedElementReference(variable);
            return objectFactory.createGroupChildElement(complexTypeNestedElementReference);
        } else {
            attributeType = createAttributeType(variable);
        }

        if (isConfigurableVariable(variable)) {
            setMetadataAttributes(variable, attributeType);
        }

        return helper.createJAXBElement(attributeType);
    }

    protected void setMetadataAttributes(Variable<?> variable, AttributeType attributeType) {
    }

    protected AttributeType createQueryType(Variable<?> variable) {
        QueryType type = new QueryType();
        type.setSupportsExpressions(true);
        type.setAssociatedConfig("config-ref");
        Query qVariable = variable.getAnnotation(org.mule.api.annotations.Query.class);
        List<QueryOperator> disabledOperators = Arrays.asList(qVariable.disabledOperators());
        type.setAndOperator(disabledOperators.contains(QueryOperator.AND) ? "disabled" : "enabled");
        type.setOrOperator(disabledOperators.contains(QueryOperator.OR) ? "disabled" : "enabled");
        type.setNativeQuery(module.hasQueryTranslator() ? "enabled" : "disabled");
        type.setLimit(qVariable.limit()? "enabled" : "disabled");
        type.setOffset(qVariable.offset()? "enabled" : "disabled");
        type.setOrderBy(qVariable.orderBy()? "enabled" : "disabled");
        helper.setAttributeTypeInfo(variable, type);
        return type;
    }

    private AttributeType createComplexListType(Variable parameter) {
        ObjectListAttributeType objectListAttributeTyper = new ObjectListAttributeType();
        objectListAttributeTyper.setListName(NameUtils.uncamel(parameter.getName()));
        setCommonCollectionAttributes(parameter, objectListAttributeTyper);
        return objectListAttributeTyper;
    }

    private boolean isComplexList(Variable parameter) {
        return parameter.asType().isArrayOrList() && !parameter.getTypeArguments().isEmpty()
                && parameter.getTypeArguments().get(0).asTypeMirror().getKind().equals(TypeKind.DECLARED)
                && parameter.getTypeArguments().get(0).isComplexTypeWithGetterAndSetter(false);
    }

    private AttributeType createTypeChooserType(Variable<?> variable) {
        TypeChooserType type = new TypeChooserType();
        type.setSupportsExpressions(true);
        type.setAssociatedConfig("config-ref");
        type.setAffects(variable.getAnnotation(MetaDataKeyParam.class).affects().toString());
        helper.setAttributeTypeInfo(variable, type);
        return type;
    }

    private NestedElementReference createNestedElementReference(Variable parameter) {
        NestedElementReference childElement = new NestedElementReference();
        String defaultValue = parameter.getDefaultValue();
        if (defaultValue != null) {
            childElement.setDefaultValue(defaultValue);
        }
        childElement.setName(MuleStudioEditorXmlGenerator.URI_PREFIX + moduleName + '/' + NameUtils.uncamel(parameter.getName()));
        childElement.setCaption(helper.getFormattedCaption(parameter));
        childElement.setAllowMultiple(false);
        childElement.setInplace(true);
        childElement.setJavaType(parameter.getJavaType());
        childElement.setRequired(!parameter.isOptional());
        return childElement;
    }

    private AttributeType handleCollectionVariable(Variable parameter) {
        AttributeType attributeType;
        if (isListOfMaps(parameter)) {
            ListOfMapAttributeType objectListAttributeType = new ListOfMapAttributeType();
            objectListAttributeType.setRequired(!parameter.isOptional());
            objectListAttributeType.setListName(NameUtils.uncamel(parameter.getName()));
            objectListAttributeType.setInnerName(NameUtils.uncamel(SchemaConstants.INNER_PREFIX + NameUtils.singularize(parameter.getName())));
            if (parameter.isMetaDataStaticKey()){
                objectListAttributeType.setMetaDataStaticKey(parameter.getAnnotation(MetaDataStaticKey.class).type());
            }
            setCommonCollectionAttributes(parameter, objectListAttributeType);
            attributeType = objectListAttributeType;
        } else if (isSimpleList(parameter) || isSimpleSet(parameter)) {
            StringListAttributeType stringListAttributeTyper = new StringListAttributeType();

            stringListAttributeTyper.setListName(NameUtils.uncamel(parameter.getName()));
            setCommonCollectionAttributes(parameter, stringListAttributeTyper);
            attributeType = stringListAttributeTyper;
        } else {
            StringMapAttributeType stringMapAttributeType = new StringMapAttributeType();
            stringMapAttributeType.setMapName(NameUtils.uncamel(parameter.getName()));
            if (parameter.isMetaDataStaticKey()){
                stringMapAttributeType.setMetaDataStaticKey(parameter.getAnnotation(MetaDataStaticKey.class).type());
            }
            setCommonCollectionAttributes(parameter, stringMapAttributeType);
            attributeType = stringMapAttributeType;
        }
        // DEVKIT-275 Need required information in Studio
        attributeType.setRequired(!parameter.isOptional());
        return attributeType;
    }

    private void setCommonCollectionAttributes(Variable parameter, CollectionAttributeType collectionAttributeType) {
        collectionAttributeType.setItemName(NameUtils.uncamel(NameUtils.singularize(parameter.getName())));
        collectionAttributeType.setLocalName(helper.getLocalId(method, parameter));
        collectionAttributeType.setCaption(helper.getFormattedCaption(parameter));
        collectionAttributeType.setJavaType(parameter.getJavaType());
        String defaultValue = parameter.getDefaultValue();
        if (defaultValue != null) {
            collectionAttributeType.setDefaultValue(defaultValue);
        }
        if (method != null) {
            collectionAttributeType.setDescription(helper.formatDescription(method.getJavaDocParameterSummary(parameter.getName())));
        } else {
            collectionAttributeType.setDescription(helper.formatDescription(parameter.getJavaDocSummary()));
        }
    }

    private boolean isListOfMaps(Variable parameter) {
        return parameter.asType().isArrayOrList() && !parameter.getTypeArguments().isEmpty() && parameter.getTypeArguments().get(0).isMap();
    }

    private boolean isSimpleMap(Variable parameter) {
        return parameter.asType().isMap() && (parameter.getTypeArguments().isEmpty() || !parameter.getTypeArguments().get(1).isCollection());
    }

    private boolean isSimpleList(Variable parameter) {
        return parameter.asType().isArrayOrList() && (parameter.getTypeArguments().isEmpty() || !parameter.getTypeArguments().get(0).isCollection());
    }

    private boolean isSimpleSet(Variable parameter) {
        return parameter.asType().isSet() && (parameter.getTypeArguments().isEmpty() || !parameter.getTypeArguments().get(0).isCollection());
    }

    private List<Parameter> getParametersSorted() {
        List<Parameter> parameters = new ArrayList<Parameter>(method.getParameters());
        Iterator<Parameter> iterator = parameters.iterator();
        while (iterator.hasNext()) {
            if (iterator.next().shouldBeIgnored()) {
                iterator.remove();
            }
        }

        Collections.sort(parameters, new VariableComparator());
        return parameters;
    }

    private AttributeType createAttributeType(Variable parameter) {
        AttributeType attributeType = helper.createAttributeTypeIgnoreEnumsAndCollections(parameter);
        if (attributeType != null) {
            helper.setAttributeTypeInfo(parameter, attributeType);
        }
        return attributeType;
    }

    protected List<AttributeType> getConnectionAttributes(ManagedConnectionModule module) {
        List<AttributeType> parameters = new ArrayList<AttributeType>();
        for (Parameter connectAttributeType : module.getConnectMethod().getParameters()) {
            AttributeType parameter = helper.createAttributeTypeIgnoreEnumsAndCollections(connectAttributeType);
            helper.setAttributeTypeInfo(connectAttributeType, parameter);
            // As these can be overriden at operation level, they are always optional
            parameter.setRequired(false);
            setMetadataAttributes(connectAttributeType, parameter);
            parameters.add(parameter);

        }
        return parameters;
    }

    private EnumType createEnumType(Variable<?> variable) {
        EnumType enumType = new EnumType();
        enumType.setSupportsExpressions(true);
        enumType.setAllowsCustom(true);
        helper.setAttributeTypeInfo(variable, enumType);
        for (Identifiable<?> enumMember : ((org.mule.devkit.model.EnumType) variable.asType()).getEnumConstants()) {
            String enumConstant = enumMember.getName();
            EnumElement enumElement = new EnumElement();
            enumElement.setValue(enumConstant);
            enumType.getOption().add(enumElement);
        }
        Collections.sort(enumType.getOption(), new EnumElementComparator());
        return enumType;
    }

    private List<Field> getConfigurableFieldsSorted() {
        List<Field> configurableFields = module.getConfigurableFields();
        Collections.sort(configurableFields, new VariableComparator());
        return configurableFields;
    }

    protected String buildVersionsString() {
        /*
         * DEVKIT-368: Studio needs the supported ESB versions in order to support
         * multiple cloud connectors.
         * TODO: Fix once Studio provides a better way to indicate that this is unbounded (a.k.a. remove 8.0.0!!!)
         */
        return "[" + module.getMinMuleVersion().toCompleteNumericVersion()  + ",8.0.0]";

    }
}